%{
/* Lexer for image processing expressions.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

/* Turn off wrapping.
 */
#define yywrap() (1)

#ifdef HAVE_FLEX

/* Flex has a different input mechanism :(
 */

#define YY_INPUT(buf,result,max_size) \
{ \
	extern int ip_input( void ); \
	int c = ip_input(); \
	result = (c == 0) ? YY_NULL : (buf[0] = c, 1); \
}

#undef unput
#define unput ip_unput

#else /*HAVE_FLEX*/

/* Assume this is plain lex.
 */

/* Redefine input, output, unput and yywrap.
 */
#undef input
#undef output
#undef unput
#undef yywrap
#undef YYLMAX

/* See parse.y for input and unput.
 */
#define output(A) (error( "output called by lex!!" ))

#define YYLMAX MAX_STRSIZE

#define unput ip_unput
#define input ip_input
#define yywrap() (1)

#endif /*HAVE_FLEX*/

#include "ip.h"

/* Stuff from bison.
 */
#include "parse.h"

/* Read a string into a buffer. Read up the term character, term can be
 * escaped with '\'.
 */
static void
read_string( char *buf, int term )
{
	int ch;
	int i;

	/* Read up to \n, term, EOF, ignoring \term
	 */
	for( i = 0; (ch = ip_input()); i++ ) {
		if( ch == EOF || ch == '\n' || ch == term || ch == '\0' )
			break;
		if( i >= MAX_STRSIZE )
			yyerror( "line too long" );

		buf[i] = ch;

		if( ch == '\\' ) {
			ch = ip_input();

			if( ch == term ) 
				buf[++i] = term;
			else
				ip_unput( ch );
		}
	}
	buf[i] = '\0';

	if( ch == '\n' )
		yyerror( "end of line inside string" );
	if( ch == EOF || ch == '\0' )
		yyerror( "no end of string" );
}

%}

%Start BINARY
%%
\/\*			{ 
				int ch;

				while( (ch = input()) != EOF )
					if( ch == '*' ) {
						if( (ch = input()) == '/' )
							break;
						else
							unput( ch );
					}

				if( ch == EOF )
					yyerror( "no end of comment!" );
			}
\/\/			{ 
				int ch;

				/* Read string up to \n, EOF.
				 */
				while( (ch = input()) != EOF && ch != '\n' )
					;
			}
\#separator		{ BEGIN 0; return( TK_SEPARATOR ); }
\#dialog		{ BEGIN 0; return( TK_DIALOG ); }
class			{ BEGIN 0; return( TK_CLASS ); }
char			{ BEGIN 0; return( TK_CHAR ); }
short			{ BEGIN 0; return( TK_SHORT ); }
int			{ BEGIN 0; return( TK_INT ); }
float			{ BEGIN 0; return( TK_FLOAT ); }
double			{ BEGIN 0; return( TK_DOUBLE ); }
signed			{ BEGIN 0; return( TK_SIGNED ); }
unsigned		{ BEGIN 0; return( TK_UNSIGNED ); }
complex			{ BEGIN 0; return( TK_COMPLEX ); }
if			{ BEGIN 0; return( TK_IF ); }
then			{ BEGIN 0; return( TK_THEN ); }
else			{ BEGIN 0; return( TK_ELSE ); }
\.\.\.			{ BEGIN 0; return( TK_DOTDOTDOT ); }
\.\.			{ BEGIN 0; return( TK_DOTDOTDOT ); }
true			{ 	
				BEGIN BINARY;

				yylval.yy_const.type = PARSE_CONST_BOOL;
				yylval.yy_const.val.bool = TRUE;

				return( TK_CONST );
			}
false			{ 	
				BEGIN BINARY;

				yylval.yy_const.type = PARSE_CONST_BOOL;
				yylval.yy_const.val.bool = FALSE;

				return( TK_CONST );
			}
[a-zA-Z_][a-zA-Z0-9_']*	{
				char *name = model_loadstate_rewrite_name( 
					yytext );

				BEGIN BINARY;

				if( name ) {
					yylval.yy_name = im_strdupn( name );
					buf_change( &lex_text, yytext, name );
				}
				else 
					yylval.yy_name = im_strdupn( yytext );

				return( TK_IDENT );
			}
\(			{ BEGIN 0; return( '(' ); }
\)			{ BEGIN BINARY; return( ')' ); }
<BINARY>\+\+		{ BEGIN 0; return( TK_JOIN ); }
<BINARY>\+		|
<BINARY>\-		{ BEGIN 0; return( *yytext ); }
\-			{ BEGIN 0; return( TK_UMINUS ); }
\+			{ BEGIN 0; return( TK_UPLUS ); }
\<			{ BEGIN 0; return( TK_LESS ); }
\<\=			{ BEGIN 0; return( TK_LESSEQ ); }
\>			{ BEGIN 0; return( TK_MORE ); }
\>\=			{ BEGIN 0; return( TK_MOREEQ ); }
\&			{ BEGIN 0; return( TK_BAND ); }
\&\&			{ BEGIN 0; return( TK_LAND ); }
\*\*			{ BEGIN 0; return( TK_POW ); }
\>\>			{ BEGIN 0; return( TK_RSHIFT ); }
\<\<			{ BEGIN 0; return( TK_LSHIFT ); }
\|			{ BEGIN 0; return( TK_BOR ); }
\|\|			{ BEGIN 0; return( TK_LOR ); }
\=\=			{ BEGIN 0; return( TK_EQ ); }
\=\=\=			{ BEGIN 0; return( TK_PEQ ); }
\!\=			{ BEGIN 0; return( TK_NOTEQ ); }
\!\=\=			{ BEGIN 0; return( TK_PNOTEQ ); }
\.			|
\^			|
\?			|
\*			|
\/			|
\%			|
\,			|
\!			|
\;			|
\[			|
\:			|
\=			|
\~			|
\@			|
\{			|
\}			{ BEGIN 0; return( *yytext ); }
\]			{ BEGIN BINARY; return( *yytext ); }
0x[0-9a-fA-F]+		{
				unsigned int i;

				BEGIN BINARY;

				if( sscanf( yytext, "0x%x", &i ) != 1 )
					yyerror( "bad number %s", yytext );
				yylval.yy_const.type = PARSE_CONST_NUM;
				yylval.yy_const.val.num = i;

				return( TK_CONST );
			}
[0-9]*(\.[0-9]+)?([eE][+-]?[0-9]+)?	{
				BEGIN BINARY;

				yylval.yy_const.type = PARSE_CONST_NUM;
				yylval.yy_const.val.num = 
					strtod( yytext, NULL );

				return( TK_CONST );
			}
\'			{
				char buf[MAX_STRSIZE];
				char buf2[MAX_STRSIZE];

				BEGIN BINARY;

				read_string( buf, '\'' );
				strccpy( buf2, buf );

				if( strlen( buf2 ) != 1 )
					yyerror( "bad char constant" );

				yylval.yy_const.type = PARSE_CONST_CHAR;
				yylval.yy_const.val.ch = buf[0];

				return( TK_CONST );
			}
\"			{ 
				char buf[MAX_STRSIZE];
				char buf2[MAX_STRSIZE];

				BEGIN BINARY;

				read_string( buf, '"' );

				yylval.yy_const.type = PARSE_CONST_STR;
				yylval.yy_const.val.str = 
					im_strdupn( strccpy( buf2, buf ) );

				return( TK_CONST );
			}
[ \t\n\01]		;
.			{
				yyerror( "illegal character \"%c\"", *yytext );
			}
