/* run the display for an image in a workspace 
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

/* Tag our drag-n-drops with these.
 */
enum {
	TARGET_COLOUR,
	TARGET_TEXT
};

static ImagedisplayClass *parent_class = NULL;

static void
colourdisplay_destroy( GtkObject *object )
{
	Colourdisplay *colourdisplay;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_COLOURDISPLAY( object ) );

	colourdisplay = COLOURDISPLAY( object );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Prefer x-color drags for 3 band non-complex imageinfos.
 */
static void
colourdisplay_set_drag_type( Colourdisplay *colourdisplay )
{
	static const GtkTargetEntry text_targets[] = {
		{ "text/plain", 0, TARGET_TEXT },
		{ "application/x-color", 0, TARGET_COLOUR }
	};

	static const GtkTargetEntry colour_targets[] = {
		{ "application/x-color", 0, TARGET_COLOUR },
		{ "text/plain", 0, TARGET_TEXT }
	};

	Imageinfo *imageinfo = IMAGEDISPLAY( colourdisplay )->conv->ii;
	IMAGE *im = imageinfo_get( FALSE, imageinfo );
	const GtkTargetEntry *targets;

	if( !GTK_WIDGET_REALIZED( GTK_WIDGET( colourdisplay ) ) || !im ) 
		return;

	if( im->Bands == 3 && !im_iscomplex( im ) )
		targets = colour_targets;
	else
		targets = text_targets;

	gtk_drag_dest_unset( GTK_WIDGET( colourdisplay ) );
	gtk_drag_dest_set( GTK_WIDGET( colourdisplay ),
		GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_MOTION | 
			GTK_DEST_DEFAULT_DROP,
		targets, IM_NUMBER( text_targets ),
		GDK_ACTION_COPY );
	gtk_drag_source_unset( GTK_WIDGET( colourdisplay ) );
	gtk_drag_source_set( GTK_WIDGET( colourdisplay ),
		GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
		targets, IM_NUMBER( text_targets ),
		GDK_ACTION_COPY | GDK_ACTION_MOVE );
}

static void
colourdisplay_realize( GtkWidget *widget )
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( widget );

	GTK_WIDGET_CLASS( parent_class )->realize( widget );

	colourdisplay_set_drag_type( colourdisplay );
}

static void
colourdisplay_drag_begin( GtkWidget *widget, GdkDragContext *context )
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( widget );
	GtkWidget *window;
	double colours[3];
	GdkColor bg;

	window = gtk_window_new( GTK_WINDOW_POPUP );
	gtk_widget_set_app_paintable( GTK_WIDGET( window ), TRUE );
	gtk_widget_set_usize( window, 48, 32 );
	gtk_widget_realize( window );
	gtk_object_set_data_full( GTK_OBJECT( widget ),
		"colourdisplay-drag-window", window,
		(GtkDestroyNotify) gtk_widget_destroy );

	imageinfo_to_rgb( IMAGEDISPLAY( colourdisplay )->conv->ii, colours );
	bg.red = 0xffff * colours[0];
	bg.green = 0xffff * colours[1];
	bg.blue = 0xffff * colours[2];

	gdk_color_alloc( gtk_widget_get_colormap( window ), &bg );
	gdk_window_set_background( window->window, &bg );

	gtk_drag_set_icon_widget( context, window, -2, -2 );
}

static void
colourdisplay_drag_end( GtkWidget *widget, GdkDragContext *context )
{
	gtk_object_set_data( GTK_OBJECT( widget ), 
		"colourdisplay-drag-window", NULL );
}

static void
colourdisplay_drag_data_get( GtkWidget *widget, GdkDragContext *context,
	GtkSelectionData *selection_data, guint info, guint time ) 
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( widget );
	Imageinfo *imageinfo = IMAGEDISPLAY( colourdisplay )->conv->ii;
	double colours[3];
	guint16 vals[4];
	char buf_text[256];
	BufInfo buf;

	switch( info ) {
	case TARGET_COLOUR:
		imageinfo_to_rgb( imageinfo, colours );

		vals[0] = colours[0] * 0xffff;
		vals[1] = colours[1] * 0xffff;
		vals[2] = colours[2] * 0xffff;
		vals[3] = 0xffff;

		gtk_selection_data_set( selection_data,
			gdk_atom_intern( "application/x-color", FALSE ),
			16, (guchar *) vals, 8 );

#ifdef DEBUG
		printf( "colourdisplay_drag_data_get: sending x-color\n" );
#endif /*DEBUG*/

		break;

	case TARGET_TEXT:
		buf_init_static( &buf, buf_text, 256 );
		imageinfo_to_text( imageinfo, &buf );

		gtk_selection_data_set( selection_data,
			gdk_atom_intern( "text/plain", FALSE ),
			8, buf_all( &buf ), strlen( buf_all( &buf ) ) );
		
#ifdef DEBUG
		printf( "colourdisplay_drag_data_get: sending text/plain\n" );
#endif /*DEBUG*/

		break;

	default:
		assert( FALSE );
		break;
	}
}

static void
colourdisplay_drag_data_received( GtkWidget *widget, GdkDragContext *context,
	gint x, gint y, GtkSelectionData *selection_data,
	guint info, guint time ) 
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( widget );
	Imageinfo *imageinfo = IMAGEDISPLAY( colourdisplay )->conv->ii;

	guint16 *vals;
	gdouble colours[3];

	if( selection_data->length < 0 ) 
		return;

	switch( info ) {
	case TARGET_COLOUR: 
		if( selection_data->format != 16 || 
			selection_data->length != 8 )
			return;

#ifdef DEBUG
		printf( "colourdisplay_drag_data_received: seen x-color\n" );
#endif /*DEBUG*/

		vals = (guint16 *)selection_data->data;

		colours[0] = (double) vals[0] / 0xffff;
		colours[1] = (double) vals[1] / 0xffff;
		colours[2] = (double) vals[2] / 0xffff;

		imageinfo_from_rgb( imageinfo, colours );
		break;

	case TARGET_TEXT:
		if( selection_data->format != 8 )
			return;

#ifdef DEBUG
		printf( "colourdisplay_drag_data_received: seen text/plain\n" );
#endif /*DEBUG*/

		if( !imageinfo_from_text( imageinfo, selection_data->data ) )
			box_alert( widget );
		break;

	default:
		assert( FALSE );
		break;
	}
}

static void
colourdisplay_set_tooltip( Colourdisplay *colourdisplay )
{
	Imagedisplay *id = IMAGEDISPLAY( colourdisplay );
	char buf_text[256];
	BufInfo buf;

	if( id->conv && id->conv->ii ) {
		buf_init_static( &buf, buf_text, 256 );
		imageinfo_to_text( id->conv->ii, &buf );
		buf_appends( &buf, "\nDouble-click to edit this colour, or "
			"drag-and-drop between colours" );

		set_tooltip( GTK_WIDGET( colourdisplay ), buf_all( &buf ) );
	}
}

static void
colourdisplay_conversion_changed( Imagedisplay *id )
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( id );

	IMAGEDISPLAY_CLASS( parent_class )->conversion_changed( id );

	if( id->conv && id->conv->mag != 5000 )
		imagedisplay_set_mag( id, 5000 );
	colourdisplay_set_drag_type( colourdisplay );
	colourdisplay_set_tooltip( colourdisplay );
}

static void
colourdisplay_set_conversion( Imagedisplay *id, Conversion *conv )
{
	Colourdisplay *colourdisplay = COLOURDISPLAY( id );

	IMAGEDISPLAY_CLASS( parent_class )->set_conversion( id, conv );

	if( id->conv && id->conv->mag != 5000 )
		imagedisplay_set_mag( id, 5000 );
	colourdisplay_set_tooltip( colourdisplay );
}

static void
colourdisplay_area_changed( Imagedisplay *id, Rect *display )
{
	colourdisplay_set_tooltip( COLOURDISPLAY( id ) );

	IMAGEDISPLAY_CLASS( parent_class )->area_changed( id, display );
}

static void
colourdisplay_class_init( ColourdisplayClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass *) klass;
	ImagedisplayClass *imagedisplay_class = (ImagedisplayClass *) klass;

	parent_class = gtk_type_class( TYPE_IMAGEDISPLAY );

	object_class->destroy = colourdisplay_destroy;
	
	widget_class->realize = colourdisplay_realize;
	widget_class->drag_begin = colourdisplay_drag_begin;
	widget_class->drag_end = colourdisplay_drag_end;
	widget_class->drag_data_get = colourdisplay_drag_data_get;
	widget_class->drag_data_received = colourdisplay_drag_data_received;

	imagedisplay_class->conversion_changed =
		colourdisplay_conversion_changed;
	imagedisplay_class->set_conversion = colourdisplay_set_conversion;
	imagedisplay_class->area_changed = colourdisplay_area_changed;
}

static void
colourdisplay_edit( Colourdisplay *colourdisplay )
{
	imageinfo_colour_edit( GTK_WIDGET( colourdisplay ), 
		IMAGEDISPLAY( colourdisplay )->conv->ii );
}

static void
colourdisplay_double_cb( GtkWidget *wid, 
	Colourdisplay *colourdisplay, guint state )
{
	colourdisplay_edit( colourdisplay ); 
}

static void
colourdisplay_init( Colourdisplay *colourdisplay )
{
#ifdef DEBUG
	printf( "colourdisplay_init\n" );
#endif /*DEBUG*/

	/* Who wants to focus one of these :/
	 */
	GTK_WIDGET_UNSET_FLAGS( GTK_WIDGET( colourdisplay ), GTK_CAN_FOCUS );

	colourdisplay_set_tooltip( colourdisplay );
        doubleclick_add( GTK_WIDGET( colourdisplay ), FALSE,
                NULL, NULL, 
		colourdisplay_double_cb, colourdisplay );
}

GtkType
colourdisplay_get_type( void )
{
	static GtkType type = 0;

	if( !type ) {
		static const GtkTypeInfo info = {
			"Colourdisplay",
			sizeof( Colourdisplay ),
			sizeof( ColourdisplayClass ),
			(GtkClassInitFunc) colourdisplay_class_init,
			(GtkObjectInitFunc) colourdisplay_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		type = gtk_type_unique( TYPE_IMAGEDISPLAY, &info );
	}

	return( type );
}

Colourdisplay *
colourdisplay_new( Conversion *conv )
{
	Colourdisplay *colourdisplay = gtk_type_new( TYPE_COLOURDISPLAY );

	if( !conv ) 
		conv = conversion_new( NULL );
	imagedisplay_set_conversion( IMAGEDISPLAY( colourdisplay ), conv );

	return( colourdisplay );
}
