/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MGRTShell.h"
/**
 * @file  MGRTShell.cc
 * @brief 
 */


MGRTShell::MGRTShell(MGRT *grt, const std::string &icon_path)
  : _grt(grt), _icon_path(icon_path)
{
  add(_top_box);
  
  _top_box.set_border_width(12);
  
  _top_box.pack_start(_menu);
  
  _top_box.pack_start(_paned, true, true);
  
  _paned.add1(_text);
  _paned.add2(_sidenote);
  
  _value_store= Gtk::TreeStore::create(_columns);
  _value_tree.set_model(_value_store);
  _value_box.pack_start(_value_combo, false, true);
  _value_box.pack_start(_value_tree, true, true);
  _value_box.pack_start(_value_entry, false, true);
  _value_box.pack_start(_detail_tree, false, true);
  _detail_tree.set_size_request(-1, 300);
  _sidenote.append_page(_value_box, "Values");
  
  Gtk::TreeViewColumn *column;
  
  column= new Gtk::TreeViewColumn("Object Tree");
  column->pack_start(_columns.icon, false);
  column->pack_start(_columns.text);
  _value_tree.append_column(*Gtk::manage(column));

  _value_tree.append_column_numeric("RefC", _columns.detail, "%i");

  _struct_store= Gtk::TreeStore::create(_columns);
  _struct_tree.set_model(_struct_store);

  column= new Gtk::TreeViewColumn("Structs Tree");
  column->pack_start(_columns.icon, false);
  column->pack_start(_columns.text);
  _struct_tree.append_column(*Gtk::manage(column));


  _sidenote.append_page(_struct_tree, "Structs");

  _module_store= Gtk::TreeStore::create(_columns);
  _module_tree.set_model(_module_store);
  _module_box.pack_start(_module_tree, true, true);
  _module_box.pack_start(_module_label, false, true);
  _sidenote.append_page(_module_box, "Modules");

  column= new Gtk::TreeViewColumn("Modules Tree");
  column->pack_start(_columns.icon, false);
  column->pack_start(_columns.text);
  _module_tree.append_column(*Gtk::manage(column));

  _container_only= false;


  _simple_icon= Gdk::Pixbuf::create_from_file(_icon_path+"/grt_value_simple.png");
  _dict_icon= Gdk::Pixbuf::create_from_file(_icon_path+"/grt_value_dict.png");
  _list_icon= Gdk::Pixbuf::create_from_file(_icon_path+"/grt_value_list.png");
  _struct_icon= Gdk::Pixbuf::create_from_file(_icon_path+"/grt_value_struct.png");

  show_all();
  
  refresh();
}



void MGRTShell::set_icon(Gtk::TreeRow row, MGRTValue value)
{
  std::string path;

  switch (value.type())
  {
  case MYX_DICT_VALUE:
    {
      MYX_GRT_STRUCT *vstr= myx_grt_dict_struct_get(_grt->grt(), value.grtValue());

      if (vstr)
        path= myx_grt_struct_get_icon_path(_grt->grt(), vstr, MYX_IT_SMALL);
      else
        path= "";
      if (!path.empty())
      {
        Glib::RefPtr<Gdk::Pixbuf> icon;
        
        icon= Gdk::Pixbuf::create_from_file(_icon_path+"/"+path);
        if (!icon)
          icon= Gdk::Pixbuf::create_from_file(_icon_path+"/GrtObject.16x16.png");
        if (!icon)
          icon= _struct_icon;
        row[_columns.icon]= icon;
      }
      else
        row[_columns.icon]= _dict_icon;
      break;
    }

  case MYX_LIST_VALUE:
    row[_columns.icon]= _list_icon;
    break;

  default:
    row[_columns.icon]= _simple_icon;
    break;
  }
}



void MGRTShell::add_list_to_store(MGRTValue list, Gtk::TreeRow &parent, Glib::RefPtr<Gtk::TreeStore> store)
{
  unsigned int c= list.count();
  Gtk::TreeIter iter;
  Gtk::TreeRow row;
  const char *sname;
  
  for (unsigned int i= 0; i < c; i++)
  {
    MGRTValue value(list[i]);
    
    iter= store->append(parent.children());
    row= *iter;

    switch (value.type())
    {
    case MYX_STRING_VALUE:
      if (list.listContentStruct())
      {
        MGRTValue rvalue= MGRTValue::refObject(_grt->grt(), value.asString());

        set_icon(row, rvalue);
        row[_columns.text]= rvalue.asString();
        row[_columns.data]= rvalue;
      }
      else
      {
        set_icon(row, value);
        row[_columns.text]= value.asString();
        row[_columns.data]= value;
      }
      break;
      
    case MYX_LIST_VALUE:
      set_icon(row, value);
      sname= value.listContentStruct();
      set_icon(row, value);
      if (sname)
        row[_columns.text]= ufmt("%s    list [dict: %s]",  "", sname);
      else
        row[_columns.text]= ufmt("%s    list [%s]", "", 
                                 myx_get_value_type_as_string(value.listContentType()));
      row[_columns.data]= value;
      add_list_to_store(value, row, store);
      break;

    case MYX_DICT_VALUE:
      set_icon(row, value);
      sname= value.contentStruct();
      if (sname)
        row[_columns.text]= ufmt("%s    dict: %s", value["name"].asString(), sname);
      else
        row[_columns.text]= ufmt("%s    dict", value["name"].asString());
      row[_columns.data]= value;
      add_dict_to_store(value, row, store);
      break;

    case MYX_INT_VALUE:
      set_icon(row, value);
      row[_columns.text]= ufmt("%i", value.asInt());
      row[_columns.data]= value;
      break;
      
    case MYX_REAL_VALUE:
      set_icon(row, value);
      row[_columns.text]= ufmt("%lf", value.asDouble());
      row[_columns.data]= value;
      break;
      
    case MYX_ANY_VALUE:
      row[_columns.text]= "???";
      break;
    }
  }
}


void MGRTShell::add_dict_to_store(MGRTValue dict, Gtk::TreeRow &parent, Glib::RefPtr<Gtk::TreeStore> store)
{
  unsigned int c= _container_only ? dict.count() : dict.countComplex();
  
  Gtk::TreeIter iter;
  
  for (unsigned int i= 0; i < c; i++)
  {
    MGRTValue value;
    const char *key;
    
    if (_container_only)
      dict.complexDictItemByIndex(i, key, value);
    else
      dict.dictItemByIndex(i, key, value);

    iter= store->append(parent.children());
    Gtk::TreeRow row= *iter;

    set_icon(row, value); 
    row[_columns.text]= key;
    row[_columns.data]= value;
  }
}


void MGRTShell::refresh()
{
  Gtk::TreeIter iter;
  Gtk::TreeRow row;
  
  _value_store->clear();
  
  MGRTValue root(_grt->global_value(_root_path));
  
  iter= _value_store->append();
  row= *iter;
  if (_root_path.empty())
    row[_columns.text]= "root";
  else
    row[_columns.text]= myx_grt_dict_name_item_as_string(root.grtValue());
  row[_columns.data]= root;
  add_dict_to_store(root, row, _value_store);
  
  
  // structs
  
  _struct_store->clear();
  
  unsigned int pc= myx_grt_package_count(_grt->grt());
  for (unsigned int i= 0; i < pc; i++)
  {
    char *pkg= myx_grt_package_by_index(_grt->grt(), i);
    
    iter= _struct_store->append();
    row= *iter;
    
    row[_columns.text]= (pkg && *pkg) ? pkg : "<base>";
    row[_columns.data]= pkg;
    
    unsigned int sc= myx_grt_package_struct_count(_grt->grt(), pkg);
    for (unsigned int j= 0; j < sc; j++)
    {
      Gtk::TreeIter iter2= _struct_store->append(row.children());
      Gtk::TreeRow row2= *iter2;
      MYX_GRT_STRUCT *gstruct= myx_grt_package_struct_by_index(_grt->grt(), pkg, j);
      int inherited;
      const char *caption;

      caption= myx_grt_struct_get_caption(_grt->grt(), gstruct, &inherited);
      
      if (inherited && caption && *caption)
      {
        row2[_columns.text]= ufmt("%s   <<%s>>", myx_grt_struct_get_name(gstruct), caption);
        row2[_columns.data]= gstruct;
      }
      else
      {
        row2[_columns.text]= ufmt("%s    (%s)", myx_grt_struct_get_name(gstruct),caption);
        row2[_columns.data]= gstruct;
      }

      unsigned int mc= myx_grt_struct_get_member_count(gstruct);
      for (unsigned int k= 0; k < mc; k++)
      {
        Gtk::TreeIter iter3= _struct_store->append(row2.children());
        Gtk::TreeRow row3= *iter3;
        MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index(gstruct, k);

        row3[_columns.text]= ufmt("%s: %s;", member->name,
                                  myx_get_value_type_as_string(myx_grt_struct_member_get_type(member)));
        row3[_columns.data]= member;
      }
    }
  }

  // modules

  for (int i= 0; i < myx_grt_module_get_count(_grt->grt()); i++)
  {
    MYX_GRT_MODULE *module= myx_grt_module_get_by_index(_grt->grt(), i);
    iter= _module_store->append();
    row= *iter;
    
    row[_columns.text]= module->name;
    row[_columns.data]= module;
    
    for (int j= 0; j < myx_grt_module_function_get_count(module); j++)
    {
      MYX_GRT_FUNCTION *func= myx_grt_module_function_get_by_index(module, j);
      Gtk::TreeIter iter2= _module_store->append(row.children());
      Gtk::TreeRow row2= *iter2;

      row2[_columns.text]= func->name;
      row2[_columns.data]= func;
    }
  }
}

