/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MGRTConnectDialog.h"
#include <gtkmm/entry.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/table.h>
#include <gtkmm/box.h>
#include <gtkmm/treestore.h>
#include "mygpriv.h"
/**
 * @file  MGRTConnectDialog.cc
 * @brief 
 */



  
// ----------------------------------------------------------------------

class MGRTConnectDialog::ParameterMapping {
  public:
    ParameterMapping();
    ~ParameterMapping();

    void update(MGRTValue &paramValues);
    
    Gtk::Label *label;
    char type;
    union {
      Gtk::Widget *widget;
      Gtk::Entry *entry;
      Gtk::CheckButton *check;
    } control;
    MGRTValue *param;
};


MGRTConnectDialog::ParameterMapping::ParameterMapping()
  : label(0), param(0)
{
  control.widget= 0;
}


MGRTConnectDialog::ParameterMapping::~ParameterMapping()
{
  delete param;
}


void MGRTConnectDialog::ParameterMapping::update(MGRTValue &paramValues)
{
  switch (type)
  {
  case 'T':
    paramValues.set((*param)["name"].asString(),
                    control.entry->get_text().c_str());
    break;
  case 'C':
    paramValues.set((*param)["name"].asString(),
                    control.check->get_active() ? "1" : "0");
    break;
  }
}

//----------------------------------------------------------------------

MGRTConnectDialog::MGRTConnectDialog(MGRT *grt, const Glib::ustring &path, const Glib::ustring &target, bool windowed)
  : _window(0), _rdbms_frame(0), _option_frame(0), _advanced_frame(0)
{
  _conn_info_path= path;
  _conn_target_path= target;
  _jdbc_only= false;
}


MGRTConnectDialog::~MGRTConnectDialog()
{
  for (std::map<Glib::ustring,ParameterMapping*>::iterator iter= _parameters.begin();
       iter != _parameters.end(); ++iter)
  {
    delete iter->second;
  }

  delete _window;
  
  delete _rdbms_frame;
  delete _option_frame;
  delete _advanced_frame;
}


void MGRTConnectDialog::setup()
{
  if (_pick_rdbms)
  {
    Gtk::Table *table;
    
    _rdbms_frame= new Gtk::Frame(_("Driver Selection"));
    
    table= new Gtk::Table(2, 3);
    _rdbms_frame->add(*table);
    table->set_border_width(12);
    table->set_row_spacings(6);
    table->set_col_spacings(8);

    _rdbms_combo= new Gtk::ComboBox();
    _driver_combo= new Gtk::ComboBox();
    
    table->attach(*new Gtk::Label(_("Database System:")), 0, 1, 0, 1, 
                  Gtk::FILL, Gtk::AttachOptions(0),
                  0, 0);
    table->attach(*_rdbms_combo, 1, 2, 0, 1, 
                  Gtk::FILL, Gtk::AttachOptions(0),
                  0, 0);

    table->attach(*new Gtk::Label(_("Driver:")), 0, 1, 1, 2,
                  Gtk::FILL, Gtk::AttachOptions(0),
                  0, 0);
    table->attach(*_driver_combo, 1, 2, 1, 2,
                  Gtk::FILL, Gtk::AttachOptions(0),
                  0, 0);
  }
    
  _option_frame= new Gtk::Frame(_("Connection Options"));
    
    
  _advanced_frame= new Gtk::Frame(_("Advanced Options"));
    
  
}
  
    
void MGRTConnectDialog::set_selects_rdbms(bool flag)
{
  _pick_rdbms= flag;
}


void MGRTConnectDialog::set_edits_schema(bool flag)
{
  _pick_schema= flag;
}


int MGRTConnectDialog::add_drivers_to_combo(MGRTValue *rdbms)
{
  MGRTValue drivers((*rdbms)["drivers"]);
  unsigned int i;
  Glib::RefPtr<Gtk::TreeStore> model= Gtk::TreeStore::create(_combo_columns);

  for (i= 0; i < drivers.count(); i++)
  {
    MGRTValue driver(drivers[i]);

    if (_jdbc_only && strcmp(driver.contentStruct(), "db.mgmt.JdbcDriver")!=0)
      continue;

    // we dont support driver fetching yet
    if (!driver["isInstalled"].asInt())
      continue;

    // driverFilter

    Gtk::TreeIter iter;
    
    // place default driver on top of dropdown
    if ((*rdbms)["defaultDriver"].grtValue() == driver.grtValue())
      iter= model->prepend();
    else
      iter= model->append();
    Gtk::TreeRow row= *iter;
    row[_combo_columns.name]= (const char*)driver["caption"];
    row[_combo_columns.data]= driver;
  }

  _driver_combo->set_model(model);
  
  int count= model->children().size();
  
  _driver_combo->set_sensitive(count > 1);

  return count;
}
 

void MGRTConnectDialog::rdbms_selected()
{
  Gtk::TreeIter iter= _rdbms_combo->get_active();
  Gtk::TreeRow row= *iter;
  
  MGRTValue rdbms((MGRTValue)row[_combo_columns.data]);

  add_drivers_to_combo(&rdbms);
  driver_selected();
}


void MGRTConnectDialog::driver_param_value_changed()
{
  if (!_setting_conn_values)
  {
    bool ok= true;
    for (std::map<Glib::ustring,ParameterMapping*>::const_iterator iter= _parameters.begin();
         iter != _parameters.end(); ++iter)
    {
      ParameterMapping *mapping= iter->second;

      if ((*mapping->param)["required"].asInt() == 1 ||
          _pick_schema && strcmp((*mapping->param)["name"].asString(), "schema")==0)
      {
        if (mapping->type == 'T')
        {
          if (mapping->control.entry->get_text() == "")
          {
            ok= false;
            break;
          }
        }
      }
    }
    _signal_ready_changed.emit(ok);
    //_connect_button->set_sensitive(true);
  }
}


void MGRTConnectDialog::build_driver_controls(Gtk::Frame *frame,
                                              MGRTValue *driver,
                                              bool advanced)
{
  unsigned int i;
  MGRTValue params((*driver)["parameters"]);
  int maxRow= 0;
  Gtk::VBox *vbox;
  Gtk::HBox *hbox;

  frame->add(*vbox);
  vbox= new Gtk::VBox();
  
  _first_control= 0;

  for (i= 0; i < params.count(); i++)
  {
    MGRTValue param(params[i]);
    int currentRow;
    ParameterMapping *mapping;

    if ((param["layoutAdvanced"].asInt()!=0) != advanced)
      continue;

    if (!_pick_schema && strcmp(param["name"].asString(), "schema") == 0)
      continue;

    currentRow= param["layoutRow"].asInt();
    if (currentRow == -1)
      currentRow = maxRow + 1;
    
    if (currentRow > maxRow)
    {
      hbox= new Gtk::HBox();
      vbox->pack_start(*hbox, false, true);
    }
    maxRow = currentRow;

    const char *caption= param["caption"];
    const char *descr= param["description"];
    const char *defVal= param["defaultValue"];
    const char *paramType= param["paramType"];

    mapping= new ParameterMapping();

    _parameters[(const char*)param["name"]]= mapping;

    mapping->param= new MGRTValue(param);

    if (strcmp(paramType, "boolean")!=0 && strcmp(paramType, "tristate")!=0)
    {
      if (caption && *caption)
      {
        mapping->label= new Gtk::Label(caption, 1.0);

        hbox->pack_start(*mapping->label, false, true);
      }

      mapping->type= 'T';
      
      // create param edit
      mapping->control.entry= new Gtk::Entry();
      if (strcmp(paramType, "password")==0)
        mapping->control.entry->set_visibility(false);

      mapping->control.entry->signal_changed().connect(sigc::mem_fun(*this,&MGRTConnectDialog::driver_param_value_changed));

      mapping->control.entry->set_text(defVal);
    }
    else
    {
      mapping->type= 'C';

      // create checkbox
      mapping->control.check = new Gtk::CheckButton();
      
      mapping->control.check->signal_toggled().connect(sigc::mem_fun(*this,&MGRTConnectDialog::driver_param_value_changed));

      mapping->control.check->set_active(strcmp(defVal, "1")==0);

      mapping->control.check->set_label(caption);
    }

    if (!_first_control)
      _first_control= mapping->control.widget;

    int width= param.get("layoutWidth", 200);
    {
      int w, h;
      mapping->control.widget->get_window()->get_size(w,h);
      if (width < w)
          width= w;
    }
    mapping->control.widget->set_size_request(width);

    if (descr && *descr)
      _tooltip.set_tip(*mapping->control.widget, descr);

    hbox->pack_start(*mapping->control.widget, false, true);

    // XXX add lookup button

    // build param description
    /*
    if (_showDescriptions)
    {
      NSTextField *text;

      text= [[[NSTextField alloc] init] autorelease];
      [text setStringValue:NSStr(descr)];
      [text setEditable:NO];
      [text sizeToFit];
      [text setFrameOrigin:NSMakePoint(offsetLeft + currentLeft[currentRow], currentTop + 4)];
      [container addSubview:text];
    }*/
  }
}


void MGRTConnectDialog::fill_stored_connections_for_driver(MGRTValue *driver)
{
  
}


void MGRTConnectDialog::driver_selected()
{
  if (_option_frame->get_child())
    _option_frame->remove();
  if (_advanced_frame->get_child())
    _advanced_frame->remove();
  
  for (std::map<Glib::ustring,ParameterMapping*>::iterator iter= _parameters.begin();
       iter != _parameters.end(); ++iter)
  {
    delete iter->second;
  }
  _parameters.clear();
  
  _first_control= 0;

  Gtk::TreeIter iter= _driver_combo->get_active();
  Gtk::TreeRow row= *iter;
  
  MGRTValue driver((MGRTValue)row[_combo_columns.data]);

  if (driver["isInstalled"].asInt())
  {
    if (_connection_combo->get_model()->children().size() > 0)
      _connection_combo->set_active(0);

    build_driver_controls(_option_frame, &driver, false);

    build_driver_controls(_advanced_frame, &driver, true);

    fill_stored_connections_for_driver(&driver);
  }

  if (_first_control)
    _first_control->grab_focus();
}


void MGRTConnectDialog::connection_changed()
{
}


void MGRTConnectDialog::refresh_rdbms_info()
{
  MGRTValue rdbmsList(_grt->global_value(_conn_info_path+"/rdbms"));
  Glib::RefPtr<Gtk::TreeStore> rdbmsModel;
  Gtk::TreeIter iter;
  int mysqlIndex= -1;
  unsigned int i;

  rdbmsModel= Gtk::TreeStore::create(_combo_columns);
  _rdbms_combo->set_model(rdbmsModel);

  for (i= 0; i < rdbmsList.count(); i++)
  {
    MGRTValue rdbms(rdbmsList[i]);

    if (strcasecmp(rdbms["name"].asString(), "MySQL")!=0)
      continue;
    else
      mysqlIndex= _rdbms_combo->get_model()->children().size();

    if (add_drivers_to_combo(&rdbms) > 0)
    {
      iter= rdbmsModel->append();
      Gtk::TreeRow row= *iter;
      
      row[_combo_columns.name]= (const char*)rdbms["caption"];
      row[_combo_columns.data]= rdbms;
    }
  }
  _rdbms_combo->set_active(mysqlIndex);
  rdbms_selected();
  connection_changed();
}


MGRTValue MGRTConnectDialog::write_connection_to_target()
{
  Gtk::TreeIter iter= _driver_combo->get_active();
  Gtk::TreeRow row= *iter;
  MGRTValue driver(row[_combo_columns.data]);
  MGRTValue conn(MGRTValue::createObject(_grt->grt(), "db.mgmt.Connection"));

  conn.set("driver", driver.dictId());

  MGRTValue paramValues(MGRTValue::createTypedDict(MYX_STRING_VALUE));

  conn.set("parameterValues", paramValues);
  for (std::map<Glib::ustring,ParameterMapping*>::iterator iter= _parameters.begin();
       iter != _parameters.end(); ++iter)
  {
    ParameterMapping *mapping= iter->second;

    mapping->update(paramValues);
  }

  conn.set("modules", driver["defaultModules"].copy().grtValue());

  _grt->set_global_value(_conn_target_path, conn);

  return conn;
}
 
