/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.h 
 * @brief Implementation of the GC layer class.
 * 
 */

#ifndef __GC_LAYER_H__
#define __GC_LAYER_H__

#include "myx_gc_base.h"

//----------------------------------------------------------------------------------------------------------------------

class CGenericCanvas;
class CFigureInstance;
class CFigure;
class CHitResults;
class CConnection;
class CConnectionInstance;

/**
 * This is the base layer class, which is used by views to display their content. There are descendants for special things
 * like feedback, grids and so on.
 */
class GENERIC_CANVAS_API CLayer: public CGCBase
{
  friend class CFigureInstance;
  friend class CFigureInstanceEnumerator;
  friend class CInstanceListener;
private:
  class CInstanceListener: private CGCListener
  {
    friend class CLayer;
  protected:
    CLayer* layer;
  public:
    virtual void __cdecl onAction(CGCBase* sender, CGCBase* origin, TActionType& action);
    virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
    virtual void __cdecl onDestroy(CGCBase* object);
    virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
  };

  wstring FName;                  // The name of the layer.
  bool FDirty;                    // True if any of the properties changed that affect the display list.
  float FScaling[3];              // The factors to scale the layer in each direction.
  float FMinimalScale;            // The lesser of the x and y scale factors.
  float FTranslation[3];          // The factors to move the layer.
  CFigureInstances FInstances;    // A list of figures instances.
  CFigureInstances FGroups;       // A list of groups.
  bool FVisible;                  // true if the layer is visible, otherwise false. Default is true.
  bool FEnabled;                  // True if the layer answers to mouse and keyboard actions.
  CInstanceListener FListener;
protected:
  void applyTransformations();
  void checkError(void);
  void makeDirty(void);
  void renderFeedback(CFigureInstance* instance);
  virtual void renderLayerContent(float currentZoom, TBoundingBox bounds);
  void validate(void);
  virtual void validateLayerContent(void);
public:
  CLayer(string name, CGenericCanvas* canvas);
  virtual ~CLayer(void);

  virtual void __cdecl addInstance(CFigureInstance* instance);
  virtual void __cdecl bringToFront(CFigureInstance* instance, CFigureInstance* relativeTo = NULL);
  virtual void __cdecl clear(void);
  virtual CFigureInstance* __cdecl createInstance(CFigure* figure);
  void getHitTestInfoAt(CHitResults* hits, TVertex point, bool singleHit);
  virtual void __cdecl enabled(bool isEnabled);
  virtual bool __cdecl enabled(void);
  virtual TVertex __cdecl layerToView(TVertex coords);
  wstring name(void) { return FName; };
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, TGCVariant value);
  virtual void __cdecl removeInstance(CFigureInstance* instance);
  virtual void __cdecl render(float currentZoom, TBoundingBox bounds);
  virtual void __cdecl scale(const float Factor[3], bool accumulative = false);
  virtual void __cdecl scale(float Sx, float Sy, float Sz, bool accumulative = false);
  virtual void __cdecl sendToBack(CFigureInstance* instance, CFigureInstance* relativeTo = NULL);
  virtual void __cdecl translate(float Tx, float Ty, float Tz, bool accumulative = false);
  virtual void __cdecl translateV(const float Factor[3], bool accumulative = false);
  virtual TVertex __cdecl viewToLayer(TVertex coords);
  virtual void __cdecl visible(bool isVisible);
  virtual bool __cdecl visible(void);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * The grid layer is a special layer variant that renders itself as grid.
 */
class GENERIC_CANVAS_API CGridLayer: public CLayer
{
private:
  class CViewListener: public CGCListener
  {
    friend class CGridLayer;
  protected:
    CGridLayer* layer;
  public:
    virtual void __cdecl onAction(CGCBase* sender, CGCBase* origin, TActionType& action) { };
    virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
    virtual void __cdecl onDestroy(CGCBase* object);
    virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
  };

  CGCView* FView;       // The view to which this layer belongs.
  CViewListener FListener;
protected:
  virtual void renderLayerContent(float currentZoom, TBoundingBox bounds);
public:
  CGridLayer(CGCView* view);
  virtual ~CGridLayer(void);

};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_LAYER_H__
                     