unit Workbench;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


interface

uses
  gnugettext, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, TntExtCtrls, Sections, TntComCtrls, TntForms, ComCtrls,
  TabHeader, StdCtrls, TntStdCtrls, pngimage, Toolbar, PNGTools,
  GenericCanvasViewer,
  TntDialogs, TntSysUtils, AuxFuncs, ApplicationDataModule,
  myx_grt_public_interface, GrtSections,
  Grt, GrtObjectTree, ActnList, XPStyleActnCtrls, ActnMan, Menus, TntMenus;

  TWorkbenchForm = class(TTntForm)
    constructor Create(AOwner: TComponent; StatusBar: TTntStatusBar = nil; Grt: TGrt = nil); reintroduce;
    destructor Destroy; override;
    procedure Open(Sender: TObject);
    procedure PalettesSizerPnlMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PalettesSizerPnlMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ProcessTextOutput(Text: WideString);
    procedure Save(Sender: TObject);
    procedure SaveAs(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    FLastMousePos: TPoint;
    FMainLayer: TLayer;
    FOriginalSideBarWidth: Integer;
    FPaperLayer: TLayer;
    FToolbars: TToolbars;
    FViewerContainer: TTntPanel;
    FActivatingView: Boolean;
    FPaletteContainer: TToolbarContainer;

    procedure ViewTabPageChange(Sender: TObject; PreviousSelectedTab: Integer; PreviousSelectedObj: TObject;
      NewSelectedTab: Integer; Obj: TObject);
    procedure CreateFigures(Amount, FigureType: Integer);
  protected
    function ActivateView(Name: PChar): TView;
    procedure BuildPalettes;
    procedure BuildToolbars;
    procedure CursorBtnClick(Sender: TObject);
    procedure DoTabChange(Sender: TObject; PreviousSelectedTab: Integer; PreviousSelectedObj: TObject;
      NewSelectedTab: Integer; obj: TObject);
    function GetWorkbenchObject: Pointer;
    procedure InitializeCanvas;
    procedure PreparePaper;
    procedure Refresh;
    procedure SetCurrentTool(CurrentTool: TWorkbenchTool);
    procedure SetGrt(Grt: TGrt); override;
    procedure SetWorkbenchObject(NewValue: Pointer);
    procedure ViewerChange(Sender, Source: TObject; Reason: TGCChangeReason);
    procedure ViewerError(Sender: TObject; const Message: string);
    procedure ViewerZoomChange(Sender: TObject);

    procedure SetActionEvents;
    procedure DoFileOpenAccept(Sender: TObject);
    procedure DoFileOpenFromDb(Sender: TObject);
  public
    procedure AddView;
    procedure InitializeControls; override;
    procedure OpenWorkbench(FileName: WideString = '');
    procedure SaveWorkbench(FileName: WideString = '');
    property CurrentTool: TWorkbenchTool read FCurrentTool write SetCurrentTool;
    property WorkbenchObject: Pointer read GetWorkbenchObject write SetWorkbenchObject;
  end;

var
  WorkbenchForm: TWorkbenchForm;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

uses
  myx_grt_gc_public_interface;
  
const
  BaseWidth  = 10000;
  BaseHeight = 7000;

//----------------------------------------------------------------------------------------------------------------------

constructor TWorkbenchForm.Create(AOwner: TComponent; StatusBar: TTntStatusBar; Grt: TGrt);

begin
  inherited Create(AOwner, StatusBar, Grt);

  DockedPanel := DockPanel;
  BuildToolbars;
  BuildPalettes;
  FCurrentTool := wbToolArrow;

  // Create views tabframe.
  ViewTabFrame.ClearTabSheets;
  ViewTabFrame.ShowButtons := True;
  AddView;
  ViewTabFrame.SelectedTab := 0;
  ViewTabFrame.OnPageChange := ViewTabPageChange;

  // Restore stored width of the sidebar.
  //SidebarPnl.Width := ApplicationDM.Options.SideBarWidth;

  // Set action events
  SetActionEvents;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TWorkbenchForm.Destroy;

begin
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.AddView;

var
  ViewName: WideString;

begin
  if FViewerContainer = nil then
  begin
    FViewerContainer := TTntPanel.Create(Self);
    FViewerContainer.Parent := CanvasPanel;
    FViewerContainer.Align := alClient;
  end;

  ViewName := Format(_('View %d'), [ViewTabFrame.TabCount + 1]);
  // Do not create a view for the first frame. There is a default view already.
  if ViewTabFrame.TabCount > 0 then
    FCanvasViewer.Canvas.CreateView(PChar(UTF8Encode(ViewName)));
  ViewTabFrame.AddTabSheet(Self, ViewName, 'tabsheet_icon_layer', FViewerContainer, nil, True, False, -1, []);
end;

//----------------------------------------------------------------------------------------------------------------------

function TWorkbenchForm.ActivateView(Name: PChar): TView;

// Activates the canvas view given by Name.

var
  Color: COLORREF;
  Background: array[0..3] of Single;
  View: TView;
  ZoomX: Single;
  ZoomY: Single;
  Tab: TTab;

begin
  if (Name = 'View 1') then
    Name := 'Default';

  FActivatingView := True;
  try
    View := FCanvasViewer.Canvas.ViewByName(Name);
    ZoomX := 0;
    ZoomY := 0;
    if View = nil then
    begin
      // Keep the current zoom factor for the new view.
      ZoomX := FCanvasViewer.ZoomX;
      ZoomY := FCanvasViewer.ZoomY;

      View := FCanvasViewer.Canvas.CreateView(Name);
      Color := ColorToRGB(clBtnShadow);
      Background[0] := GetRValue(Color) / 255;
      Background[1] := GetGValue(Color) / 255;
      Background[2] := GetBValue(Color) / 255;
      Background[3] := 0;
      View.Color(Background);
      View.AddLayer(FPaperLayer);

      Tab := ViewTabFrame.TabSheetsByName[Name];
      if (Tab <> nil) then
        ViewTabFrame.SelectedTabSheet := Tab
      else
        ViewTabFrame.AddTabSheet(Self, Name, 'tabsheet_icon_layer', FViewerContainer, nil, True, False, -1, []);
    end;
    FCanvasViewer.Canvas.CurrentView(View);
    if ZoomX > 0 then
    begin
      FCanvasViewer.SetZoom(ZoomX, ZoomY, Point(FCanvasViewer.ClientWidth div 2,
        FCanvasViewer.ClientHeight div 2));
    end;

    Result := View;
  finally
    FActivatingView := False;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.BuildPalettes;

begin
  FGrtObjectTree := TGrtObjectTree.Create(Self);
  FGrtObjectTree.Parent := PalSchemaPnl;
  FGrtObjectTree.Align := alClient;

  FGrtModelTree := TGrtObjectTree.Create(Self);
  FGrtModelTree.Parent := PalModelPnl;
  FGrtModelTree.Align := alClient;

  //Set properties of sidebar tabcontrols
  Pal1TabFrame.DrawBottom := True;
  Pal1TabFrame.ShowDeleteButtons := False;
  Pal1TabFrame.ClearTabSheets;
  Pal1TabFrame.AddTabSheet(Pal1Pnl, _('Schemata'), '', PalSchemaPnl, nil, True, True, -1, []);
  Pal1TabFrame.AddTabSheet(Pal1Pnl, _('Datatypes'), '', PalDatatypePnl, nil, False, True, -1, []);
  Pal1TabFrame.OnPageChange := DoTabChange;

  Pal2TabFrame.DrawBottom := True;
  Pal2TabFrame.ShowDeleteButtons := False;
  Pal2TabFrame.ClearTabSheets;
  Pal2TabFrame.AddTabSheet(Pal2Pnl, _('Layers'), '', PalModelPnl, nil, True, True, -1, []);
  Pal2TabFrame.AddTabSheet(Pal2Pnl, _('History'), '', PalHistroyPnl, nil, False, True, -1, []);
  Pal2TabFrame.AddTabSheet(Pal2Pnl, _('Description'), '', PalDescriptionPnl, nil, False, True, -1, []);
  Pal2TabFrame.OnPageChange := DoTabChange;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.BuildToolbars;

var
  PNGImg: TPNGObject;
  ToolbarBGBitmap: TBitmap;
  CurrentGroup: TToolbarItemGroup;
  ToolbarContainer: TToolbarContainer;

begin
  FToolbars := TToolbars.Create;

  FPaletteContainer := FToolbars.AddToolbar(ModelToolbarPnl,
    'Model', False, True, -10);

  ToolbarContainer := FToolbars.AddToolbar(ToolOptionsToolbarPnl,
   'Toolbar', True{ApplicationDM.Options.UseToolbarGradient}, False);

  //Make background
  if (True{ApplicationDM.Options.UseToolbarGradient}) then
  begin
    //Toolbar BG
    PNGImg := LoadPNGImageFromResource('toolbar_bg2');
    ToolbarBGBitmap := TBitmap.Create;
    ToolbarBGBitmap.Width := PNGImg.Width;
    ToolbarBGBitmap.Height := PNGImg.Height;
    ToolbarBGBitmap.Canvas.Brush.Color := clBtnFace;
    ToolbarBGBitmap.Canvas.FillRect(Rect(0, 0,
      ToolbarBGBitmap.Width, ToolbarBGBitmap.Height));
    PNGImg.Draw(ToolbarBGBitmap.Canvas, Rect(0, 0,
      PNGImg.Width, PNGImg.Height));
    PNGImg.Free;

    ToolbarContainer.BGBitmap := ToolbarBGBitmap;

    ToolbarBGBitmap.Free;
  end;

  CurrentGroup := FPaletteContainer.AddGroup('MouseStd', False);

  // -----------------------------------------
  CurrentGroup.AddItem(
    'CursorArrowBtn', 'wb_arrow',
    _('Arrow'), _('Standard mouse pointer'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, True, True);

  CurrentGroup.AddItem(
    'CursorHandBtn', 'wb_hand',
    _('Hand'), _('The hand tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorMaginifierBtn', 'wb_magnifier',
    _('Magnifier'), _('The magnifier tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorExposeBtn', 'wb_expose',
    _('Expose'), _('The expose tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  // -----------------------------------------

  CurrentGroup := FPaletteContainer.AddGroup('MouseRubber',
    True, False, 2);
  CurrentGroup.Space := 0;

  CurrentGroup.AddItem(
    'CursorRubberBtn', 'wb_rubber',
    _('Rubber'), _('The rubber tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  // -----------------------------------------

  CurrentGroup := FPaletteContainer.AddGroup('MouseDoc',
    True, False, 2);
  CurrentGroup.Space := 0;

  CurrentGroup.AddItem(
    'CursorRegionBtn', 'wb_region',
    _('Region'), _('The region tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorNoteBtn', 'wb_note',
    _('Note'), _('The note tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorImageBtn', 'wb_image',
    _('Image'), _('The image tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  // -----------------------------------------

  CurrentGroup := FPaletteContainer.AddGroup('MouseRel',
    True, False, 2);
  CurrentGroup.Space := 0;

  CurrentGroup.AddItem(
    'CursorRel1nBtn', 'wb_rel_1n',
    _('Relationship 1 to many'), _('The 1:n tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorRel11Btn', 'wb_rel_11',
    _('Relationship 1 to 1'), _('The 1:1 tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorRelnmBtn', 'wb_rel_nm',
    _('Relationship many to many'), _('The n:m tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  // -----------------------------------------

  CurrentGroup := FPaletteContainer.AddGroup('MouseDb',
    True, False, 2);
  CurrentGroup.Space := 0;

  CurrentGroup.AddItem(
    'CursorTableBtn', 'wb_db_table',
    _('Table'), _('The table tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorViewBtn', 'wb_db_view',
    _('View'), _('The view tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'CursorSPBtn', 'wb_db_sp',
    _('Stored procedure'), _('The SP tool'),
    2, False, False, False, CursorBtnClick,
    nil, nil, True, False, True);


  // -----------------------------------------
  // Toolbar

  CurrentGroup := ToolbarContainer.AddGroup('UndoRedoSync',
    True, False, 5);
  CurrentGroup.Space := 6;

  CurrentGroup.AddItem(
    'UndoBtn', 'tiny_undo',
    _('Undo'), _('Undo recent changes'),
    3, False, False, False, nil,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'RedoBtn', 'tiny_redo',
    _('Redo'), _('Redo last undone change'),
    3, False, False, False, nil,
    nil, nil, True, False, True);

  CurrentGroup.AddItem(
    'SyncBtn', 'tiny_execute',
    _('Synchronize'), _('Synchronize model with database'),
    3, False, False, False, nil,
    nil, nil, True, False, True);

  // -----------------------------------------

  CurrentGroup := ToolbarContainer.AddGroup('OpenSave',
    True, False, 5);
  CurrentGroup.Space := 6;

  CurrentGroup.AddItem(
    'OpenBtn', 'tiny_open',
    _('Open'), _('Open model'),
    3, False, False, False, nil,
    nil, nil, True, False, True,
    ApplicationDM.FileOpenAction);

  CurrentGroup.AddItem(
    'SaveBtn', 'tiny_save',
    _('Save'), _('Save model'),
    3, False, False, False, nil,
    nil, nil, True, False, True,
    ApplicationDM.FileSaveAction);

  // -----------------------------------------

  CurrentGroup := ToolbarContainer.AddGroup('GridRuler',
    True, False, 5);
  CurrentGroup.Space := 6;

  CurrentGroup.AddItem(
    'GridBtn', 'tiny_grid',
    _('Grid'), _('Display grid'),
    3, False, False, False, nil,
    nil, nil, True, False, True,
    ApplicationDM.SnapToGridAction);

  CurrentGroup.AddItem(
    'RulerBtn', 'tiny_ruler',
    _('Ruler'), _('Display ruler'),
    3, False, False, False, nil,
    nil, nil, True, False, True,
    ApplicationDM.DisplayRulersAction);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.CursorBtnClick(Sender: TObject);

var
  Item: TToolbarItem;
  Group: TToolbarItemGroup;
  Container: TToolbarContainer;
  
begin
  if (Sender is TPNGButton) then
  begin
    Item := TToolbarItem(TPNGButton(Sender).Owner);
    Group := Item.ToolbarItemGroup;
    Container := Group.ToolbarContainer;

    CurrentTool := TWorkbenchTool(
      Container.GetGroupIndex(Group) * 10 +
      Item.ToolbarItemGroup.GetItemIndex(Item));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.DoTabChange(Sender: TObject; PreviousSelectedTab: Integer; PreviousSelectedObj: TObject;
  NewSelectedTab: Integer; obj: TObject);

begin
  //
end;

//----------------------------------------------------------------------------------------------------------------------

function TWorkbenchForm.GetWorkbenchObject: Pointer;

begin
  Result := myx_grt_dict_item_get_value(myx_grt_get_root(Grt.PGrt), 'workbench');
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.InitializeCanvas;

var
  Error: TGCError;
  Color: COLORREF;
  Background: array[0..3] of Single;
  
begin
  FCanvasViewer := TGenericCanvasViewer.Create(Self);
  with FCanvasViewer do
  begin
    OnError := ViewerError;
    OnChange := ViewerChange;
    OnZoomChange := ViewerZoomChange;

    Parent := FViewerContainer;
    Align := alClient;
    BaseSizeX := BaseWidth;
    BaseSizeY := BaseHeight;

    Color := ColorToRGB(clBtnShadow);
    Background[0] := GetRValue(Color) / 255;
    Background[1] := GetGValue(Color) / 255;
    Background[2] := GetBValue(Color) / 255;
    Background[3] := 0;
    Canvas.CurrentView.Color(Background);

    AutoScrollInterval := 500;
    //Options := Options + [cvoAutoCenterZoom] - [];

    Error := LoadStyles('./xml/layout.styles.db.xml');
    if Error <> GC_NO_ERROR then
      raise Exception.Create('XML parse error.');
    Error := LoadLayouts('./xml/layout.figures.db.xml');
    if Error <> GC_NO_ERROR then
      raise Exception.Create('XML parse error.');
    
    PreparePaper;
    FMainLayer := Canvas.CreateLayer('Main', True);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.InitializeControls;

begin
  inherited;

  InitializeCanvas;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.Open(Sender: TObject);

begin
  OpenWorkbench;
end;

//----------------------------------------------------------------------------------------------------------------------

// TODO: Temporary method. Only for testing. Remove as soon as possible!
procedure TWorkbenchForm.CreateFigures(Amount: Integer; FigureType: Integer);

const
  Objects: array[0..6] of PChar = (
    'model',
    'simple',
    'db.Table',
    'column',
    'structured-datatype',
    'simple-datatype',
    'distinct-datatype'
  );

var
  Figure: TFigure;
  Scale: Double;
  I: Integer;
  Instance: TFigureInstance;

begin
  for I := 0 to Amount - 1 do
  begin
    if FigureType > -1 then
    begin
      Figure := FCanvasViewer.Canvas.CreateFigure(Objects[FigureType], 'full');
      Scale := 1;
    end
    else
    begin
      Figure := FCanvasViewer.Canvas.CreateFigure(Objects[Random(7)], 'full');
      Scale := 0.3 + Random(1);
    end;
    Instance := FMainLayer.CreateInstance(Figure);
    Instance.Scale(Scale, Scale, 1, False);
    Instance.Translate(Random(BaseWidth) * 0.9, Random(BaseHeight) * 0.9, 0, False);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.OpenWorkbench(FileName: WideString);

var
  OpenDlg: TTntOpenDialog;
  NewWorkbenchObject: Pointer;
  {Figure: TFigure;
  Layer: TLayer;
  View: TView;}

begin
  if (FileName = '') then
  begin
    OpenDlg := TTntSaveDialog.Create(Self);
    try
      OpenDlg.Title := _('Load Workbench');

      OpenDlg.DefaultExt := 'mwb';
      OpenDlg.Filter := _('MySQL Workbench') + ' (*.mwb)|*.mwb|' +
        _('XML files') + ' (*.xml)|*.xml|' +
        _('All files') + ' (*.*)|*.*';

      OpenDlg.InitialDir := ApplicationDM.GetLastFileDialogPaths('LoadWorkbench');

      if (OpenDlg.InitialDir = '') then
        OpenDlg.InitialDir := ApplicationDM.GetLastFileDialogPaths('SaveWorkbench');

      if OpenDlg.Execute then
      begin
        FileName := OpenDlg.FileName;
        ApplicationDM.SetLastFileDialogPaths('LoadWorkbench',  WideExtractFilePath(FileName));
      end;
    finally
      OpenDlg.Free;
    end;
  end;

  if FileName <> '' then
  begin
    NewWorkbenchObject := myx_grt_retrieve_from_file(Grt.PGrt, FileName);
    WorkbenchObject := NewWorkbenchObject;
    Grt.GlobalAsString['/workbench/options/filename'] := FileName;
    Refresh;

    CreateFigures(100, 2);

    ViewTabFrame.SelectedTab := 0;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.PalettesSizerPnlMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

begin
  if (Shift = [ssLeft]) then
  begin
    FOriginalSideBarWidth := ApplicationDM.Options.SideBarWidth;
    FLastMousePos.X := Mouse.CursorPos.X;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.PalettesSizerPnlMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);

begin
  if (Shift = [ssLeft]) then
  begin
    ApplicationDM.Options.SideBarWidth := FOriginalSideBarWidth - Mouse.CursorPos.X + FLastMousePos.X;
    SidebarPnl.Width := ApplicationDM.Options.SideBarWidth;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.PreparePaper;

var
  Paper: TFigure;
  
begin
  if FPaperLayer = nil then
  begin
    FPaperLayer := FCanvasViewer.Canvas.CreateLayer('Paper', True);
    FPaperLayer.Enabled(False);
  end;
  Paper := FCanvasViewer.Canvas.CreateFigure('paper', 'full');
  FPaperLayer.CreateInstance(Paper);
  Paper.Scale(BaseWidth / 62, BaseHeight / 62, 1, False);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.ProcessTextOutput(Text: WideString);

begin
  if (Trim(Text) <> '') then
    StatusBar.Panels[2].Text := Text;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.Refresh;

begin
  FGrtObjectTree.ObjectTreeRoot := '/workbench/catalog/schemata';

  FGrtModelTree.ObjectTreeRoot := '/workbench/layers';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.Save(Sender: TObject);

begin
  SaveWorkbench(Grt.GlobalAsString['/workbench/options/filename']);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SaveAs(Sender: TObject);

begin
  SaveWorkbench;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SaveWorkbench(FileName: WideString);

var
  SaveDlg: TTntSaveDialog;
  Error: MYX_GRT_ERROR;
  Text: string;

begin
  if (FileName = '') then
  begin
    SaveDlg := TTntSaveDialog.Create(Self);
    try
      SaveDlg.Title := _('Save Workbench As ...');

      SaveDlg.DefaultExt := 'mwb';
      SaveDlg.Filter := _('MySQL Workbench') + ' (*.mwb)|*.mwb|' +
        _('XML files') + ' (*.xml)|*.xml|' +
        _('All files') + ' (*.*)|*.*';

      SaveDlg.InitialDir :=
        ApplicationDM.GetLastFileDialogPaths('SaveWorkbench');

      if (SaveDlg.Execute) then
      begin
        FileName := SaveDlg.FileName;

        ApplicationDM.SetLastFileDialogPaths('SaveWorkbench',
          WideExtractFilePath(FileName));
      end;
    finally
      SaveDlg.Free;
    end;
  end;

  if (FileName <> '') then
  begin
    Grt.GlobalAsString['/workbench/options/filename'] := FileName;

    Error := myx_grt_store_to_file(Grt.PGrt, WorkbenchObject, FileName);

    if Error <> MYX_GRT_NO_ERROR then
    begin
      Text := Format(_('The following error occured while storing the workbench. Error: %d.'), [Ord(Error)]);
      ShowModalDialog(_('Error'), Text, myx_mtError, _('Ok'));
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SetCurrentTool(CurrentTool: TWorkbenchTool);

var
  I,
    J: Integer;
  Group: TToolbarItemGroup;

begin
  if (FCurrentTool <> CurrentTool) then
  begin
    FCurrentTool := CurrentTool;

    for I := 0 to FPaletteContainer.GroupCount - 1 do
    begin
      Group := FPaletteContainer.GroupsByIndex[I];

      for J := 0 to Group.ItemCount - 1 do
      begin
        if (I * 10 + J <> Ord(CurrentTool)) and
          (Group.ItemsByIndex[J].Down) then
          Group.ItemsByIndex[J].Down := False
        else
          if (I * 10 + J = Ord(CurrentTool)) then
            Group.ItemsByIndex[J].Down := True;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SetGrt(Grt: TGrt);

begin
  inherited SetGrt(Grt);

  FGrtObjectTree.Grt := Grt;
  FGrtObjectTree.ObjectTreeRoot := '/workbench/catalog/schemata';

  FGrtModelTree.Grt := Grt;
  FGrtModelTree.ObjectTreeRoot := '/workbench/layers';

  myx_grt_gc_bridge_initialize(Grt.PGrt, FCanvasViewer.Canvas, '/workbench/canvas');
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SetWorkbenchObject(NewValue: Pointer);

begin
  //Store
  myx_grt_dict_item_set_value(myx_grt_get_root(Grt.PGrt), 'workbench', NewValue);

  //Release value one time, so it is only registered in the root dict
  myx_grt_value_release(NewValue);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.ViewerChange(Sender, Source: TObject; Reason: TGCChangeReason);

begin

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.ViewerError(Sender: TObject; const Message: string);

begin
  StatusBar.Panels[2].Text := Message;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.ViewerZoomChange(Sender: TObject);

begin

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);

begin
  if Assigned(FCanvasViewer) then
  begin
    myx_grt_gc_bridge_free(Grt.PGrt, FCanvasViewer.Canvas, '/workbench/canvas');
    FreeAndNil(FCanvasViewer);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.ViewTabPageChange(Sender: TObject; PreviousSelectedTab: Integer; PreviousSelectedObj: TObject;
  NewSelectedTab: Integer; Obj: TObject);

begin
  if (Not(FActivatingView)) then
  begin
    if NewSelectedTab = 0 then
      ActivateView('Default')
    else
      ActivateView(PChar('View ' + IntToStr(NewSelectedTab + 1)));
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.SetActionEvents;

begin
  ApplicationDM.FileOpenAction.OnAccept := DoFileOpenAccept;
  ApplicationDM.FileOpenFromDbAction.OnExecute := DoFileOpenFromDb;
end;


//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.DoFileOpenAccept(Sender: TObject);

begin
  //
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchForm.DoFileOpenFromDb(Sender: TObject);

begin
  //
end;

//----------------------------------------------------------------------------------------------------------------------

end.

