/* Copyright (C) 2003, 2004, 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#define BACKUP_CONTENT_BUFFER_LEN 256 //used by get_backup_content
#define DEFAULT_SCHEMA_NAME "DEFAULT_SCHEMA"
#define DEFAULT_CATALOG_NAME "DEFAULT_CATALOG"
#define ADMIN_DUMP_VERSION "1.4"
#define TABLE_NAME_PCRE "(`[^`]+`|\"[^\"]+\"|\\S+)"

#define INITIAL_BUFFER_LENGTH 1024*1024 /* initial size of 1M */

#include "myx_admin_library.h"
#include "myx_util_functions.h"
#include "myx_simple_sql_parsing.h"
#include "myx_xml_util_functions.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "myx_library.h"
#include <errno.h>
#include <ctype.h>
#include <glib/gprintf.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <time.h>
#include <string.h>

//#define MYX_DEBUG_MEMORY
//#include "myx_debug_memory.h"

// TODO: add gettext support
#define _(s) s

/* structure declarations */



/* for getting the content of a backup-file */
typedef struct
{
  MYX_INTL_FILE *intl_file;  
  MYX_BACKUP_CONTENT *backup_content;
  MYX_SQL_PARSE_ENVIRONMENT we;
  pcre *re_create;
  pcre *re_create_index;
  pcre *re_create_view;
  pcre *re_create_proc;
  pcre *re_create_func;
  pcre *re_db_use;
  pcre_extra *pe_create;
  pcre_extra *pe_create_index;
  pcre_extra *pe_create_view;
  pcre_extra *pe_create_proc;
  pcre_extra *pe_create_func;
  pcre_extra *pe_db_use;
  char *cur_db_name;
  const char *default_catalog_name;
  const char *default_schema_name;
} MYX_BCS_STATUS;


/* for restoring backup-files */
typedef struct
{
  MYX_INTL_FILE *intl_file;
  MYX_BACKUP_CONTENT *backup_content;
  MYX_SQL_PARSE_ENVIRONMENT we;
  pcre *re_create; 
  pcre *re_create_index;
  pcre *re_create_view;
  pcre *re_create_proc;
  pcre *re_create_func;
  pcre *re_db_use;
  pcre *re_drop;
  pcre *re_drop_view;
  pcre *re_drop_proc;
  pcre *re_drop_func;
  pcre *re_insert;
  pcre *re_alter;
  pcre *re_lock;
  pcre *re_unlock;
  pcre *re_set;  
  pcre *re_charset_set1;
  pcre *re_charset_set2;
  pcre *re_charset_set3;
  pcre *re_charset_set4;
  pcre *re_charset_set5;
  pcre *re_create_database;
  pcre *re_delimiter;
  const char *target_catalog;
  const char *target_schema;
  char *current_schema_in_sqlfile; /* gets updated when we read 'USE db' 
                                      in the sql-file */
  char *current_catalog_in_sqlfile; /* In the future when we have catalog-
                                       support this is going to be updated
                                       by 'USE CATALOG catalogname' or so*/

  char *current_schema_in_db; /* keeps track of the currently chosen default
                                 database. the variable gets updated when we
                                 issue 'USE db' of course. */
  MYSQL *mysql;
  int options;
  char *buffer;
  int buffer_len;
  void (*report_warning) (const char *msg, void *user_data);
  void *report_warning_data;
} MYX_RBS_STATUS;

/* for backing up to sql-files */
typedef struct
{
  MYSQL_RES *mysql_result;
  FILE *sql_file;
  MYX_BACKUP_CONTENT *backup_content;
  MYSQL *mysql;
  int options;
  unsigned int current_table_index;
  int table_finished;
  char *current_table_quoted;
  char *current_schema;
  char *current_schema_quoted;
  int current_table_rows_processed;
  int current_table_rows;
  char *extended_insert;
  int starting;
  char quote_char;
  int ignore;
} MYX_BS_STATUS;

/* forward declarations */

static int find_table(MYX_BACKUP_CONTENT *b, const char *table_name, 
                      const char *schema_name, const char *catalog_name);

static void free_backup_table_content(MYX_BACKUP_TABLE *t);

static int interact_with_server(MYX_RBS_STATUS *rs, char *buffer,
                                int buffer_len, MYX_BACKUP_ERROR *error);
static int comfort_query(MYX_RBS_STATUS *rs, char *buffer,
                         int buffer_len, MYX_BACKUP_ERROR *error);
static int write_create_statement_to_file(MYX_BS_STATUS *status,
                                          MYX_BACKUP_ERROR *error);
static int write_row_to_file(MYX_BS_STATUS *status, MYSQL_ROW row,
                             MYX_BACKUP_ERROR *error);
static int write_sql_file_header(MYX_BS_STATUS *status,
                                 MYX_BACKUP_ERROR *error);
static int write_sql_file_footer(MYX_BS_STATUS *status,
                                 MYX_BACKUP_ERROR *error);
static int  compare_backup_tables(const void *a, const void *b);
static int write_path_info_to_file(MYX_BS_STATUS *status,
                                   MYX_BACKUP_ERROR *error);
static MYX_BACKUP_CONTENT* copy_backup_content(MYX_BACKUP_CONTENT *content);
static MYX_BACKUP_CONTENT* select_all_tables_4(MYSQL* mysql, MYX_BACKUP_CONTENT *content);
static MYX_BACKUP_CONTENT* select_all_tables_5(MYSQL* mysql, MYX_BACKUP_CONTENT *content);
static int in_backup_content(const char *schema_name,
                             MYX_BACKUP_CONTENT *backup_content);

static MYX_BACKUP_PROFILE *read_in_backup_profile_1_0(xmlNodePtr backup_node);
static MYX_BACKUP_PROFILE *read_in_backup_profile_1_1(xmlNodePtr backup_node);
static MYX_BACKUP_PROFILE *read_in_backup_profile_1_2(xmlNodePtr backup_node);
static void read_in_table(xmlNodePtr table_node, MYX_BACKUP_TABLE *table);
static void read_in_entity(xmlNodePtr entity_node, MYX_BACKUP_TABLE *table);
static char* check_statement(const char *line,
                             int line_len, pcre *re, pcre_extra *pe);
static int prepare_line_is_create_statement(pcre **re, pcre_extra **pe);
static int prepare_line_is_create_index_statement(pcre **re, pcre_extra **pe);
static int prepare_line_is_create_view_statement(pcre **re, pcre_extra **pe);
static int prepare_line_is_create_proc_statement(pcre **re, pcre_extra **pe);
static int prepare_line_is_create_func_statement(pcre **re, pcre_extra **pe);
static void finalize_line_is_create_statement(pcre *re, pcre_extra *pe);
static int prepare_line_is_db_use_statement(pcre **re, pcre_extra **pe);
static void finalize_line_is_db_use_statement(pcre *re, pcre_extra *pe);

MYX_BCS_STATUS* myx_new_bcs_status(const char *filename,
                                   const char *filename_charset,
                                   MYX_BACKUP_ERROR *error);
int myx_free_bcs_status(MYX_BCS_STATUS *b, int free_backup_content);
int myx_get_backup_content_from_sql_file_incremental(MYX_BCS_STATUS *bc_status,
                                                     int granularity,
                                                     MYX_BACKUP_ERROR *error);

MYX_RBS_STATUS * myx_new_rbs_status(MYSQL *mysql, const char *filename,
                                    const char *filename_charset,
                                    MYX_BACKUP_CONTENT *content,
                                    const char *target_catalog,
                                    const char *target_schema,
                                    int options,
                                    MYX_BACKUP_ERROR *error,
                                    void (*report_warning)
                                      ( const char *msg,void *user_data),
                                    void *user_data);
int myx_free_rbs_status(MYX_RBS_STATUS *rs);
int myx_restore_backup_from_sql_file_incremental(
                        MYX_RBS_STATUS *rs,
                        int (*progress_report)(bigint bytes_read,
                                               bigint bytes_total,
                                               void *user_data),
                        bigint file_size,
                        void *user_data,
                        MYX_BACKUP_ERROR *error);
MYX_BS_STATUS * myx_new_bs_status(MYSQL *mysql, const char *filename,
                                  MYX_BACKUP_CONTENT *content,
                                  MYX_BACKUP_OPTIONS options,
                                  MYX_BACKUP_ERROR *error);
int myx_free_bs_status(MYX_BS_STATUS *b);
int myx_make_backup_to_sql_file_incremental(MYX_BS_STATUS *status,
                                            unsigned int num_rows,
                                            MYX_BACKUP_ERROR *error);

int myx_set_bs_options(MYX_BS_STATUS *status, int options); //TODO: remove that function?

static int check_if_ignore_table(MYSQL *mysql, const char *quoted_schema_name, const char *table_name);

/* taken from Viktor Vagins patch */
#define SAFE_IO(func) \
{func; if (errno == ENOSPC){*error=MYX_BACKUP_OUTPUTDEVICE_FULL;return -1;}}

// Translation of library errors to backup errors.
const MYX_BACKUP_ERROR errorMapping[] = 
{
  MYX_BACKUP_NO_ERROR,            // MYX_NO_ERROR 
  MYX_BACKUP_CANT_OPEN_FILE,      // MYX_ERROR_CANT_OPEN_FILE
  MYX_BACKUP_SERVER_ERROR,        // MYX_ERROR_CANT_CONNECT_TO_INSTANCE
  MYX_BACKUP_XML_PARSE_ERROR,     // MYX_XML_PARSE_ERROR
  MYX_BACKUP_XML_PARSE_ERROR,     // MYX_XML_NO_VALID_DOCUMENT 
  MYX_BACKUP_XML_PARSE_ERROR,     // MYX_XML_EMPTY_DOCUMENT 
  MYX_BACKUP_SQL_ERROR,           // MYX_SQL_ERROR 
  MYX_BACKUP_STOPPED,             // MYX_STOP_EXECUTION 
  MYX_BACKUP_MALLOC_FAILED,       // MYX_ALLOC_CHANGE_ERROR
  MYX_BACKUP_UNKNOWN,             // MYX_OBJECT_NOT_FOUND 
  MYX_BACKUP_CANT_READ_FROM_FILE, // MYX_CANT_READ_FROM_FILE 
  MYX_BACKUP_CHARSET_CONVERSION,  // MYX_CHARSET_CONVERSION_ERROR
  MYX_BACKUP_WRONG_CHARSET,       // MYX_CHARSET_WRONG_CHARSET_SPECIFIED
  MYX_BACKUP_MALLOC_FAILED        // MYX_MEMORY_LIMIT_EXCEEDED
};

/*
 * Public functions
 */

/* Profile functions */

/*
 *----------------------------------------------------------------------
 *
 *
 * SYNOPSIS
 *  profile_directory must include a trailing path delimiter
 * DESCRIPTION
 *
 * RETURN VALUE
 *
 * NOTES
 *----------------------------------------------------------------------
 */
MYX_ADMIN_LIB_ERROR myx_save_profile(const char *profile_name,
                                     const char *profile_directory,
                                     MYX_BACKUP_PROFILE *backup_profile)
{
  xmlDocPtr doc;
  xmlNodePtr root_node, entities_node, entity_node;
  MYX_ADMIN_LIB_ERROR res;
  char *filename;

  if (!profile_name || !profile_directory || !backup_profile)
    return -1;

  doc= xmlNewDoc((xmlChar*)"1.0");

  root_node= doc->children= xmlNewDocRawNode(doc,NULL,(xmlChar*)"backup_profile",NULL);

  xmlNewTextChild(root_node,NULL,(xmlChar*)"version", (xmlChar*)"1.2");
  
  xmlNewTextChild(root_node,NULL,(xmlChar*)"profile_name", (xmlChar*)backup_profile->profile_name);
  xmlNewTextChild(root_node,NULL,(xmlChar*)"last_used", (xmlChar*)backup_profile->last_used);
  NewTextChild_int_content(root_node,NULL, (xmlChar*)"options", backup_profile->options);
  NewTextChild_int_content(root_node,NULL, 
                           (xmlChar*)"backup_type", backup_profile->backup_type);

  entities_node= xmlNewTextChild(root_node, NULL, (xmlChar*)"entities", NULL);

  if (backup_profile->backup_content)
  {
    MYX_BACKUP_CONTENT *bc = backup_profile->backup_content;
    MYX_BACKUP_TABLE *table= bc->tables;
    MYX_BACKUP_TABLE *tables_end= table + bc->tables_num;
    for (; table < tables_end; table++)
    {
      entity_node= xmlNewTextChild(entities_node, NULL, (xmlChar*)"entity", NULL);
      xmlNewTextChild(entity_node, NULL, (xmlChar*)"name", (xmlChar*)table->table);
      xmlNewTextChild(entity_node, NULL, (xmlChar*)"schema", (xmlChar*)table->schema);
      xmlNewTextChild(entity_node, NULL, (xmlChar*)"catalog", (xmlChar*)table->catalog);
      if(table->flags & MYX_BTF_IS_VIEW)
      {
        xmlNewTextChild(entity_node, NULL, (xmlChar*)"entity_type", (xmlChar*)"view");
      }
      else if(table->flags & MYX_BTF_IS_PROCEDURE)
      {
        xmlNewTextChild(entity_node, NULL, (xmlChar*)"entity_type", (xmlChar*)"proc");
      }
      else if(table->flags & MYX_BTF_IS_FUNCTION)
      {
        xmlNewTextChild(entity_node, NULL, (xmlChar*)"entity_type", (xmlChar*)"func");
      }
      else
      {
        xmlNewTextChild(entity_node, NULL, (xmlChar*)"entity_type", (xmlChar*)"table");
      }
    }
  }

  filename= g_build_path("/", profile_directory, profile_name, NULL); // Don't forget the NULL parameter to finish the parameter list!
  res= myx_xmlSaveFile(filename, doc);
  g_free(filename);
  xmlFreeDoc(doc);
  return (int)res == -1 ? -1 : 0;
}

int myx_free_profile(MYX_BACKUP_PROFILE *profile)
{
  if (profile)
  {
    xmlFree(profile->profile_name);
    xmlFree(profile->last_used);
    if (profile->backup_content)
      myx_free_backup_content(profile->backup_content);
    g_free(profile);
  }
  return 0;
}

MYX_BACKUP_PROFILE *myx_load_profile(const char *profile_name,
                                     const char *profile_directory,
                                     MYX_ADMIN_LIB_ERROR *error_code)
{
  MYX_BACKUP_PROFILE *backup_profile= NULL;

  char *filename= g_build_filename(profile_directory, profile_name, NULL);
  if (!file_exists(filename))
  {
    *error_code= MYX_ERROR_CANT_OPEN_FILE;
  }
  else
  {
    xmlDocPtr doc= myx_xmlParseFile(filename);
    if (doc == NULL )
    {
      *error_code= MYX_XML_PARSE_ERROR;
    }
    else
    {
      xmlNodePtr root= xmlDocGetRootElement(doc);
      if (root == NULL)
      {
        *error_code= MYX_XML_EMPTY_DOCUMENT;
      }
      else
      {
        if (xmlStrcmp(root->name, (const xmlChar *) "backup_profile"))
        {
          *error_code= MYX_XML_NO_VALID_DOCUMENT;
        }
        else
        {
          xmlNodePtr version = try_to_get_child(doc, root, "version");
          *error_code= MYX_ADMIN_NO_ERROR;
          if(version)
          {
            char *version_num= NULL;
            try_to_get_string_field(doc, version, "version", &version_num);
            if(strcmp(version_num, "1.1") == 0)
            {
              backup_profile= read_in_backup_profile_1_1(root);
            }
            else if(strcmp(version_num, "1.2") == 0)
            {
              backup_profile= read_in_backup_profile_1_2(root);
            }
            else
            {
              *error_code= MYX_XML_PARSE_ERROR;
              backup_profile= NULL;
            }
            g_free(version_num);
          }
          else
          {
            backup_profile= read_in_backup_profile_1_0(root);
          }
        }
      }
      xmlFreeDoc(doc);
    }
  }
  g_free(filename);

  return backup_profile;
}

MYX_BACKUP_ERROR myx_make_backup_with_profile(MYSQL *mysql,
                                              MYX_BACKUP_PROFILE *profile,
                                              const char *path,
                                              int callback_interval,
                                              int (*progress_report)
                                              (
                                                const char* current_table_name,
                                                int num_tables,
                                                int num_tables_processed,
                                                int num_rows,
                                                int num_rows_processed,
                                                void *user_data
                                              ),
                                              void *user_data)
{
  return myx_make_backup(mysql, path, profile->backup_content,
                         profile->backup_type, profile->options,
                         callback_interval, progress_report, user_data);
}


/* MAKE-BACKUP functions */


/*
 *----------------------------------------------------------------------
 * Starts a backup of the given content. The content can consist of several tables
 * from different schemas and catalogs. display_backup_progress points to a callback
 * function to display the current progress of the backup.
 *
 * SYNOPSIS
 *   mysql : The mysql-connection that will be used
 *   filename: The name of the file that will hold the backup
 *   content: lists all tables that should be backed up
 *   backup_type: at the moment only MYX_BT_SQL_SCRIPT
 *   options: see the description
 *   progress_report: a function that will be called
 *   user_data: a pointer to user-defined data to be passed to the progress_report
 *   callback_interval: determines how many rows must be processed so that the
 *                      callback-function is called)
 * DESCRIPTION
 *  There are lot of options that determine how the actual backup is
 *  done:
 *   MYX_B_NO_CREATES:           no create Table statements will be written
 *   MYX_B_NO_EXTENDED_INSERT:   instead of mysql's faster insert syntax that
 *                                 allows several values in one insert the ansi-compatible
 *                                 normal syntax is used
 *   MYX_B_ADD_DROP_TABLE:       before each create-statement a "DROP TABLE IF 
 *                                EXISTS" is written
 *   MYX_B_COMMENT:              write some default comments at the beginning of the file
 *   MYX_B_DONT_WRITE_FULL_PATH  don't write "create database"/"use db"
 *                                  -statements to the file
 *   MYX_B_LOCK_ALL_TABLES       All tables are locked with FLush read locks
 *                                 before doing anything
 *   MYX_B_SINGLE_TRANSACTION    Let the backup be done in a single 
 *                                 transaction (for innodb),
 *                                 Mutually exclusive with MYX_B_LOCK_ALL_TABLES
 *   MYX_B_DISABLE_KEYS          write  ALTER TABLE %s DISABLE KEYS before the insers; and
 *                                ALTER TABLE %s ENABLE KEY after all inserts; 
 *                                  makes inserting into myisam-tables a lot faster
 *                               
 *   MYX_B_COMPLETE_INSERTS       INSERTS are of the form "insert into 
 *                                   (col1,col2) values (a,b)"
 *                                instead of "insert into values (a,b)" 
 *   MYX_B_ANSI_QUOTES=512        Identifiers are quoted with " instead of mysql's normal `
 *   MYX_B_ADD_LOCKS              Adds a LOCK TABLES statement before an insert-block and 
 *                                an UNLOCK TABLES afterwards
 ** RETURN VALUE
 *   0 on success, non-zero to indicate an error
 * NOTES
 *   If a file with filename already exists, it will be overwritten.
 *
 *----------------------------------------------------------------------
 */
MYX_BACKUP_ERROR myx_make_backup(MYSQL *mysql, const char *filename,
                                 MYX_BACKUP_CONTENT *content,
                                 MYX_BACKUP_TYPE backup_type, int options,
                                 int callback_interval,
                                 int (*progress_report)
                                    (
                                      const char* curr_tbl_name,
                                      int num_tables, 
                                      int num_tables_processed,
                                      int num_rows,
                                      int num_rows_processed,
                                      void *user_data
                                    ),
                                 void *user_data)
{
  if (backup_type == MYX_BT_SQL_SCRIPT)
  {
    MYX_BS_STATUS *bs_status;
    int last_table_index= -1;
    int interval_counter= callback_interval;
    MYX_BACKUP_ERROR error=0;

    if (! (bs_status= myx_new_bs_status(mysql, filename,
                                        content, options, &error)))
    {
      return error;
    }

    /* backup one line at a time */
    while ( myx_make_backup_to_sql_file_incremental(bs_status, 1000, &error) > 0)
    {
      int new_table= (last_table_index != (int)bs_status->current_table_index);

      interval_counter--;
      if (progress_report &&  bs_status->current_table_quoted && (new_table || interval_counter <= 0))
      {
        if (new_table)
          last_table_index= bs_status->current_table_index;
        interval_counter= callback_interval;
        if ( (*progress_report)(bs_status->current_table_quoted,
                                bs_status->backup_content->tables_num,
                                bs_status->current_table_index+1,
                                bs_status->current_table_rows,
                                bs_status->current_table_rows_processed,
                                user_data) )
        {
          myx_free_bs_status(bs_status);
          return -1;
        }
      }
    }
    myx_free_bs_status(bs_status);
    return error;
  }
  return -1; /* unknown backup type */
}

/*
 *----------------------------------------------------------------------
 * Creates the MYX_BS_STATUS, that saves needed information for
 * myx_make_backup_to_sql_file_incremental(MYX_BS_STATUS *status, int num_rows).
 *
 * SYNOPSIS
 *   content: specifies which tables are to be backed up
 *   options: 0 or a combination of options;
 *            if 0 the default-options will be used
 * DESCRIPTION
 *
 * RETURN VALUE
 *
 * NOTES
 *----------------------------------------------------------------------
 */
MYX_BS_STATUS *myx_new_bs_status(MYSQL *mysql, const char *filename,
                                 MYX_BACKUP_CONTENT *content,
                                 MYX_BACKUP_OPTIONS options, 
                                 MYX_BACKUP_ERROR *error)
{
  MYX_BS_STATUS *b;
  FILE *file;
  unsigned int i;
  char *qs;

  if (options & MYX_B_LOCK_ALL_TABLES & MYX_B_SINGLE_TRANSACTION)
  {
    *error= MYX_BACKUP_ILLEGAL_OPTION;
    return NULL;
  }

  if ((options & MYX_B_ANSI_QUOTES) && 
      !mysql_version_is_later_or_equal_than(mysql,4,1))
  {
    *error= MYX_BACKUP_CANNOT_SET_ANSI_QUOTES;
    return NULL;
  }

  if (! (file= myx_fopen(filename, "w")) )
  {
    *error=  MYX_BACKUP_CANT_OPEN_FILE;
    return NULL;
  }
  b= g_malloc0(sizeof(MYX_BS_STATUS) );
  b->sql_file= file;

  b->mysql= mysql;
  if(options & MYX_B_COMPLETE_SCHEMATAS)
  {
    if(mysql_version_is_later_or_equal_than(mysql,5,0)) 
    {
      b->backup_content= select_all_tables_5(mysql, content);
    }
    else
    {
      b->backup_content= select_all_tables_4(mysql, content);
    }
  }
  else
  {
    b->backup_content= copy_backup_content(content);
  }
  b->options= options;
  b->current_table_index= -1;
  b->starting= 1; /* set to display that we have not fetched anything yet */

  if (b->options == 0)
  {
    b->options= MYX_B_ADD_DROP_TABLE|MYX_B_COMMENT|
                MYX_B_DISABLE_KEYS|MYX_B_ADD_LOCKS;
  }

  b->quote_char= (b->options & MYX_B_ANSI_QUOTES) ? '"' : '`';

  for(i= 0; i < b->backup_content->tables_num; i++)
  {
    //b->backup_content->tables[i].flags= content->tables[i].flags;
    qs= quote_identifier_with(b->backup_content->tables[i].schema, b->quote_char);
    
    if((b->backup_content->tables[i].flags != 0) || check_if_ignore_table(mysql, qs, b->backup_content->tables[i].table))
    {
      // If any of the special types flags is set already then we have a view, SP or SF here, which do not require
      // the backup of content.
      b->backup_content->tables[i].flags |= MYX_BTF_IGNORE_CONTENT;
    }
    g_free(qs);
  }

  if (b->options & MYX_B_SORT_TABLES)
  {
    qsort(b->backup_content->tables, b->backup_content->tables_num,
          sizeof(MYX_BACKUP_TABLE), compare_backup_tables);
  }

  return b;
}

/*
 *----------------------------------------------------------------------
 * Overwrites the current options for a backup-operation with the
 * options specified in the argument
 *
 * SYNOPSIS
 *
 * DESCRIPTION
 *  Should be called only right after myx_new_bs_status.
 *
 * RETURN VALUE
 *  If the options argument is negative, then the current options
 *  value of status is returned.
 *  The newly set option are returned otherwise.
 * NOTES
 *----------------------------------------------------------------------
 */
int myx_set_bs_options(MYX_BS_STATUS *status, int options)
{
  if (options >= 0)
    status->options= options;
  return status->options;
}

int myx_free_bs_status(MYX_BS_STATUS *b)
{
  if (b)
  {
    if (b->sql_file)
      fclose(b->sql_file);
    g_free(b->current_table_quoted);
    g_free(b->current_schema_quoted);
    g_free(b->extended_insert);
    myx_free_backup_content(b->backup_content);
    g_free(b);
  }
  return 0;
}

static const char * show_master_status_fields[]=
{
  "File",               // 0
  "Position",           // 1
  "Binlog_Do_DB",       // 2
  "Binlog_Ignore_DB"    // 3
};
static const char ** show_master_status_fields_end=
             show_master_status_fields + sizeof(show_master_status_fields)/sizeof(char*);

/**
 * @brief check if should NOT dump table's contents (e.g. MERGE table or VIEW)
 *
 * @param mysql mysql server connection handle
 * @param quoted_schema_name quoted name of the table to check
 * @param table_name unquoted name of the table to check
 * 
 * @return 1 if table should ignore the table
 */
static int check_if_ignore_table(MYSQL *mysql, const char *quoted_schema_name, const char *table_name)
{
  char buff[1024];
  MYSQL_RES *res;
  MYSQL_ROW row;
  int result= 1;

  sprintf(buff, "show table status from %s like \"%s\"", quoted_schema_name, table_name);
  if (mysql_query(mysql, buff))
  {
    return 0;      /* assume table is ok */
  }
  res = mysql_store_result(mysql);
  if (!(row= mysql_fetch_row(res)))
  {
    return 1;        /* no such table found - don't dump its content */
  }
  if (!(row[1]))
      result= 1;
  else
  {
    if (strcmp(row[1], "MRG_MyISAM") &&
        strcmp(row[1], "MRG_ISAM")) {
      result= 0;
    }
  }
  mysql_free_result(res);
  return result;
}

/*
 *----------------------------------------------------------------------
 * backs up num-rows from the current table
 *
 * SYNOPSIS
 *
 * DESCRIPTION
 *
 * RETURN VALUE
 *  a value greater 0 if we should continue
 *  0 or negative in case an error has happened or if we are finished
 * NOTES
 *----------------------------------------------------------------------
 */
int myx_make_backup_to_sql_file_incremental(MYX_BS_STATUS *status,
                                            unsigned int num_rows,
                                            MYX_BACKUP_ERROR *error)
{
  MYSQL_ROW row= NULL;

  /* if we don't have any more tables to back up.. */
  if ( status->table_finished && 
       status->current_table_index+1 >= status->backup_content->tables_num)
  {
    return write_sql_file_footer(status, error) ? -1 : 0;
  }

  /* if we start fetching rows of a new table */
  if (status->table_finished || status->starting)
  {
    gchar *sel_stmt;
    MYSQL_RES *result;
    char *current_table, *current_schema;

    if (status->starting)
    {
      gchar sql_mode_cmd[200] = "/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, sql_mode='";
      int mode_entry_count = 0;
      if (status->options & MYX_B_ANSI_QUOTES)
      {
        mode_entry_count++;
        g_sprintf(sql_mode_cmd, "%sANSI_QUOTES", sql_mode_cmd);
      };
      if (status->options & MYX_B_COMPATIBILITY_MODE)
      {
        g_sprintf(sql_mode_cmd, "%s%sMYSQL323", sql_mode_cmd, (mode_entry_count > 0) ? "," : "");
        mode_entry_count++;
      };
      g_sprintf(sql_mode_cmd, "%s' */", sql_mode_cmd);

      if (write_sql_file_header(status, error))
        return -1;

      // Backup method
      status->starting=0;
      if (status->options & MYX_B_LOCK_ALL_TABLES)
      {
        if (myx_mysql_query(status->mysql, "FLUSH TABLES WITH READ LOCK"))
        {
          *error= MYX_BACKUP_CANNOT_FLUSH_TABLES_WITH_READ_LOCK;
          return -1; /* fatal */
        }
      }
      else if (status->options & MYX_B_SINGLE_TRANSACTION)
      {
        // make sure the correct ISOLATION LEVEL is set
        if (mysql_full_version_is_later_or_equal_than(status->mysql, 4, 0, 5))
        {
          if (myx_mysql_query(status->mysql, "SET SESSION TRANSACTION ISOLATION LEVEL REPEATABLE READ"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }

        // start the transaction
        if (mysql_full_version_is_later_or_equal_than(status->mysql, 4, 1, 8))
        {
          if (myx_mysql_query(status->mysql, "START TRANSACTION WITH CONSISTENT SNAPSHOT"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }
        else
        {
          if (myx_mysql_query(status->mysql, "BEGIN"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }
      }
      else if (status->options & MYX_B_POINT_IN_TIME_BACKUP)
      {
        if (myx_mysql_query(status->mysql, "FLUSH TABLES"))
        {
          *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
          return -1;  /* fatal */
        }
        if (myx_mysql_query(status->mysql, "FLUSH TABLES WITH READ LOCK"))
        {
          *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
          return -1;  /* fatal */
        }
        // make sure the correct ISOLATION LEVEL is set
        if (mysql_full_version_is_later_or_equal_than(status->mysql, 4, 0, 5))
        {
          if (myx_mysql_query(status->mysql, "SET SESSION TRANSACTION ISOLATION LEVEL REPEATABLE READ"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }
        if (mysql_full_version_is_later_or_equal_than(status->mysql, 4, 1, 8))
        {
          if (myx_mysql_query(status->mysql, "START TRANSACTION WITH CONSISTENT SNAPSHOT"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }
        else
        {
          if (myx_mysql_query(status->mysql, "BEGIN"))
          {
            *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
            return -1;  /* fatal */
          }
        }
        if (myx_mysql_query(status->mysql, "SHOW MASTER STATUS"))
        {
          *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
          return -1;  /* fatal */
        }
        else
        {
          MYSQL_RES *res;

          if((res= mysql_store_result(status->mysql)))
          {
            MYSQL_ROW row;
            MYSQL_FIELD *fields;
            int num_fields;
            int fi[5];
            char *binlog_filename= NULL;
            char *binlog_pos= NULL;

            
            //Depending on the version of the server there might be different columns
            num_fields= mysql_num_fields(res);
            fields= mysql_fetch_fields(res);

            build_field_subst(show_master_status_fields, show_master_status_fields_end,
              fields, fields + num_fields, fi);

            if((row= mysql_fetch_row(res)))
            {
              if (fi[0] > -1)
                binlog_filename= myx_convert_dbstr_utf8(status->mysql, row[fi[0]]);
              if (fi[1] > -1)
                binlog_pos= g_strdup(row[fi[1]]);
            }

            if (binlog_filename && binlog_pos)
            {
              SAFE_IO(fprintf(status->sql_file,
                "--\n"
                "-- Position to start replication or point-in-time recovery from\n"
                "--\n\n"
                "-- CHANGE MASTER TO MASTER_LOG_FILE='%s', MASTER_LOG_POS=%s;\n\n",
                binlog_filename, binlog_pos));
            }

            mysql_free_result(res);
          }
        }

        if (myx_mysql_query(status->mysql, "UNLOCK TABLES"))
        {
          *error= MYX_BACKUP_CANNOT_START_TRANSACTION;
          return -1;  /* fatal */
        }
      }


      if (myx_mysql_query(status->mysql,sql_mode_cmd))
      {
        *error= MYX_BACKUP_CANNOT_SET_ANSI_QUOTES;
        return -1;
      }
    }

    status->table_finished= 0;
    status->current_table_index++;
    current_table=
             status->backup_content->tables[status->current_table_index].table;
    current_schema=
            status->backup_content->tables[status->current_table_index].schema;
    status->ignore= status->backup_content->tables[status->current_table_index].flags & MYX_BTF_IGNORE_CONTENT;
    g_free(status->current_table_quoted);
    if(current_table)
    {
      status->current_table_quoted= quote_identifier_with(current_table,
                                                          status->quote_char);
    }
    else
    {
      status->current_table_quoted= NULL;
    }
    g_free(status->current_schema);
    g_free(status->current_schema_quoted);
    status->current_schema_quoted= quote_identifier_with(current_schema,
                                                         status->quote_char);
    status->current_schema= g_strdup(current_schema);
    status->current_table_rows_processed= 0;

    if((current_table) && !status->ignore)
    {
      /* find out how many rows this table has.. */
      sel_stmt= g_strconcat("SELECT count(*) FROM ",
                            status->current_schema_quoted,
                            ".", status->current_table_quoted, NULL);
      if ( myx_mysql_query(status->mysql, sel_stmt) ||
          !(result= mysql_store_result(status->mysql)) )
      {
        g_free(sel_stmt);
        *error= MYX_BACKUP_SERVER_ERROR;
        return 0;
      }
      g_free(sel_stmt);

      row= mysql_fetch_row(result);
      if (row)
      {
        status->current_table_rows= atoi(row[0]);
      }
      else
      {
        mysql_free_result(result);
        *error= MYX_BACKUP_SERVER_ERROR;
        return 0;
      }
      mysql_free_result(result);
    }
    else
    {
      status->current_table_rows= 0;
      status->current_table_rows_processed= 0;
    }

    /* write the path-information for this table */
    if (!(status->options & MYX_B_DONT_WRITE_FULL_PATH) &&
        write_path_info_to_file(status, error))
    {
      return -1;
    }

    if(current_table)
    {
      /* get the create-statement of the new table */
      //if (!(status->options & MYX_B_NO_CREATES) &&
      //    write_create_statement_to_file(status, error))
      //{
      //  return -1;
      //}
      
      // if writing table create stmt failed, then ignore this table
      // (bug #12577)
      if (!(status->options & MYX_B_NO_CREATES))
      {
        status->ignore = write_create_statement_to_file(status, error) || status->ignore;
        *error= 0;
      }

      if(!status->ignore)
      {

        if (status->options & MYX_B_COMMENT)
        {
          SAFE_IO(fprintf(status->sql_file, "\n--\n"
            "-- Dumping data for table %s.%s\n"
            "--\n", 
            status->current_schema_quoted,
            status->current_table_quoted));
        }

        /* get the rows of the new table */
        sel_stmt= g_strconcat("SELECT /*!40001 SQL_NO_CACHE */ * FROM ",
                              status->current_schema_quoted, ".",
                              status->current_table_quoted, NULL);
        if (myx_mysql_query(status->mysql, sel_stmt) ||
            !(status->mysql_result= mysql_use_result(status->mysql)) )
        {
          g_free(sel_stmt);
          *error= MYX_BACKUP_SERVER_ERROR;
          return 0;
        }
        g_free(sel_stmt);

        if ((status->options &  MYX_B_DISABLE_KEYS) && !status->ignore)
        {
          SAFE_IO(fprintf(status->sql_file,
                          "\n/*!40000 ALTER TABLE %s DISABLE KEYS */;\n",
                          status->current_table_quoted));
        }

        if (status->options & MYX_B_ADD_LOCKS)
        {
          SAFE_IO(fprintf(status->sql_file,"LOCK TABLES %s WRITE;\n",
                  status->current_table_quoted));
        }
      }
      else // if(!status->ignore)
      {
        SAFE_IO(fprintf(status->sql_file, "\n"));
      }
    }
  }

  if(!status->ignore && status->current_table_quoted)
  {
    /* we will fetch num_rows rows every time 
    (but less if the table has no more rows) */
    for (;num_rows && (row= mysql_fetch_row(status->mysql_result));
        num_rows--, status->current_table_rows_processed++)
    {
      //if(!status->ignore)
      //{
        if (write_row_to_file(status, row, error))
          return -1;
      //}
    }
  } 
  else
  {
    status->current_table_rows_processed= status->current_table_rows;
    row= 0;
  }

  /* if backing up this table has ended */
  if (status->current_table_rows_processed == status->current_table_rows)
  {
    /* just to be sure because the documentation says that you have to
                                      call fetch_row until it returns NULL.. */
    if (row)
      row= mysql_fetch_row(status->mysql_result);
    assert(row == NULL);

    status->table_finished= 1;
    /* free the result of the last select * query */
    mysql_free_result(status->mysql_result);
    status->mysql_result= NULL;

    /* it is possible that our extended insert needs to be written out */
    if (!(status->options&MYX_B_NO_EXTENDED_INSERT) && status->extended_insert)
    {
      char *tmp_str;
      char * pos= status->extended_insert + strlen(status->extended_insert) - 1;
      for ( ; *pos!=','; pos--);
      *pos= ';';  /*overwrite the comma with ;*/
      pos++;
      *pos= 0;

      tmp_str= myx_convert_dbstr_utf8(status->mysql, status->extended_insert);
      g_free(status->extended_insert);
      status->extended_insert= NULL;

      SAFE_IO(fprintf(status->sql_file, "%s\n", tmp_str));
      g_free(tmp_str);
    }
    
    if(status->current_table_quoted)
    {
      if (status->options & MYX_B_ADD_LOCKS)
        SAFE_IO(fprintf(status->sql_file,"UNLOCK TABLES;\n"));

      if ((status->options &  MYX_B_DISABLE_KEYS) && !status->ignore)
      {
        SAFE_IO(fprintf(status->sql_file,
                        "/*!40000 ALTER TABLE %s ENABLE KEYS */;\n\n",
                        status->current_table_quoted));
      }

      /* if this is the last table */
      if (status->current_table_index+1 >= status->backup_content->tables_num)
      {
        if (status->options & MYX_B_LOCK_ALL_TABLES)
        {
          if (myx_mysql_query(status->mysql, "UNLOCK TABLES"))
          {
            *error=  MYX_BACKUP_SERVER_ERROR;
            return -1;
          }
        }
        else if ((
          (status->options & MYX_B_SINGLE_TRANSACTION) ||
          (status->options & MYX_B_POINT_IN_TIME_BACKUP)) &&
          myx_mysql_query(status->mysql, "COMMIT"))
        {
          *error= MYX_BACKUP_SERVER_ERROR;
          return -1;  /* fatal */
        }

        if (myx_mysql_query(status->mysql, "/*!40101 SET sql_mode=@OLD_SQL_MODE */"))
        {
          *error= MYX_BACKUP_SERVER_ERROR;
          return -1;
        }
      }
    }
  }

  return 1;
}

/* GET-BACKUP-CONTENT Functions */

/*
 *----------------------------------------------------------------------
 * Returns all catalogs/schemas/tables stored in the backup
 *
 * SYNOPSIS
 *   default_catalog_name: If no catalog is given for a table in
 *                  the given file, it is assumed to be in the
 *                  default_catalog_name
 *   default_schema_name: if a table in the given file has no
 *                  schema(=database) then it is assumed that
 *                  the table lies in the default_schema_name
 *   report_interval: Specifies the number of bytes that have to
 *                    be read before progress report is called (again).
 *   progress_report: call-back function
 *                    returns 0 if we should continue our work
 *                    progress_report may be NULL
 *   user_data:     user specified data to be passed back when
 *                  progress_report is called
 * DESCRIPTION
 *
 * RETURN VALUE
 *
 * NOTES
 *
 *----------------------------------------------------------------------
 */
MYX_BACKUP_CONTENT *
myx_get_backup_content(const char *filename,
                       const char *filename_charset,
                       MYX_BACKUP_TYPE backup_type,
                       int report_interval,
                       int (*progress_report) (bigint bytes_read,
                                               bigint bytes_total,
                                               void *user_data),
                       void *user_data, MYX_BACKUP_ERROR *error )
{
  MYX_BACKUP_CONTENT *content;
  if (backup_type == MYX_BT_SQL_SCRIPT)
  {
    MYX_BCS_STATUS *stat;
    bigint next_call, file_size;

    *error= 0;

    if (! (stat= myx_new_bcs_status(filename, filename_charset, error)) )
      return NULL;

    file_size= get_file_size(filename);
    next_call= report_interval;

    while (myx_get_backup_content_from_sql_file_incremental(stat,1,error) > 0 )
    {
      if ( (progress_report) && (stat->we.bytes_read >= next_call) )
      {
        next_call+= report_interval;
        if ((*progress_report)(stat->we.bytes_read, file_size, user_data))
        {
          myx_free_bcs_status(stat,1);
          return NULL;
        }
      }
    }

    if (*error)
    {
      myx_free_bcs_status(stat, 1);
      return NULL;
    }

    if (progress_report && (file_size % report_interval))
      (*progress_report)(stat->we.bytes_read, file_size, user_data);

    content= stat->backup_content;
    myx_free_bcs_status(stat,0);
    return content;
  }
  return NULL;
}

int myx_free_backup_content(MYX_BACKUP_CONTENT *backup)
{
  if (backup)
  {
    MYX_BACKUP_TABLE * table= backup->tables;
    MYX_BACKUP_TABLE * tables_end= table + backup->tables_num;
    for (; table!=tables_end; table++)
      free_backup_table_content(table);
    g_free(backup->tables);
    g_free(backup);
  }
  return 0;
}

MYX_BCS_STATUS* myx_new_bcs_status(const char *filename,
                                   const char *filename_charset,
                                   MYX_BACKUP_ERROR *error)
{
  MYX_BCS_STATUS * b;
  MYX_INTL_FILE * intl_file= myx_new_intl_file(filename,
                                               filename_charset, error);
  if (!intl_file)
    return NULL;

  b= g_malloc0( sizeof(MYX_BCS_STATUS) );
  b->intl_file= intl_file;

  b->backup_content= g_malloc0(sizeof(MYX_BACKUP_CONTENT) );
  myx_init_sql_parse_environment(&b->we);

  prepare_line_is_create_statement(&b->re_create, &b->pe_create);
  prepare_line_is_create_index_statement(&b->re_create_index, &b->pe_create_index); 
  prepare_line_is_create_view_statement(&b->re_create_view, &b->pe_create_view);
  prepare_line_is_create_proc_statement(&b->re_create_proc, &b->pe_create_proc);
  prepare_line_is_create_func_statement(&b->re_create_func, &b->pe_create_func);
  prepare_line_is_db_use_statement(&b->re_db_use, &b->pe_db_use);

  b->default_schema_name= DEFAULT_SCHEMA_NAME;
  b->default_catalog_name= DEFAULT_CATALOG_NAME;

  return b;
}

int myx_free_bcs_status(MYX_BCS_STATUS *b, int free_backup_content)
{
  if (b)
  {
    myx_free_intl_file(b->intl_file);
    myx_done_sql_parse_environment(&b->we);
    g_free(b->cur_db_name);
    if (free_backup_content)
      g_free(b->backup_content);
    finalize_line_is_db_use_statement(b->re_db_use, b->pe_db_use);
    finalize_line_is_create_statement(b->re_create, b->pe_create);
    g_free(b);
  }
  return 0;
}

/*
 *----------------------------------------------------------------------
 *
 *
 * SYNOPSIS
 *
 * DESCRIPTION
 *  granularity = number of sql-statements
 * RETURN VALUE
 *  the number of successfully processed sql-statements
 *  0 if there are no more sql-statements
 *  a negative number in case of an error
 * NOTES
 *----------------------------------------------------------------------
 */
int myx_get_backup_content_from_sql_file_incremental(MYX_BCS_STATUS *bc_status,
                                                     int granularity,
                                                     MYX_BACKUP_ERROR *error)
{
  int len, tables_found;
  char buffer[BACKUP_CONTENT_BUFFER_LEN];
  char *bufptr= (char*)buffer;
  int buffer_len= sizeof(buffer);
  int count= 0;
  int flags= 0;
  MYX_BACKUP_TABLE *bt;
  char *tmp;
  MYX_LIB_ERROR libraryError = MYX_NO_ERROR;

  while (count < granularity &&
         (len= myx_get_next_sql_statement_file(&bc_status->we,
                                               bc_status->intl_file,
                                               &bufptr, &buffer_len,
                                               0, 0, 0, 0, &libraryError)) )
  {    
    if (libraryError != MYX_NO_ERROR)
      break;

    /* do some checks now */
    if ( (tmp= check_statement(buffer,len, /* CREATE TABLE */
                               bc_status->re_create,bc_status->pe_create)) )
    {
      flags= 0;
    }
    else if ( (tmp= check_statement(buffer,len, /* CREATE VIEW */
                               bc_status->re_create_view,bc_status->pe_create_view)) )
    {
      flags= MYX_BTF_IS_VIEW;
    }
    else if ( (tmp= check_statement(buffer,len, /* CREATE PROCEDURE */
                               bc_status->re_create_proc,bc_status->pe_create_proc)) )
    {
      flags= MYX_BTF_IS_PROCEDURE;
    }
    else if ( (tmp= check_statement(buffer,len, /* CREATE FUNCTION */
                               bc_status->re_create_func,bc_status->pe_create_func)) )
    {
      flags= MYX_BTF_IS_FUNCTION;
    }

    if(tmp)
    {
      tables_found= ++(bc_status->backup_content->tables_num);

      bc_status->backup_content->tables= 
        g_realloc(bc_status->backup_content->tables, 
                  tables_found * sizeof(MYX_BACKUP_TABLE));
      bt= bc_status->backup_content->tables + tables_found-1;

      bt->catalog= g_strdup(bc_status->default_catalog_name);
      bt->schema= g_strdup(bc_status->cur_db_name ? 
                           bc_status->cur_db_name : 
                           bc_status->default_schema_name);
      bt->table= unquote_identifier(tmp);
      bt->flags= flags;
    }
    else if ( (tmp= check_statement(buffer,len,bc_status->re_db_use,
                                    bc_status->pe_db_use)) ) /* USE */
    {
      g_free(bc_status->cur_db_name);
      bc_status->cur_db_name= unquote_identifier(tmp);
    }
    count++;
  }

  *error = errorMapping[libraryError];
    
  return count;
}

/* RESTORE-BACKUP Functions */

void store_old_cs_set_utf8(MYSQL * mysql)
{
  myx_mysql_query(mysql,
                  "SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT");
  myx_mysql_query(mysql,
                  "SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS");
  myx_mysql_query(mysql,
                  "SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION");
  myx_mysql_query(mysql,"SET NAMES utf8");
}

void restore_old_cs(MYSQL * mysql)
{
  myx_mysql_query(mysql,
                  "SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT");
  myx_mysql_query(mysql,
                  "SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS");
  myx_mysql_query(mysql,
                  "SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION");
}

/*
 *----------------------------------------------------------------------
 * Restores a backup with the given filename into the database that
 * is reached by the mysql-struct, but only the catalogs/schemas/tables
 * specified by content. The backup can be restored in a different catalog and
 * or schema.
 *
 * SYNOPSIS
 *   target_catalog: the catalog that the table should be restored in;
 *                   if NULL or '' the catalog in content will be used
 *   target_schema:  the schema that the table should be restored in;
 *                   if NULL or '' the schema in content will be used
 *   content:        specifies with tables (identified by tablename,schema-
 *                   name and catalogname) that are defined in filename
 *                   are to be restored
 *   progress_report,
 *   report_interval: See myx_get_backup_content
 *   report_warning: this function will be called to report a not-fatal
 *                   error. If NULL the warning is printed to stderr
 * DESCRIPTION
 *  Possible option values:
 *    -) MYX_RBS_FORCE:                 Continue even in case of an sql-error
 *    -) MYX_RBS_DONT_CREATE_TARGETS:   Don't create the target schema and/or
 *                                       target catalog if it doesn't exist
 * RETURN VALUE
 *  0 on success, errorcode otherwise
 * NOTES
 *----------------------------------------------------------------------
 */
MYX_BACKUP_ERROR myx_restore_backup(MYSQL *mysql, const char *filename, 
                                    const char *filename_charset, 
                                    MYX_BACKUP_CONTENT *content, 
                                    const char *target_catalog, 
                                    const char *target_schema,
                                    MYX_BACKUP_TYPE backup_type, 
                                    int options,
                                    int report_interval,
                                    int (*progress_report)(bigint bytes_read,
                                                           bigint bytes_total,
                                                           void *user_data),
                                    void *ruser_data,
                                    void (*report_warning)(const char *msg,
                                                           void *user_data),
                                    void *wuser_data)
{
  MYX_BACKUP_ERROR error= 0;

  if (backup_type == MYX_BT_SQL_SCRIPT)
  {
    MYX_RBS_STATUS *rs;
    bigint next_call;
    bigint file_size;

    file_size= get_file_size(filename);
    next_call= report_interval;

    if ( !(rs= myx_new_rbs_status(mysql, filename, filename_charset, content,
                                  target_catalog, target_schema, options,
                                  &error, report_warning, wuser_data)) )
    {
      return error;
    }

    if (mysql_full_version_is_later_or_equal_than(mysql, 4, 1, 1))
      store_old_cs_set_utf8(mysql);

    while (myx_restore_backup_from_sql_file_incremental(rs, progress_report, file_size, ruser_data, &error) > 0)
    {
/*
      if ((progress_report) && (rs->we.bytes_read >= next_call))
      {
        next_call+= report_interval;
        if ((*progress_report)(rs->we.bytes_read, file_size, ruser_data))
        {
          myx_free_rbs_status(rs);
          if (mysql_full_version_is_later_or_equal_than(mysql, 4, 1, 1))
            restore_old_cs(mysql);
          return 0;
        }
      }
*/
    }
    
    if(error)
    {
      myx_set_mysql_error(g_strdup(mysql_error(mysql)), mysql_errno(mysql));
    }

    if (mysql_full_version_is_later_or_equal_than(mysql, 4, 1, 1))
      restore_old_cs(mysql);

    if ((file_size % report_interval) && (progress_report))
      (*progress_report)(rs->we.bytes_read, file_size, ruser_data);

    myx_free_rbs_status(rs);

    return error;
  }

  return 1;
}

MYX_RBS_STATUS *myx_new_rbs_status(MYSQL *mysql, const char *filename,
                                   const char *filename_charset,
                                   MYX_BACKUP_CONTENT *content,
                                   const char *target_catalog,
                                   const char *target_schema,
                                   int options, MYX_BACKUP_ERROR *error,
                                   void(*report_warning)(const char *msg,
                                                         void *user_data),
                                   void *user_data )
{
  MYX_RBS_STATUS *rs;
  const char *error_str;
  int erroffset;

  rs= g_malloc0( sizeof(MYX_RBS_STATUS) );

  if (! (rs->intl_file= myx_new_intl_file(filename, filename_charset, error)) )
  {
    g_free(rs);
    return NULL;
  }

  if (! (rs->buffer= g_try_malloc(INITIAL_BUFFER_LENGTH)) )
  {
    *error= MYX_BACKUP_MALLOC_FAILED;
    g_free(rs);
    return NULL;
  }
  rs->buffer_len= INITIAL_BUFFER_LENGTH;

  myx_init_sql_parse_environment(&rs->we);

  prepare_line_is_create_statement(&rs->re_create, NULL);
  prepare_line_is_create_index_statement(&rs->re_create_index, NULL);
  prepare_line_is_create_view_statement(&rs->re_create_view, NULL);
  prepare_line_is_create_proc_statement(&rs->re_create_proc, NULL);
  prepare_line_is_create_func_statement(&rs->re_create_func, NULL);
  prepare_line_is_db_use_statement(&rs->re_db_use, NULL);

  rs->mysql= mysql;

  /* In Delphi there are no NULL-Strings- there is
   * only a string with a length of 0
   * => treat '' the same as NULL strings */
  rs->target_catalog= !target_catalog || !(*target_catalog)
                      ? NULL : target_catalog;
  rs->target_schema= !target_schema || !(*target_schema)
                      ? NULL : target_schema;

  rs->backup_content= content;
  rs->current_schema_in_sqlfile= g_strdup(DEFAULT_SCHEMA_NAME);
  rs->current_catalog_in_sqlfile= g_strdup(DEFAULT_CATALOG_NAME);
  rs->current_schema_in_db= g_strdup(" ");
  rs->options= options;
  rs->report_warning= report_warning;
  rs->report_warning_data= user_data;

  /* setup the regular expressions */
  /* drop table may only contain one table at the moment */
  rs->re_drop= pcre_compile
                          ("drop\\s+table\\s+(?:if\\s+exists\\s+)?"TABLE_NAME_PCRE"\\s*;",
                           PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                           &error_str, &erroffset, NULL);
  rs->re_drop_view= pcre_compile
                          ("drop\\s+view\\s+(?:if\\s+exists\\s+)?"TABLE_NAME_PCRE"\\s*;",
                           PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                           &error_str, &erroffset, NULL);
  rs->re_drop_proc= pcre_compile
                          ("drop\\s+procedure\\s+(?:if\\s+exists\\s+)?"TABLE_NAME_PCRE"\\s*;",
                           PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                           &error_str, &erroffset, NULL);
  rs->re_drop_func= pcre_compile
                          ("drop\\s+function\\s+(?:if\\s+exists\\s+)?"TABLE_NAME_PCRE"\\s*;",
                           PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                           &error_str, &erroffset, NULL);
  rs->re_insert= pcre_compile("insert\\s+into\\s+"TABLE_NAME_PCRE,
                              PCRE_ANCHORED|PCRE_CASELESS,
                              &error_str, &erroffset, NULL);
  rs->re_alter= pcre_compile("(?:\\/\\*\\!\\d+\\s+)?alter\\s+table\\s+"TABLE_NAME_PCRE,
                             PCRE_ANCHORED|PCRE_CASELESS, &error_str,
                             &erroffset, NULL);
  rs->re_lock= pcre_compile("lock\\s+tables\\s+"TABLE_NAME_PCRE"\\s+write\\s*;",
                            PCRE_ANCHORED|PCRE_CASELESS, &error_str,
                            &erroffset, NULL);
  rs->re_unlock= pcre_compile("unlock\\s+(tables)",
                              PCRE_ANCHORED|PCRE_CASELESS,
                              &error_str, &erroffset, NULL);

  rs->re_charset_set1= pcre_compile(".*(SET).*CHARACTER_SET_CLIENT.*",
                                    PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                    &error_str, &erroffset, NULL);
  rs->re_charset_set2= pcre_compile(".*(SET).*NAMES.*",
                                    PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                    &error_str, &erroffset, NULL);
  rs->re_charset_set3= pcre_compile(".*(SET).*CHARACTER_SET_RESULTS.*",
                                    PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                    &error_str, &erroffset, NULL);
  rs->re_charset_set4= pcre_compile(".*(SET).*COLLATION_CONNECTION.*",
                                    PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                    &error_str, &erroffset, NULL);
  rs->re_charset_set5= pcre_compile(".*(SET).*CHARACTER_SET_CONNECTION.*",
                                    PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                    &error_str, &erroffset, NULL);

  rs->re_set= pcre_compile("(?:\\/\\*\\!\\d+\\s+)?(SET)",
                           PCRE_ANCHORED|PCRE_CASELESS,
                           &error_str, &erroffset, NULL);
  rs->re_create_database= pcre_compile("create\\s+database\\s+(?:\\/\\*\\!\\d+"
                                       "\\s*if\\s+not\\s+exists\\*\\/\\s+)?"
                                       "(\\S+).*;",
                                       PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                       &error_str, &erroffset, NULL);

  rs->re_delimiter= pcre_compile("delimiter\\s+(.*)",
                                       PCRE_ANCHORED|PCRE_CASELESS|PCRE_DOTALL,
                                       &error_str, &erroffset, NULL);

  if (!rs->re_drop || !rs->re_insert || !rs->re_alter ||
      !rs->re_lock || !rs->re_unlock || 
      !rs->re_charset_set1 || !rs->re_charset_set2 ||
      !rs->re_charset_set3 || !rs->re_charset_set4 || !rs->re_charset_set5 ||
      !rs->re_set || !rs->re_create_database || !rs->re_delimiter)
  {
    myx_done_sql_parse_environment(&rs->we);
    myx_free_intl_file(rs->intl_file);
    finalize_line_is_create_statement(rs->re_create, NULL);
    finalize_line_is_db_use_statement(rs->re_db_use, NULL);
    g_free(rs);
    *error= MYX_BACKUP_PCRE_ERROR;
    return NULL;
  }

  return rs;
}

static void ensure_charset_present(char **pbuf, int *plen, const char *defcharset)
{
  int vector[6];
  const char *error;
  int erroffset;
  char *buf, *c;
  size_t cslen;

  pcre *re= pcre_compile(".*(.*).*DEFAULT[\\s]+CHARSET=[a-zA-Z0-9]+.*", /* the pattern */
                        PCRE_CASELESS|PCRE_MULTILINE,                   /* default options */
                        &error,                                         /* for error message */
                        &erroffset,                                     /* for error offset */
                        NULL);                                          /* use default character tables */
  if(pcre_exec(re, NULL, *pbuf, *plen, 0, 0, vector, 6) > 0)
  {
    // DEFAULT CHARSET is present in the statement
    return;
  }
  cslen= strlen (defcharset);
  buf= g_malloc((gulong)(*plen + cslen + 32)); // 128 fro "DEFAULT CHARSET="
  strcpy(buf, *pbuf);
  for(c= buf + *plen - 1; (*c == ' ') && (c > buf); c--);
  strcpy(c, " DEFAULT CHARSET=");
  c += sizeof(" DEFAULT CHARSET=")-1;
  strcpy(c, defcharset);
  c += cslen;
  strcpy(c, ";");
  g_free(*pbuf);
  *pbuf= buf;
  *plen= (int)strlen(buf);
}

int myx_free_rbs_status(MYX_RBS_STATUS *rs)
{
  if (rs)
  {
    myx_free_intl_file(rs->intl_file);
    myx_done_sql_parse_environment(&rs->we);
    finalize_line_is_create_statement(rs->re_create, NULL);
    finalize_line_is_db_use_statement(rs->re_db_use, NULL);
    g_free(rs->current_schema_in_sqlfile);
    g_free(rs->current_catalog_in_sqlfile);

    pcre_free(rs->re_drop);
    pcre_free(rs->re_insert);
    pcre_free(rs->re_alter);
    pcre_free(rs->re_lock);
    pcre_free(rs->re_unlock);
    pcre_free(rs->re_charset_set1);
    pcre_free(rs->re_charset_set2);
    pcre_free(rs->re_charset_set3);
    pcre_free(rs->re_charset_set4);
    pcre_free(rs->re_charset_set5);
    pcre_free(rs->re_set);
    pcre_free(rs->re_create_database);

    g_free(rs->buffer);
    g_free(rs);
  }
  return 0;
}

/* returns the number of successfully processed sql-statements
 * 0 if there is no more sql-statement
 * a negative number in case of an error
 */
int myx_restore_backup_from_sql_file_incremental(
                  MYX_RBS_STATUS *rs,
                  int (*progress_report)(bigint bytes_read,
                                         bigint bytes_total,
                                         void *user_data),
                  bigint file_size,
                  void *user_data,
                  MYX_BACKUP_ERROR *error)
{
  int len;
  int count = 0, found;
  char *tmp, *create;
  char *warning;
  MYX_LIB_ERROR libraryError = MYX_NO_ERROR;

  /* We will only execute statements we know. Otherwise we could do something
   * bad without knowing it.
   */
  for (count= 0;
       count < 1 &&
       (len= myx_get_next_sql_statement_file(&rs->we,rs->intl_file,
                                             &rs->buffer,
                                             &rs->buffer_len,1,
                                             progress_report,
                                             file_size,
                                             user_data,
                                             &libraryError)) > 0;
       count++)
  {
    if (libraryError != MYX_NO_ERROR)
    {
      *error = errorMapping[libraryError];
      return -1;
    };

    assert(rs->buffer);
    create= NULL;

    /* do some checks now */
    if ((tmp= check_statement(rs->buffer,len, rs->re_insert, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_drop,   NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_drop_view, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_drop_proc, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_drop_func, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_alter,  NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_lock,   NULL)) ||
        (tmp= create= check_statement(rs->buffer,len, rs->re_create, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_create_view, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_create_proc, NULL)) ||
        (tmp= check_statement(rs->buffer,len, rs->re_create_func, NULL)))
    {
      if(create && mysql_version_is_later_or_equal_than(rs->mysql, 4, 1))
      {
        ensure_charset_present(&rs->buffer, &len, rs->intl_file->charset);
        rs->buffer_len= len;
      }
      tmp= unquote_identifier(tmp);
      found= find_table(rs->backup_content, tmp,
                        rs->current_schema_in_sqlfile,
                        rs->current_catalog_in_sqlfile);
      g_free(tmp);
      if (found && interact_with_server(rs, rs->buffer, len, error))
        return -1;
    }
    else if(tmp= check_statement(rs->buffer,len, rs->re_create_index, NULL))
    {
      g_free(tmp);
      if (interact_with_server(rs, rs->buffer, len, error)) 
        return -1;
    }
    else if ( (tmp= check_statement(rs->buffer,len,rs->re_unlock, NULL)) )
    {
      g_free(tmp);
      if (interact_with_server(rs, rs->buffer, len, error)) 
        return -1;
    }
    else if ((tmp= check_statement(rs->buffer,len,rs->re_charset_set1,NULL)) ||
             (tmp= check_statement(rs->buffer,len,rs->re_charset_set2,NULL)) ||
             (tmp= check_statement(rs->buffer,len,rs->re_charset_set3,NULL)) ||
             (tmp= check_statement(rs->buffer,len,rs->re_charset_set4,NULL)) ||
             (tmp= check_statement(rs->buffer,len,rs->re_charset_set5,NULL)))
    {              /* SET characterset */
      g_free(tmp); /*    don't send it to the server! */
    }
    else if ((tmp= check_statement(rs->buffer,len,rs->re_set, NULL))) /* SET */
    {
      g_free(tmp);
      if (comfort_query(rs, rs->buffer, len, error))
        return -1;
    }
    else if ((tmp= check_statement(rs->buffer,len,rs->re_db_use, NULL)))
    {                                                                 /* USE */
      tmp= unquote_identifier(tmp);
      g_free(rs->current_schema_in_sqlfile);
      rs->current_schema_in_sqlfile= tmp;

      if (!rs->target_schema && /* the user has not supplied a target schema */
          in_backup_content(tmp, rs->backup_content) && 
          comfort_query(rs, rs->buffer, len, error))
      {                         /* there is at least one table that */
        return -1;              /*   should get restored in this schema */
      }
    }
    else if ((tmp= check_statement(rs->buffer,len,
                                   rs->re_create_database,NULL)))
    {                                                     /* CREATE DATABASE */
      unquote_identifier(tmp);
      if (!rs->target_schema && /* the user has not supplied a target schema */
          in_backup_content(tmp, rs->backup_content) && 
          comfort_query(rs, rs->buffer, len, error))
      {                         /* there is at least one table that */
        g_free(tmp);
        return -1;              /*   should get restored in this schema */
      }
      g_free(tmp);
    }
    else if ((tmp= check_statement(rs->buffer,len,
                                   rs->re_delimiter,NULL))) /* DELIMITER */
    {
      // Delimiter was already determined by the myx_get_next_sql_statement_file call above.
    }
    else
    {
      /*Unknown statement*/
      char buffer[500];
      sprintf(buffer, 
              "Warning: Do not know how to handle this statement at line %d:\n"
              ,rs->we.stmt_begin_line);
      warning= g_strconcat(buffer,
                           rs->buffer,"\nIgnoring this statement. Please file"
                           " a bug-report including the statement if this "
                           "statement should be recognized.\n",NULL);

      if (rs->report_warning)
      {
        (*rs->report_warning)(warning, rs->report_warning_data);
      }
      else
      {
        fprintf(stderr,"%s", warning);
      }
      g_free(warning);
    }
  }
  return count;
}

/*
 * ******************
 *
 * Private functions
 *
 * ******************
 */

/* returns true if backup_content contains a table in schema schema_name */
static int in_backup_content(const char *schema_name,
                             MYX_BACKUP_CONTENT *backup_content)
{
  MYX_BACKUP_TABLE * table= backup_content->tables;
  MYX_BACKUP_TABLE * tables_end= table + backup_content->tables_num;
  for (; table!=tables_end; table++)
  {
    if (        table->schema &&
        !strcmp(table->schema, schema_name))
    {
      return 1;
    }
  }
  return 0;
}

static MYX_BACKUP_PROFILE *read_in_backup_profile_1_1(xmlNodePtr backup_node)
{
  MYX_BACKUP_PROFILE *bprofile;
  xmlNodePtr cur;
  xmlDocPtr doc;

  doc= backup_node->doc;
  bprofile= g_malloc0(sizeof(MYX_BACKUP_PROFILE));
  bprofile->backup_content= g_malloc0(sizeof(MYX_BACKUP_CONTENT));

  for (cur= backup_node->children; cur; cur= cur->next)
  {
    try_to_get_string_field(doc,cur, "profile_name",&bprofile->profile_name);
    try_to_get_string_field(doc,cur, "last_used",   &bprofile->last_used);
    try_to_get_int_field   (doc,cur, "options",     &bprofile->options);
    try_to_get_int_field   (doc,cur, "backup_type",
                            (int*)&bprofile->backup_type);

    if ( !xmlStrcmp(cur->name, (xmlChar*)"tables") &&  cur->type == XML_ELEMENT_NODE)
    {
      xmlNodePtr cur2;
      MYX_BACKUP_TABLE * table;
      bprofile->backup_content->tables_num= get_child_count(cur,(xmlChar*)"table");
      table= bprofile->backup_content->tables=
                         g_malloc0(sizeof(MYX_BACKUP_TABLE) *
                                   bprofile->backup_content->tables_num);
      for (cur2= cur->children; cur2; cur2= cur2->next)
      {
        if (!xmlStrcmp(cur2->name, (xmlChar*)"table") )
        {
          read_in_table(cur2, table);
          table++;
        }
      }
    }
  }
  return bprofile;
}

static MYX_BACKUP_PROFILE *read_in_backup_profile_1_2(xmlNodePtr backup_node)
{
  MYX_BACKUP_PROFILE *bprofile;
  xmlNodePtr cur;
  xmlDocPtr doc;

  doc= backup_node->doc;
  bprofile= g_malloc0(sizeof(MYX_BACKUP_PROFILE));
  bprofile->backup_content= g_malloc0(sizeof(MYX_BACKUP_CONTENT));

  for (cur= backup_node->children; cur; cur= cur->next)
  {
    try_to_get_string_field(doc,cur, "profile_name",&bprofile->profile_name);
    try_to_get_string_field(doc,cur, "last_used",   &bprofile->last_used);
    try_to_get_int_field   (doc,cur, "options",     &bprofile->options);
    try_to_get_int_field   (doc,cur, "backup_type",
                            (int*)&bprofile->backup_type);

    if ( !xmlStrcmp(cur->name, (xmlChar*)"entities") &&  cur->type == XML_ELEMENT_NODE)
    {
      xmlNodePtr cur2;
      MYX_BACKUP_TABLE * table;
      bprofile->backup_content->tables_num= get_child_count(cur,(xmlChar*)"entity");
      table= bprofile->backup_content->tables=
                         g_malloc0(sizeof(MYX_BACKUP_TABLE) *
                                   bprofile->backup_content->tables_num);
      for (cur2= cur->children; cur2; cur2= cur2->next)
      {
        if (!xmlStrcmp(cur2->name, (xmlChar*)"entity") )
        {
          read_in_entity(cur2, table);
          table++;
        }
      }
    }
  }
  return bprofile;
}

static MYX_BACKUP_PROFILE *read_in_backup_profile_1_0(xmlNodePtr backup_node)
{
  MYX_BACKUP_PROFILE *bprofile;
  xmlNodePtr cur;
  xmlDocPtr doc;

  doc= backup_node->doc;
  bprofile= g_malloc0(sizeof(MYX_BACKUP_PROFILE));
  bprofile->backup_content= g_malloc0(sizeof(MYX_BACKUP_CONTENT));

  for (cur= backup_node->children; cur; cur= cur->next)
  {
    try_to_get_string_field(doc,cur, "profile_name",&bprofile->profile_name);
    try_to_get_string_field(doc,cur, "last_used",   &bprofile->last_used);
    try_to_get_int_field   (doc,cur, "options",     &bprofile->options);
    try_to_get_int_field   (doc,cur, "backup_type",
                            (int*)&bprofile->backup_type);

    if ( !xmlStrcmp(cur->name, (xmlChar*)"tables") &&  cur->type == XML_ELEMENT_NODE)
    {
      xmlNodePtr cur2;
      MYX_BACKUP_TABLE * table;
      bprofile->backup_content->tables_num= get_child_count(cur,(xmlChar*)"table");
      table= bprofile->backup_content->tables=
                         g_malloc0(sizeof(MYX_BACKUP_TABLE) *
                                   bprofile->backup_content->tables_num);
      for (cur2= cur->children; cur2; cur2= cur2->next)
      {
        if (!xmlStrcmp(cur2->name, (xmlChar*)"table") )
        {
          read_in_table(cur2, table);
          table++;
        }
      }
    }
  }

  return bprofile;
}


static void read_in_table(xmlNodePtr table_node, MYX_BACKUP_TABLE *table)
{
  xmlNodePtr cur;
  xmlDocPtr doc;

  doc= table_node->doc;
  table->table= NULL;
  table->schema= NULL;
  table->catalog= NULL;
  table->flags= 0;

  for (cur= table_node->children; cur; cur= cur->next)
  {
    try_to_get_string_field(doc,cur, "name",   &table->table);
    try_to_get_string_field(doc,cur, "schema", &table->schema);
    try_to_get_string_field(doc,cur, "catalog",&table->catalog);
  }
}

static void read_in_entity(xmlNodePtr entity_node, MYX_BACKUP_TABLE *table)
{
  char *t= NULL;
  xmlNodePtr cur;
  xmlDocPtr doc;

  doc= entity_node->doc;
  table->table= NULL;
  table->schema= NULL;
  table->catalog= NULL;

  for (cur= entity_node->children; cur; cur= cur->next)
  {
    try_to_get_string_field(doc,cur, "name",   &table->table);
    try_to_get_string_field(doc,cur, "schema", &table->schema);
    try_to_get_string_field(doc,cur, "catalog",&table->catalog);
    try_to_get_string_field(doc,cur, "entity_type", &t);
    if(t)
    {
      if(strcmp(t, "view") == 0)
      {
        table->flags= MYX_BTF_IS_VIEW;
      } 
      else if(strcmp(t, "proc") == 0)
      {
        table->flags= MYX_BTF_IS_PROCEDURE;
      }
      else if(strcmp(t, "func") == 0)
      {
        table->flags= MYX_BTF_IS_FUNCTION;
      }
      else
      {
        table->flags= 0;
      }
      g_free(t);
      t= 0;
    }
  }
}

static GPtrArray *select_all_schemata(MYX_BACKUP_CONTENT *bc)
{
  unsigned int i;
  GPtrArray *array= g_ptr_array_new();
  GHashTable *hash= g_hash_table_new(g_str_hash, g_str_equal);
  
  for(i= 0; i < bc->tables_num; i++)
  {
    if(!g_hash_table_lookup(hash, bc->tables[i].schema))
    {
      MYX_BACKUP_TABLE *bt= g_malloc(sizeof(MYX_BACKUP_TABLE));
      bt->catalog= bc->tables[i].catalog;
      bt->schema= bc->tables[i].schema;
      bt->table= NULL;
      g_ptr_array_add(array, bt);
      g_hash_table_insert(hash, bc->tables[i].schema, bc->tables[i].schema);
    }
  }

  g_hash_table_destroy(hash);
  return array;
}

static MYX_BACKUP_CONTENT* select_all_tables_4(MYSQL* mysql, 
                                               MYX_BACKUP_CONTENT *bc)
{
  unsigned int i, j;
  MYSQL_RES *rs1= NULL, *rs2= NULL, *rs3= NULL;
  MYSQL_ROW row;
  MYX_BACKUP_CONTENT *newbc= (MYX_BACKUP_CONTENT *)g_malloc(sizeof(MYX_BACKUP_CONTENT));

  GPtrArray *all_tables = g_ptr_array_new();
  GPtrArray *all_schemas = select_all_schemata(bc);

  for(i= 0; i < all_schemas->len; i++)
  {
    MYX_BACKUP_TABLE *bt= 
              (MYX_BACKUP_TABLE *)g_ptr_array_index(all_schemas, i);
    if(mysql_select_db(mysql, bt->schema) 
      || myx_mysql_query(mysql, "SHOW TABLES")
      || !(rs1= mysql_store_result(mysql)))
    {
      return NULL;
    }
    j= 0;
    while((row= mysql_fetch_row(rs1)))
    {
      MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));
      btnew->catalog= bt->catalog;
      btnew->schema= bt->schema;
      btnew->table= row[0];
      btnew->flags= 0;
      g_ptr_array_add(all_tables, btnew);
      ++j;
    }

    if(!j)  // empty schema
    {
      MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));
      btnew->catalog= bt->catalog;
      btnew->schema= bt->schema;
      btnew->table= NULL;
      btnew->flags= 0;
      g_ptr_array_add(all_tables, btnew);
    }
  }

  newbc->tables = g_malloc(sizeof(MYX_BACKUP_TABLE)*all_tables->len);
  newbc->tables_num = all_tables->len;

  for(i= 0; i < all_tables->len; i++)
  {
    MYX_BACKUP_TABLE *bt= (MYX_BACKUP_TABLE *)g_ptr_array_index(all_tables, i);
    newbc->tables[i].catalog= g_strdup(bt->catalog);
    newbc->tables[i].schema= g_strdup(bt->schema);
    newbc->tables[i].table= g_strdup(bt->table);
    newbc->tables[i].flags= bt->flags;
    g_free(bt);
  }

  g_ptr_array_free(all_tables, 0);
  for(i= 0; i < all_schemas->len; i++)
  {
    g_free(g_ptr_array_index(all_schemas, i));
  }
  g_ptr_array_free(all_schemas, 0);

  if(rs1)
  {
    mysql_free_result(rs1);
  }
  if(rs2)
  {
    mysql_free_result(rs2);
  }
  if(rs3)
  {
    mysql_free_result(rs3);
  }
  return newbc;
}

static MYX_BACKUP_CONTENT* select_all_tables_5(MYSQL* mysql, 
                                               MYX_BACKUP_CONTENT *bc)
{
  char *t;
  unsigned int i, j;
  MYSQL_RES *rs1= NULL, *rs2= NULL, *rs3= NULL;
  MYSQL_ROW row;
  MYX_BACKUP_CONTENT *newbc= (MYX_BACKUP_CONTENT *)g_malloc(sizeof(MYX_BACKUP_CONTENT));

  GPtrArray *all_tables = g_ptr_array_new();
  GPtrArray *all_schemas = select_all_schemata(bc);

  for(i= 0; i < all_schemas->len; i++)
  {
    MYX_BACKUP_TABLE *bt= 
              (MYX_BACKUP_TABLE *)g_ptr_array_index(all_schemas, i);
    int fields;
    if (mysql_select_db(mysql, bt->schema) 
       || (mysql_query(mysql, "SHOW FULL TABLES") // for mysql 5.0+
           && mysql_query(mysql, "SHOW TABLES")) // for < mysql 5.0
       || !(rs1= mysql_store_result(mysql)))
    {
      return NULL;
    }
    j= 0;
    fields= mysql_num_fields(rs1);
    while((row= mysql_fetch_row(rs1)))
    {
      MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));
      btnew->catalog= bt->catalog;
      btnew->schema= bt->schema;
      btnew->table= row[0];
      if (fields > 1 && strcmp2(row[1], "VIEW") == 0)
      {
        btnew->flags= MYX_BTF_IS_VIEW;
      }
      else
      {
        btnew->flags= 0;
      }
      g_ptr_array_add(all_tables, btnew);
      ++j;
    }

    t= NULL;
    if(!mysql_select_db(mysql, bt->schema) 
      && !mysql_query(mysql, t= g_strconcat("SHOW PROCEDURE STATUS WHERE DB='", bt->schema, "'", NULL))
      && (rs2= mysql_store_result(mysql)))
    {
      while((row= mysql_fetch_row(rs2)))
      {
        MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));

        btnew->catalog= bt->catalog;
        btnew->schema= bt->schema;
        btnew->table= row[1];
        btnew->flags= MYX_BTF_IS_PROCEDURE;
        g_ptr_array_add(all_tables, btnew);
        ++j;
      }
    }
    g_free(t);

    t= NULL;
    if(!mysql_select_db(mysql, bt->schema) 
      && !mysql_query(mysql, t= g_strconcat("SHOW FUNCTION STATUS WHERE DB='", bt->schema, "'", NULL))
      && (rs3= mysql_store_result(mysql)))
    {
      while((row= mysql_fetch_row(rs3)))
      {
        MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));

        btnew->catalog= bt->catalog;
        btnew->schema= bt->schema;
        btnew->table= row[1];
        btnew->flags= MYX_BTF_IS_FUNCTION;
        g_ptr_array_add(all_tables, btnew);
        ++j;
      }
    }
    g_free(t);

    if(!j)  // empty schema
    {
      MYX_BACKUP_TABLE *btnew= 
                (MYX_BACKUP_TABLE *)g_malloc(sizeof(MYX_BACKUP_TABLE));
      btnew->catalog= bt->catalog;
      btnew->schema= bt->schema;
      btnew->table= NULL;
      btnew->flags= 0;
      g_ptr_array_add(all_tables, btnew);
    }
  }

  newbc->tables = g_malloc(sizeof(MYX_BACKUP_TABLE)*all_tables->len);
  newbc->tables_num = all_tables->len;

  for(i= 0; i < all_tables->len; i++)
  {
    MYX_BACKUP_TABLE *bt= (MYX_BACKUP_TABLE *)g_ptr_array_index(all_tables, i);
    newbc->tables[i].catalog= g_strdup(bt->catalog);
    newbc->tables[i].schema= g_strdup(bt->schema);
    newbc->tables[i].table= g_strdup(bt->table);
    newbc->tables[i].flags= bt->flags;
    g_free(bt);
  }

  g_ptr_array_free(all_tables, 0);
  for(i= 0; i < all_schemas->len; i++)
  {
    g_free(g_ptr_array_index(all_schemas, i));
  }
  g_ptr_array_free(all_schemas, 0);

  if(rs1)
  {
    mysql_free_result(rs1);
  }
  if(rs2)
  {
    mysql_free_result(rs2);
  }
  if(rs3)
  {
    mysql_free_result(rs3);
  }

  return newbc;
}

static MYX_BACKUP_CONTENT* copy_backup_content(MYX_BACKUP_CONTENT *content)
{
  MYX_BACKUP_TABLE * src_table= content->tables;
  MYX_BACKUP_TABLE * end_src_tables= src_table + content->tables_num;
  MYX_BACKUP_TABLE * table;
  MYX_BACKUP_CONTENT *copy= g_malloc0(sizeof(MYX_BACKUP_CONTENT));
  copy->tables_num= content->tables_num;
  copy->tables= g_malloc0(sizeof(MYX_BACKUP_TABLE) * copy->tables_num);

  for (table= copy->tables; src_table!=end_src_tables; table++, src_table++)
  {
    table->flags= src_table->flags;
    table->catalog= g_strdup(src_table->catalog);
    table->schema= g_strdup(src_table->schema);
    if(src_table->table && src_table->table[0])
    {
      table->table= g_strdup(src_table->table);
    }
    else
    {
      table->table= NULL;
    }
  }

  return copy;
}

static int  compare_backup_tables(const void *a, const void *b)
{
  int cat_compare;
  MYX_BACKUP_TABLE *t1= (MYX_BACKUP_TABLE *)a;
  MYX_BACKUP_TABLE *t2= (MYX_BACKUP_TABLE *)b;

  // if table content is ignored, then it is either a VIEW or MERGE table
  // dump them last so that they restored after all 'normal' tables
  if((t1->flags & MYX_BTF_IGNORE_CONTENT) != (t1->flags & MYX_BTF_IGNORE_CONTENT))
  {
    if(t1->flags & MYX_BTF_IGNORE_CONTENT)
    {
      return -1;
    }
    else
    {
      return 1;
    }
  }

  /* allow NULL values for the catalogs for now */
  if ( !t1->catalog || !t2->catalog ||
       !(cat_compare= strcmp(t1->catalog, t2->catalog)) )
  {
    return (strcmp(t1->schema, t2->schema));
  }
  else
  {
    return cat_compare;
  }
}

static int write_create_schema_to_file(MYSQL *mysql, 
                                       FILE *sql_file, 
                                       const char *schema_quoted,
                                       MYX_BACKUP_ERROR *error)
{
  MYSQL_ROW row;
  MYSQL_RES *result;
  char *database_name;
  char *stmt= g_strconcat("SHOW CREATE DATABASE WITH IF NOT EXISTS ",
                            schema_quoted,  NULL);

  if (myx_mysql_query(mysql, stmt) ||
      !(result= mysql_store_result(mysql)) )
  {
    SAFE_IO(fprintf(sql_file, "--\n"
      "-- Create schema %s\n"
      "--\n\n", schema_quoted));

    SAFE_IO(fprintf(sql_file,
                    "CREATE DATABASE /*!32312 IF NOT EXISTS*/ %s;\n",
                    schema_quoted));
  }
  else
  {
    row= mysql_fetch_row(result);
    if (!row)
    {
      g_free(stmt);
      *error= MYX_BACKUP_SERVER_ERROR;
      return -1;
    }
    database_name= myx_convert_dbstr_utf8(mysql, row[1]);
    SAFE_IO(fprintf(sql_file, "%s;\n", database_name));
    g_free(database_name);
  }
  g_free(stmt);
  SAFE_IO(fprintf(sql_file, "USE %s;\n",
                  schema_quoted));

  return 0;
}

static int write_path_info_to_file(MYX_BS_STATUS *status,
                                   MYX_BACKUP_ERROR *error)
{
  int result;
  MYX_BACKUP_TABLE *current, *last;
  current= status->backup_content->tables + status->current_table_index;
  last= current - 1;

  /* We ignore the catalog for now */
  if ( status->current_table_index == 0 ||
       strcmp(current->schema, last->schema) )
  {
    if(myx_identifier_needs_quotes(status->current_schema))
    {
      result= write_create_schema_to_file(status->mysql, status->sql_file, 
                                     status->current_schema_quoted, error);
    } 
    else 
    {
      result= write_create_schema_to_file(status->mysql, status->sql_file, 
                                     status->current_schema, error);
    }
    
    if(result)
    {
      return -1;
    }
  }
  return 0;
}

static int write_sql_file_header(MYX_BS_STATUS *status,
                                 MYX_BACKUP_ERROR *error)
{
  gchar Buffer[200] = "/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO";

  if (status->options & MYX_B_COMMENT)
  {
    SAFE_IO(fprintf(status->sql_file,
            "-- MySQL Administrator dump %s\n--\n", ADMIN_DUMP_VERSION));
    SAFE_IO(fputs("-- ------------------------------------------------------\n",
                  status->sql_file));
    SAFE_IO(fprintf(status->sql_file,
                    "-- Server version\t%s\n",
                    mysql_get_server_info(status->mysql)));
    SAFE_IO(fprintf(status->sql_file, "\n\n"));
  }

  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET NAMES %s */;\n\n", "utf8"));
  SAFE_IO(fprintf(status->sql_file, "/*!40014 SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0 */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40014 SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0 */;\n"));

  if (status->options & MYX_B_ANSI_QUOTES)
    g_sprintf(Buffer, "%s,ANSI_QUOTES", Buffer);
  if (status->options & MYX_B_COMPATIBILITY_MODE)
    g_sprintf(Buffer, "%s,MYSQL323", Buffer);
  g_sprintf(Buffer, "%s' */;\n\n\n", Buffer);
  SAFE_IO(fprintf(status->sql_file, Buffer));

  return 0;
}

static int write_sql_file_footer(MYX_BS_STATUS *status, MYX_BACKUP_ERROR *error)
{
  SAFE_IO(fprintf(status->sql_file,"/*!40101 SET SQL_MODE=@OLD_SQL_MODE */;\n"));
  SAFE_IO(fprintf(status->sql_file,"/*!40014 SET FOREIGN_KEY_CHECKS=@OLD_FOREIGN_KEY_CHECKS */;\n"));
  SAFE_IO(fprintf(status->sql_file,"/*!40014 SET UNIQUE_CHECKS=@OLD_UNIQUE_CHECKS */;\n"));

  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;\n"));
  SAFE_IO(fprintf(status->sql_file, "/*!40101 SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION */;\n"));

  fprintf(status->sql_file,"/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;\n");

  return 0;
}

static int write_create_table_statement_to_file(MYX_BS_STATUS *status,
                                          MYX_BACKUP_ERROR *error)
{
  char *stmt, *create_statement;
  MYSQL_ROW row;
  MYSQL_RES *result;

  /* make sure show create table's output is quoted */
  if (myx_mysql_query(status->mysql, "SET SQL_QUOTE_SHOW_CREATE=1"))
  {
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_COMMENT)
  {
    SAFE_IO(fprintf(status->sql_file, "\n--\n"
      "-- Table structure for table %s.%s\n"
      "--\n", 
      status->current_schema_quoted,
      status->current_table_quoted));
  }

  stmt= g_strconcat("SHOW CREATE TABLE ",
                    status->current_schema_quoted, ".",
                    status->current_table_quoted, NULL);
  if ((myx_mysql_query(status->mysql, stmt)) ||
      !(result= mysql_store_result(status->mysql)) )
  {
    g_free(stmt);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }
  g_free(stmt);
  row= mysql_fetch_row(result);
  if (!row)
  {
    mysql_free_result(result);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_ADD_DROP_TABLE)
    SAFE_IO(fprintf(status->sql_file,
                    "\nDROP TABLE IF EXISTS %s;\n",
                    status->current_table_quoted));

  /* TODO: check if the quotes are set right */

  /* row[1] has the create-statement */
  create_statement= myx_convert_dbstr_utf8(status->mysql, row[1]);
  SAFE_IO(fprintf(status->sql_file,"%s;\n", create_statement));
  g_free(create_statement);

  mysql_free_result(result);
  return 0;
}

static int write_create_view_statement_to_file(MYX_BS_STATUS *status,
                                               MYX_BACKUP_ERROR *error)
{
  char *stmt, *create_statement;
  MYSQL_ROW row;
  MYSQL_RES *result;

  /* make sure show create table's output is quoted */
  if (myx_mysql_query(status->mysql, "SET SQL_QUOTE_SHOW_CREATE=1"))
  {
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_COMMENT)
  {
    SAFE_IO(fprintf(status->sql_file, "\n--\n"
      "-- View structure for view %s.%s\n"
      "--\n", 
      status->current_schema_quoted,
      status->current_table_quoted));
  }

  stmt= g_strconcat("SHOW CREATE VIEW ",
                    status->current_schema_quoted, ".",
                    status->current_table_quoted, NULL);
  if ((myx_mysql_query(status->mysql, stmt)) ||
      !(result= mysql_store_result(status->mysql)) )
  {
    g_free(stmt);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }
  g_free(stmt);
  row= mysql_fetch_row(result);
  if (!row)
  {
    mysql_free_result(result);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_ADD_DROP_TABLE)
    SAFE_IO(fprintf(status->sql_file,
                    "\nDROP VIEW IF EXISTS %s;\n",
                    status->current_table_quoted));

  /* TODO: check if the quotes are set right */

  /* row[1] has the create-statement */
  create_statement= myx_convert_dbstr_utf8(status->mysql, row[1]);
  SAFE_IO(fprintf(status->sql_file,"%s;\n", create_statement));
  g_free(create_statement);

  mysql_free_result(result);
  return 0;
}

static int write_create_sp_statement_to_file(MYX_BS_STATUS *status,
                                          MYX_BACKUP_ERROR *error)
{
  char *entity, *title;
  char *stmt, *create_statement;
  MYSQL_ROW row;
  MYSQL_RES *result;

  if(status->backup_content->tables[status->current_table_index].flags & MYX_BTF_IS_PROCEDURE)
  {
    entity= "PROCEDURE";
    title= "Procedure";
  }
  else
  {
    entity= "FUNCTION";
    title= "Function";
  }

  /* make sure show create table's output is quoted */
  if (myx_mysql_query(status->mysql, "SET SQL_QUOTE_SHOW_CREATE=1"))
  {
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_COMMENT)
  {
    SAFE_IO(fprintf(status->sql_file, "\n--\n"
      "-- %s %s.%s\n"
      "--\n", 
      title,
      status->current_schema_quoted,
      status->current_table_quoted));
  }

  stmt= g_strconcat("SHOW CREATE ",
                    entity,
                    " ",
                    status->current_schema_quoted, ".",
                    status->current_table_quoted, NULL);
  if ((myx_mysql_query(status->mysql, stmt)) ||
      !(result= mysql_store_result(status->mysql)) )
  {
    g_free(stmt);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }
  g_free(stmt);
  row= mysql_fetch_row(result);
  if (!row)
  {
    mysql_free_result(result);
    *error= MYX_BACKUP_SERVER_ERROR;
    return -1;
  }

  if (status->options & MYX_B_ADD_DROP_TABLE)
    SAFE_IO(fprintf(status->sql_file,
                    "\nDROP %s IF EXISTS %s;\n",
                    entity,
                    status->current_table_quoted));

  /* row[2] has the create-statement */
  create_statement= myx_convert_dbstr_utf8(status->mysql, row[2]);
  SAFE_IO(fprintf(status->sql_file,"DELIMITER $$\n\n%s $$\n\nDELIMITER ;", create_statement));
  g_free(create_statement);

  mysql_free_result(result);
  return 0;
}

static int write_create_statement_to_file(MYX_BS_STATUS *status,
                                          MYX_BACKUP_ERROR *error)
{
  int flags= status->backup_content->tables[status->current_table_index].flags;

  if(flags & (MYX_BTF_IS_PROCEDURE|MYX_BTF_IS_FUNCTION))
  {
    return write_create_sp_statement_to_file(status, error);
  } 
  else if(flags & MYX_BTF_IS_VIEW)
  {
    return write_create_view_statement_to_file(status, error);
  }
  else
  {
    return write_create_table_statement_to_file(status, error);
  }
}



static int write_row_to_file(MYX_BS_STATUS *status,
                             MYSQL_ROW row, MYX_BACKUP_ERROR *error)
{
  unsigned int i;
  char *insert_beg;
  static char hex_char[]= {'0','1','2','3','4','5','6','7',
                           '8','9','A','B','C','D','E','F'};

  char *value_stmt= g_strdup(" ");

  unsigned int num_fields= mysql_num_fields(status->mysql_result);
  unsigned long * lengths= mysql_fetch_lengths(status->mysql_result);
  MYSQL_FIELD * fields= mysql_fetch_fields(status->mysql_result);

  if (!(status->options & MYX_B_COMPLETE_INSERTS))
  {
    insert_beg= g_strconcat("INSERT INTO ",
                            status->current_table_quoted," VALUES ",NULL);
  }
  else
  {
    insert_beg= g_strconcat("INSERT INTO ",
                            status->current_table_quoted," (", NULL);
    for(i= 0; i < num_fields; i++)
    {
      char *quoted= quote_identifier_with(fields[i].name, status->quote_char);
      insert_beg= str_g_append_and_free(insert_beg,
                                        g_strconcat(quoted,",",NULL));
      g_free(quoted);
    }
    insert_beg[strlen(insert_beg)-1]= ')';
    insert_beg= str_g_append(insert_beg, " VALUES \n");
  }

  if (! status->extended_insert)
    status->extended_insert= g_strdup(insert_beg);

  for(i= 0; i < num_fields; i++)
  {
    bigint length= lengths[i];
    MYSQL_FIELD *field= fields+i;
    value_stmt= str_g_append(value_stmt, i == 0 ? "(" : ",");

    if (!row[i]) /* if == NULL */
    {
      value_stmt= str_g_append(value_stmt, "NULL"); // write NULL
    }
    else
    {
      if (!length) /* if == empty string */
      {
        value_stmt= str_g_append(value_stmt, "''"); // write ''
      }
      else
      {
        if (field->flags & BLOB_FLAG && field->flags & BINARY_FLAG)
        { /* a real blob */
          int j,k;
          char * from;
          char *hex_str= g_malloc((gulong)(length*2+2+1));
          hex_str[0]= '0';
          hex_str[1]= 'x';

          from= row[i];
          for (j= 2,k= 0; k < length; j+=2, k++)
          {
            unsigned int data= (unsigned int) (unsigned char) from[k];
            hex_str[j]= hex_char[data >> 4];
            hex_str[j+1]= hex_char[data & 15];
          }
          hex_str[j]= 0;

          value_stmt= str_g_append_and_free(value_stmt, hex_str);
        }
        else if (IS_NUM_FIELD(field))
        {
          value_stmt=
            !strcmp(row[i],"nan") ||
            !strcmp(row[i],"inf") || !strcmp(row[i],"-inf")
              ? str_g_append(value_stmt, "NULL")
              : field->type != FIELD_TYPE_DECIMAL
                  ? str_g_append(value_stmt, row[i])
                  : str_g_append_and_free(value_stmt,
                                          g_strconcat("'",row[i],"'",NULL));
        }
        else
        {
          char *tmp_str= g_malloc((unsigned int)length*2+1+2);
          unsigned int len= mysql_real_escape_string(status->mysql,
                                                     tmp_str+1, row[i],
                                                     (unsigned int)length);
          tmp_str[0]= '\'';
          tmp_str[len+1]= '\'';
          tmp_str[len+2]= '\0';
          value_stmt= str_g_append_and_free(value_stmt, tmp_str);
          //g_free(tmp_str);
        }
      }
    }
  }

  value_stmt= str_g_append(value_stmt, ")");

  if (status->options & MYX_B_NO_EXTENDED_INSERT)
  {
    char *tmp_str= myx_convert_dbstr_utf8(status->mysql, value_stmt);
    SAFE_IO(fprintf(status->sql_file, "%s %s;\n",insert_beg, tmp_str));
    g_free(tmp_str);
  }
  else  /*extended inserts are not always written out*/
  {
    size_t len= strlen(status->extended_insert);
    status->extended_insert= str_g_append(status->extended_insert, value_stmt);
    if (len < 1000)
    {
      status->extended_insert= str_g_append(status->extended_insert, ",\n");
    }
    else /* we write it out */
    {
      char *tmp_str;
      status->extended_insert= str_g_append(status->extended_insert, ";");

      tmp_str= myx_convert_dbstr_utf8(status->mysql, status->extended_insert);
      SAFE_IO(fprintf(status->sql_file, "%s\n", tmp_str));
      g_free(tmp_str);

      g_free(status->extended_insert);
      status->extended_insert= NULL;
    }
  }

  g_free(insert_beg);
  g_free(value_stmt);

  return 0;
}


/* A simple wrapper around mysql_query().
 *
 * Return value:
 *     -1 in case of a fatal error
 *      0 if successful
 */
static int comfort_query(MYX_RBS_STATUS *rs,
                         char *buffer, int buffer_len, MYX_BACKUP_ERROR *error)
{
  char *warning;

  if(!rs->we.alt_delimiter && (buffer[buffer_len-1] == ';'))
  {
    buffer_len -= 1;
  }
  else if(rs->we.alt_delimiter)
  {
    if(strncmp(buffer + buffer_len - strlen(rs->we.alt_delimiter), rs->we.alt_delimiter, strlen(rs->we.alt_delimiter)) == 0)
    {
      buffer_len -= (int)strlen(rs->we.alt_delimiter);
    }
  }

  assert(buffer);

  if (myx_mysql_real_query(rs->mysql, buffer, buffer_len) )
  {
    if (! (rs->options &  MYX_RBS_FORCE))
    {
      *error= MYX_BACKUP_SERVER_ERROR;
      return -1;
    }

    warning= g_strconcat("Warning: Error while executing this query:", buffer,
                         "\nThe server has returned this error message:",
                         mysql_error(rs->mysql), NULL);
    if (rs->report_warning)
    {
      (*rs->report_warning)(warning, rs->report_warning_data);
    }
    else
    {
      fprintf(stderr, "%s\n",warning);
    }
    g_free(warning);
  }
  return 0;
}

/* sends the query to the database, issueing a use database if necessary before
*/
static int interact_with_server(MYX_RBS_STATUS *rs,
                                char *buffer, int buffer_len,
                                MYX_BACKUP_ERROR *error)
{
  if (!rs->target_schema)
  {
    /* The user has not supplied an explicit target schema.
     * We will take the target-information from the sql-file ie.
     * USE and create db statements will not be filtered when they
     * are read from the sql-file but be executed. Thus we can assume
     * here that we are already in the correct schema.
     */
  }
  else if (strcmp(rs->current_schema_in_db, rs->target_schema))
  {
    /* Change the default database to target_schema.
     * If target_schema does not exist yet, we will create it.*/
    char *q_use= g_strconcat("USE ", rs->target_schema, NULL);

    if (!myx_mysql_query(rs->mysql,q_use)) 
    {
      g_free(q_use);
    }
    else /* Error executing USE */
    {
      if ( (mysql_errno(rs->mysql) == 1049) &&           /* Unknown database */
           !(rs->options & MYX_RBS_DONT_CREATE_TARGETS))
      {
        char *q_cre= g_strconcat("CREATE DATABASE ", rs->target_schema, NULL);

        if ( comfort_query(rs, q_cre, (int)strlen(q_cre), error ) ||
             comfort_query(rs, q_use, (int)strlen(q_use), error ))
        {
          /* error is already set by comfort_query */
          g_free(q_use);

          g_free(q_cre);
          return -1;
        }
        g_free(q_cre);
      }
      else
      {
        char *warning;
        g_free(q_use);
        if (! (rs->options & MYX_RBS_FORCE))
        {
          *error= MYX_BACKUP_SERVER_ERROR;
          return -1;
        }
        warning= g_strconcat("Warning: Could not change the db.\n The server "
                             "has returned this error message:",
                             mysql_error(rs->mysql), NULL);
        if (rs->report_warning)
        {
          (*rs->report_warning)(warning, rs->report_warning_data);
        }
        else
        {
          fprintf(stderr,"%s", warning);
        }
        g_free(warning);
      }
    }

    g_free(rs->current_schema_in_db);
    rs->current_schema_in_db= g_strdup(rs->target_schema);
  }

  // go ahead and send the query
  return comfort_query(rs, buffer,buffer_len, error);
}

static void free_backup_table_content(MYX_BACKUP_TABLE *t)
{
  g_free(t->schema);
  g_free(t->catalog);
  g_free(t->table);
}

/* returns >0 if the given table is part of the backup-content */
static int find_table(MYX_BACKUP_CONTENT *b, const char *table_name,
                      const char *schema_name, const char *catalog_name)
{
  MYX_BACKUP_TABLE * table= b->tables;
  MYX_BACKUP_TABLE * tables_end= table + b->tables_num;

  for (; table!=tables_end; table++)
  {
    if (!strcmp(table->table, table_name) &&
        !strcmp(table->schema, schema_name)) // Ignore catalogs for now
    {
      return 1;
    }
  }
  return 0;
}


/* Rexexp functions */

#define O_VECTOR_COUNT 6

static char* check_statement(const char *line, int line_len,
                             pcre *re, pcre_extra *pe)
{
  int o_vector[O_VECTOR_COUNT];
  const char *ret_val;
  int rc;
  char *return_value;

  if ( (rc= pcre_exec(re,pe, line, line_len,0,0,
                      o_vector, O_VECTOR_COUNT) ) > 0)
  {
    pcre_get_substring(line, o_vector, rc, 1, &ret_val);

    return_value= g_strdup(ret_val);
    pcre_free_substring((char*)ret_val);
    return return_value;
  }

  return NULL;
}

/*returns the table-name or NULL if this is no create-statement*/
static int prepare_line_is_create_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;

  if (! (*re = pcre_compile("create\\s+table\\s+"TABLE_NAME_PCRE, /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static int prepare_line_is_create_index_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;
  const char *p= "create\\s+index\\s+"TABLE_NAME_PCRE;

  if (! (*re = pcre_compile(p, /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static int prepare_line_is_create_view_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;
  const char *p= "create\\s+[a-z=A-Z]*\\s+view\\s+[^.]*[.]"TABLE_NAME_PCRE;

  if (! (*re = pcre_compile(p, /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static int prepare_line_is_create_proc_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;

  if (! (*re = pcre_compile("create\\s+procedure\\s+.*[.]"TABLE_NAME_PCRE, /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static int prepare_line_is_create_func_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;

  if (! (*re = pcre_compile("create\\s+function\\s+.*[.]"TABLE_NAME_PCRE, /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static void finalize_line_is_create_statement(pcre *re, pcre_extra *pe)
{
  if (re) pcre_free(re);
  if (pe) pcre_free(pe);
}

static int prepare_line_is_db_use_statement(pcre **re, pcre_extra **pe)
{
  const char *error;
  int erroffset;

  if (! (*re = pcre_compile("use\\s+(\\S+)\\s*;", /* the pattern */
                            PCRE_CASELESS|PCRE_ANCHORED, /* default options */
                            &error, /* for error message */
                            &erroffset, /* for error offset */
                            NULL)) ) /* use default character tables */
  {
    fprintf(stderr, "%s\n",error);
    return 1;
  }

  if (pe)
  {
    *pe = pcre_study(
                     *re, /* result of pcre_compile() */
                     0, /* no options exist */
                     &error);
  }
  return 0;
}

static void finalize_line_is_db_use_statement(pcre *re, pcre_extra *pe)
{
  if (re) pcre_free(re);
  if (pe) pcre_free(pe);
}




MYX_BACKUP_CONTENT * myx_get_restore_drop_list(MYSQL *mysql, MYX_BACKUP_CONTENT *content)
{
  MYX_BACKUP_CONTENT *drop_list= g_new0(MYX_BACKUP_CONTENT, 1);
  struct Schemas {
    MYX_SCHEMA_TABLES *tables;
    const char *catalog;
    const char *schema;
  } *schemas;
  unsigned int i, k;
  unsigned int schemas_num= 0;

  schemas= g_new0(struct Schemas, content->tables_num);
  for (i= 0; i < content->tables_num; i++)
  {
    int found= 0;
    for (k= 0; k < schemas_num; k++)
    {
      if (strcmp3(schemas[k].catalog, content->tables[i].catalog)==0
          && strcmp3(schemas[k].schema, content->tables[i].schema)==0)
      {
        found= 1;
        break;
      }
    }
    if (found) continue;

    schemas[schemas_num].catalog= content->tables[i].catalog;
    schemas[schemas_num].schema= content->tables[i].schema;
    schemas[schemas_num].tables= myx_get_schema_tables(mysql,
                                             content->tables[i].catalog,
                                             content->tables[i].schema);
    schemas_num++;
  }

  drop_list->tables= g_malloc0(sizeof(MYX_BACKUP_TABLE)*content->tables_num);
  drop_list->tables_num= 0;
  for (i= 0; i < content->tables_num; i++)
  {
    unsigned int s;
    int found= 0;
    for (s= 0; s < schemas_num; s++)
    {
      if (strcmp(schemas[s].catalog, content->tables[i].catalog)==0
          && strcmp(schemas[s].schema, content->tables[i].schema)==0
          && schemas[s].tables != NULL)
      {
        found= 1;
        break;
      }
    }
    if (found)
    {
      for (k= 0; k < schemas[s].tables->schema_tables_num; k++)
      {
        if (strcmp(schemas[s].tables->schema_tables[k].table_name, content->tables[i].table)==0)
        {
          drop_list->tables[drop_list->tables_num].catalog= g_strdup(content->tables[i].catalog);
          drop_list->tables[drop_list->tables_num].schema= g_strdup(content->tables[i].schema);
          drop_list->tables[drop_list->tables_num].table= g_strdup(content->tables[i].table);
          drop_list->tables_num++;
          break;
        }
      }
    }
  }


  for (k= 0; k < schemas_num; k++)
  {
    if (schemas[k].tables)
      myx_free_schema_tables(schemas[k].tables);
  }
  g_free(schemas);
  
  return drop_list;
}


char * myx_get_backup_error_string(MYX_BACKUP_ERROR error)
{
  switch (error)
  {
  case MYX_BACKUP_NO_ERROR:
    return g_strdup(_("Success."));
  case MYX_BACKUP_SERVER_ERROR:
    return g_strdup(_("MySQL Error."));
  case MYX_BACKUP_CANT_OPEN_FILE:
    return g_strdup(_("Cannot open file."));
  case MYX_BACKUP_ILLEGAL_OPTION:
    return g_strdup(_("Illegal option."));
  case MYX_BACKUP_PCRE_ERROR:
    return g_strdup(_("Internal parsing error (pcre)."));
  case MYX_BACKUP_MALLOC_FAILED:
    return g_strdup(_("Memory allocation failed."));
  case MYX_BACKUP_OUTPUTDEVICE_FULL:
    return g_strdup(_("Disk full."));
  case MYX_BACKUP_CANNOT_FLUSH_TABLES_WITH_READ_LOCK:
    return g_strdup(_("Cannot flush tables with read lock."));
  case MYX_BACKUP_CANNOT_START_TRANSACTION:
    return g_strdup(_("Cannot start transaction."));
  case MYX_BACKUP_CANNOT_SET_ANSI_QUOTES:
    return g_strdup(_("Cannot set ANSI quotes."));
  }
  return NULL;
}

