// Copyright (c) 2010, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// This implementation was inspired by John Goerzen's hg-buildpackage.

class MercurialBuildPackage
{
	public static function main()
	{
		var usage = "Usage: " + Constants.buildPackage() + " [-V|--version]"
		+ " [-v|--verbose] [-d|--no-check-dependencies]"
		+ " [-s|-sa|--include-source] [-f|--from-version changelogversion]"
		+ " [-c|--configfile pbuilderconfigfile]"
		;
		var options = new GetPot( neko.Sys.args() );
		Util.maybeReportVersion( options, usage );
		// Collect verbosity options.
		verbosity = 0;
		while( options.got( ["--verbose","-v"] ) ) ++verbosity;
		// Create a build-log file.
		var dpkgArch = Process.runButExitOnError
		(
			"dpkg-architecture", ["-qDEB_HOST_ARCH"]
		);
		if( ! dpkgArch.hasNext() )
		{
			Util.die( "Could not determine architecture." );
		}
		var arch = dpkgArch.next();
		// Check for presence of Mercurial directory.
		Mercurial.exitIfNoWorkingDirectory( "." );
		// Extract package information from the changelog file.
		var changeLog = DebianChangeLog.parseCurrentChangeLog();
		var fileNameBase = "../" + changeLog.source + "_"
			+ changeLog.version.archiveVersion();
		var archFileName = fileNameBase + "_" + arch;
		var logFileName = archFileName + ".build";
		log = neko.io.File.write( logFileName, false );		
		// Find out which branch we are on.
		var branch = Process.runButExitOnError( "hg", ["branch"] );
		if( ! branch.hasNext() ) Util.die( "No branch information?" );
		info( 1, "Branch " + branch.next() );
		info( 2, "Building for architecture " + arch );
		// Use pbuilder?
		var pbuilderConfig : String = null;
		if( options.got( ["-c","--configfile"] ) )
		{
			pbuilderConfig = options.next();
			if( pbuilderConfig == null ) Util.die( usage );
		}
		// Check for proper build dependencies.
		var checkDepends = ! options.got( ["-d","--no-check-dependencies"] );
		if( pbuilderConfig != null && ! checkDepends )
		{
			Util.die
			(
				"No build-dependency-check does not work with pbuilder."
			);
		}
		if( pbuilderConfig == null && checkDepends )
		{
			Process.runButExitOnError( "dpkg-checkbuilddeps", [] );
		}
		var includeSource = options.got( ["-s","-sa","--include-source"] );
		var fromVersion : String = null;
		if( options.got( ["-f","--from-version"] ) )
		{
			fromVersion = options.next();
			if( fromVersion == null ) Util.die( usage );
		}
		Util.exitOnExtraneousArguments( options, usage );
		// Exit if there is no upstream source tarball(s).
		checkUpstreamConsistency( changeLog );
		// Make sure debian/rules is executable.
		Process.runButExitOnError( "chmod", ["+x","debian/rules"] );
		// Call clean target.
		info( 1, "Starting debian/rules clean." );
		var clean = new Process
		(
			Constants.fakeroot(), ["debian/rules", "clean"]
		);
		infos( 2, clean.stdout() );
		if( clean.process.exitCode() != 0 )
		{
			infos( 0, clean.stderr() );
			Util.die( "All build-depends for clean need to be installed." );
		}
		// Build source package excluding Mercurial stuff.  dpkg-source itself
		// will exclude Quilt directories.
		var cwd = neko.Sys.getCwd();
		var dpkgArgs = Constants.tarExclude().concat( [ "-i.hg", "-b", cwd ] );
		info( 1, "Starting dpkg-source " + dpkgArgs.join(" ") + " in .." );
		var source = Process.runButExitOnError ( "dpkg-source", dpkgArgs, ".." );
		infos( 0, source );
		// Use pbuilder?
		if( pbuilderConfig != null )
		{
			pbuilderBuild( pbuilderConfig, fileNameBase + ".dsc" );
		}
		else
		{
			regularBuild( includeSource, fromVersion, archFileName);
		}
		neko.Sys.exit( 0 );
	}

	//
	// Exits the program if the changelog says that this package should have an
	// upstream source but there is no corresponding tarball.
	//
	static function checkUpstreamConsistency( changeLog : DebianChangeLog )
	{
		if( changeLog.version.debianRevision == null )
		{
			info( 2, "Building native package." );
			return;
		}
		// Check existance of an upstream main tarball.
		var mainTarballStem = "../" + changeLog.source + "_" +
			changeLog.version.upstream + ".orig.tar.";
		for( extension in ["gz","bz2","lzma"] )
		{
			var tarball = mainTarballStem + extension;
			if( FileUtil.fileExists( tarball ) )
			{				
				info( 3, "Located upstream tarball " + tarball );
				return;
			}
		}
		Util.dies
		( [
			"No upstream tarball " + mainTarballStem  + "{gz,bz2,lzma} located.",
			"Use " + Constants.pristineTar() + " to extract the tarball(s)."
		] );
	}
	
	//
	// Use pbuilder to build the package.
	//
	static function pbuilderBuild( configFileName : String, dscFileName : String )
	{
		var pbuild =
		[
			"pbuilder", "--build", "--configfile", configFileName, dscFileName
		];
		info( 0, "Starting " + pbuild.join(" ") );
		var build = new Process( Constants.sudo(), pbuild );
		for( line in build.stdout() ) info( 0, line );
		if( build.process.exitCode() != 0 )
		{
			Util.writeErrors( build.stderr() );	
			neko.Sys.exit( 1 );				
		}
	}
	
	//
	// Use debian/rules and dpkg-genchanges to build the package.
	//
	static function regularBuild
	(
		includeSource : Bool,
		fromVersion : String,
		archFileName : String
	)
	{
		info( 1, "Starting debian/rules build." );
		var build = new Process( "debian/rules", ["build"] );
		for( line in build.stdout() ) info( 0, line );
		if( build.process.exitCode() != 0 )
		{
			Util.writeErrors( build.stderr() );	
			neko.Sys.exit( 1 );		
		}
		info( 1, "Starting debian/rules binary." );
		var binary = new Process
		(
			Constants.fakeroot(),
			["debian/rules","binary"]
		);
		for( line in binary.stdout() ) info( 0, line );
		if( binary.process.exitCode() != 0 )
		{
			Util.writeErrors( binary.stderr() );
			neko.Sys.exit( 1 );		
		}
		// Use dpkg-genchanges to generate changes file.
		var changesFileName = archFileName + ".changes";
		var changesArgs = [];
		if( includeSource ) changesArgs.push( "-sa" );
		if( fromVersion != null ) changesArgs.push( "-v" + fromVersion );
		info( 1, "Starting dpkg-genchanges " + changesArgs.join(" ") );
		var changes = Process.runButExitOnError( "dpkg-genchanges", changesArgs );
		var changesFile = neko.io.File.write( changesFileName, false );
		for( line in changes ) changesFile.writeString( line + "\n" );
	}

	//
	// Higher values means more output.
	//
	static var verbosity : Int;

	//
	// Build-log file.
	//
	static var log : neko.io.FileOutput;

	//
	// Always write a line of info to the log, but only Write to stdout if the
	// verbosity level is high enough.
	//
	static function info( level : Int, line : String )
	{
		log.writeString( line + "\n" );
		if( verbosity >= level ) Util.writeInfo( line );
	}

	//
	// Always write lines of info to the log, but only Write to stdout if the
	// verbosity level is high enough.
	//
	static function infos( level : Int, lines : Iterator<String> )
	{
		for( line in lines )
		{			
			log.writeString( line + "\n" );
			if( verbosity >= level ) Util.writeInfo( line );
		}
	}
}
