/*
 * Copyright (C) 2016 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of version 3 of the GNU Lesser General Public License as published
 * by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Gary Wang  <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_TASKHANDLER_H_
#define MCLOUD_API_TASKHANDLER_H_

#include <mcloud/api/task.h>

#include <core/net/http/streaming_request.h>

#include <memory>
#include <string>
#include <atomic>
#include <future>

namespace mcloud {
namespace api {

/*!
    \class TaskHandler
    \brief TaskHandler is a class that can be embeded in DownloadTask and UploadTask to easy handle task running state
 */

struct TaskHandler {

    typedef std::function<void(void*)> PrepareHandler;

    typedef std::function<void(float)> ProgressHandler;

    typedef std::function<void(const std::string&)> ReadyReadHandler;

    typedef std::function<bool(void)> FinishedHandler;

    typedef std::function<void(Task::Status)> StatusHandler;

    typedef std::shared_ptr<TaskHandler> Ptr;

    /*!
     * \brief Constructs a task hanlder object and initialize task status as Unstart.
     * \sa status()
     */
    TaskHandler(): 
        task_status(Task::Status::Unstart),
        status_handler(nullptr) {
    }

    virtual ~TaskHandler() {
        detach_request();
    }

    /*!
     * \brief Cancel a Task and set status as Canceled.
     * \sa status()
     */
    void cancel() {
        set_status(Task::Status::Canceled);
        detach_request();
    }

    /*!
     * \brief Pause a task and set status as Paused.
     * \sa status()
     */
    void pause() {
        set_status(Task::Status::Paused);
        if (task_request)
            task_request->pause();
    }

    /*!
     * \brief Resume a task and set status as Running.
     * \sa status()
     */
    void resume() {
        set_status(Task::Status::Running);
        if (task_request)
            task_request->resume();
    }

    /*!
     * \brief Returns current status of a Task.
     * \sa Client::Status
     */
    Task::Status status() const {
        return task_status.load();
    }

    /*!
     * \brief Set current status \a s of a Task.
     * \sa status()
     */
    void set_status(Task::Status status) {
        task_status.store(status);
        if (status_handler) {
            status_handler(TaskHandler::status());
        }
    }

    /*!
     * \brief Increase broken counter of a Task.
     * Task item will not be regarded as a broken one even if it's failed during sync-up
     * until broken counter exceeds broken limit.
     * \sa is_broken(), reset_broken_counter()
     */
    void add_broken_counter() {
        broken_counter.increase();
    }

    /*!
     * \brief Returns true if task item is broken, otherwise returns false.
     * \sa reset_broken_counter(), add_broken_counter()
     */
    bool is_broken() const {
        return broken_counter.value() >= broken_limit;
    }

    /*!
     * \brief Reset a task item broken counter.
     * internal broken counter will be set to 0
     */
    void reset_broken_counter() {
        broken_counter.reset();
    }

    /*!
     * \brief attach network request \a request and \a future from internal http client
     * \sa detach_request()
     */
    void attach_request(std::shared_ptr<core::net::http::StreamingRequest> request,
                   std::shared_future<bool> future) {
        task_request = std::make_shared<TaskRequest>(request, future);
    }

    /*!
     * \brief detach network request from internal http client
     * \sa attach_request()
     */
    void detach_request() {
        task_request.reset();
    }

    /*!
     * \brief wait for task to finished
     */
    void wait_to_finished() {
        if (task_request) {
            task_request->wait_finished();
        }
    }

    /*!
     * \brief wait for task to timeout after \a seconds
     * \return std::future_status
     */
    std::future_status wait_for(std::chrono::seconds secs) {
        if (task_request) {
            return task_request->wait_for(secs);
        }
        return std::future_status::ready;
    }

    /*!
     * \brief get task result
     */
    bool get_result() {
        if (task_request) {
            return task_request->get_result();
        }
        return false;
    }

    TaskHandler::PrepareHandler &on_prepare() {
        return prepare_handler;
    }

    TaskHandler::ProgressHandler &on_progress() {
        return progress_handler;
    }

    TaskHandler::FinishedHandler &on_finished() {
        return finished_handler;
    }

    TaskHandler::ReadyReadHandler &on_ready_read() {
        return ready_read_handler;
    }

    TaskHandler::StatusHandler &on_status() {
        return status_handler; 
    }

protected:
    TaskHandler (const TaskHandler &) = delete;
    TaskHandler (TaskHandler &&) = delete;

private:
    struct BrokenCounter : std::atomic<int>  {
        BrokenCounter() : std::atomic<int>(0) {}
        void increase() { fetch_add(1); }
        void reset() { store(0); }
        int  value() const { return load(); }
    };

    struct TaskRequest {
        TaskRequest(std::shared_ptr<core::net::http::StreamingRequest> request,
                      std::shared_future<bool> future) :
            request_(request),
            future_(future)
        {}

        std::shared_ptr<core::net::http::StreamingRequest> request_;
        std::shared_future<bool> future_;

        void wait_finished() {
            future_.wait();
        }

        std::future_status wait_for(std::chrono::seconds secs) {
            return future_.wait_for(secs);
        }

        void pause() {
            request_->pause();
        }

        void resume() {
            request_->resume();
        }

        bool get_result() {
            return future_.get();
        }
    };

    BrokenCounter       broken_counter;

    std::atomic<Task::Status> task_status;

    PrepareHandler      prepare_handler;

    ProgressHandler     progress_handler;

    FinishedHandler     finished_handler;

    ReadyReadHandler    ready_read_handler;

    StatusHandler       status_handler;

    std::shared_ptr<TaskRequest>    task_request;

    const int broken_limit = 2;
};

}
}

#endif // MCLOUD_API_TASKHANDLER_H_
