/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_UPLOADTASK_H_
#define MCLOUD_API_UPLOADTASK_H_

#include <mcloud/api/task.h>
#include <mcloud/api/visibility.h>

#include <memory>
#include <vector>

namespace mcloud {
namespace api {

class TaskHandler;

/*!
    \struct UploadRequest
    \brief UploadRequest is a upload request item which consists of a upload folder id, up-front buffer size, upload content name 
 */
struct MCLOUD_API_DLL_PUBLIC UploadRequest
{
    std::string file_path;
    std::string folder_id;
    std::string content_name;
};
typedef std::vector<UploadRequest> UploadRequestList;

/*!
    \struct UploadBufferCb
    \brief UploadBufferCb is a upload request item which consists of a upload folder id, upload buffer size, upload content name and reading callback function allows content data can be received in buffering via call back function. 
*/
struct MCLOUD_API_DLL_PUBLIC UploadBufferCb
{
    Task::Buffer_Callback read_cb;
    size_t      buffer_size;
    std::string folder_id;
    std::string content_name;
};
typedef std::vector<UploadBufferCb> UploadBufferCbList;

class UploadTaskPriv;
class SyncManagerPriv;

/*!
    \class UploadTask
    \brief UploadTask is a task item that can be accessed from application layer to fetch
    basic upload item information, also it's used for content upload by sync manager.
 */
class MCLOUD_API_DLL_PUBLIC UploadTask : public Task {
public:
    typedef std::shared_ptr<UploadTask> Ptr;

    virtual ~UploadTask() = default;

    UploadTask(const UploadTask& ) = delete;

    UploadTask& operator=(const UploadTask& ) = delete;

    /*!
     * \brief Returns an unique id of upload task.
     */
    const std::string & task_id() const override;

    /*!
     * \brief Returns an unique id of cloud content on mcloud.
     */
    const std::string & content_id() const override;

    /*!
     * \brief Returns a display name of local content.
     */
    const std::string & content_name() const override;

    /*!
     * \brief Returns uploaded local content file path.
     */
    const std::string & file_path() const override;

    /*!
     * \brief Contains the error string if an error occurs during content uploading.
     */
    const std::string & error_string() const override;

    /*!
     * \brief Returns upload url assigned by mcloud for this task.
     * \note the url will be expired after a period of time.
     */
    const std::string & task_url() const override;

    /*!
     * \brief Returns current sync-up status for this task.
     */
    Task::Status status() const override;

    /*!
     * \brief Handler for upload progress of a task.
     * \sa Task::ProgressHandler
     */
    Task::ProgressHandler & progress_changed() override;

    /*!
     * \brief Handler for upload status of a task.
     * \sa Task::StatusHandler
     */
    Task::StatusHandler & status_changed() override;

    /*!
     * \brief cancel the task.
     * \sa Task::StatusHandler
     */
    void cancel() override;

    /*!
     * \brief Returns file size of upload content.
     */
    size_t file_size() const;

    /*!
     * \brief Returns true if a uploaded item does not exist in cloud, otherwise return false.
     */
    bool is_need_upload() const;

private:
    UploadTask(std::shared_ptr<UploadTaskPriv> p);

    friend class SyncManagerPriv;

    std::shared_ptr<UploadTaskPriv>   p_;
};

}
}

#endif // MCLOUD_API_UPLOADTASK_H_
