/*!================================================================
 module:        OMS_KernelDefaultInterface.hpp

 responsible:   ThomasA
 
 special area:  OMS
 
 description:   OMS kernel default interface
 
 see:
 



    ========== licence begin  GPL
    Copyright (c) 1998-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




 ===================================================================*/



#ifndef OMS_KERNELDEFAULTINTERFACE_HPP
#define OMS_KERNELDEFAULTINTERFACE_HPP

#include "liveCache/LVC_KernelInterface.hpp"

/*!*****************************************************************************

   class:        OMS_KernelDefaultInterface
   description:  This class implements a singleton, that provides a default
                 implementation of the LVC_KernelInterface. 
                 The singleton is used in cases where kernel functions are
                 called while the kernel is not available.
                 This is the case during registration of a COM dll.

*******************************************************************************/

class OMS_KernelDefaultInterface : public LVC_KernelInterface 
{
  public :
  /*!-------------------------------------------------------------------------
	function:     Allocate
	description:  Allocates contiguous memory for 'count' bytes from LVCMem_Allocator.

    arguments:    count [in] number of requested bytes
    
	return value: pointer to memory chunk
  --------------------------------------------------------------------------*/
  
  virtual void* Allocate(SAPDB_ULong count);

  /*!-------------------------------------------------------------------------
	function:     Clock
	description:  returns the number of seconds since system start
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/
  
  virtual tsp00_Int4 Clock();

  /*!-------------------------------------------------------------------------
	function:     Deallocate
	description:  Deallocates contiguous memory identified by p from LVCMem_Allocator.

    arguments:    p [in] memory to be deallocated
    
	return value: 0
  --------------------------------------------------------------------------*/

  virtual void Deallocate(void* p);

  /*!-------------------------------------------------------------------------
	function:     DeregisterAllocator
	description:  deregisters an allocator in the kernel allocator registry.  
		   
	arguments:    allocatorInfo [in] a structure containing allocator info.
	return value: none
   --------------------------------------------------------------------------*/

  virtual void DeregisterAllocator(RTEMem_AllocatorInfo& allocatorInfo);

  /*!-------------------------------------------------------------------------
	function:     GetOmsAllocator
	description:  returns a pointer to the global liveCache oms allocator 
		   
	arguments:    none
	return value: pointer to the liveCache oms allocator
  --------------------------------------------------------------------------*/

  virtual SAPDBMem_IRawAllocator* GetOmsAllocator();

  /*!-------------------------------------------------------------------------
	function:     GetKnlVersion
	description:  fills a buffer with liveCache version information 
		   
	arguments:    pBuffer [out] address of the buffer to be filled
                bufferLength [in] the size of the buffer
	return value: 0
  --------------------------------------------------------------------------*/

  virtual void* GetKnlVersion(void* pBuffer,int bufferLength);

  /*!-------------------------------------------------------------------------
	function:     GetLockRequestTimeout
	description:  yields the value of the parameter REQUEST_TIMEOUT 
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/

  virtual tsp00_Int4 GetLockRequestTimeout();

  /*!-------------------------------------------------------------------------
	function:     GetOmsRegionCnt
	description:  yields the value of the parameter _OMS_REGIONS 
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/

  virtual int   GetOmsRegionCnt();

  /*!-------------------------------------------------------------------------
	function:     GetSinkPtr
	description:  yields the address of the current kernel sink, i.e. the
                kernel callback of the current session.
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/

  virtual IliveCacheSink* GetSinkPtr();

  /*!-------------------------------------------------------------------------
	function:     GetOmsVersionThreshold
	description:  yields the value of the parameter OMS_VERS_THRESHOLD 
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/

  virtual tsp00_Uint4 GetOmsVersionThreshold();

  /*!-------------------------------------------------------------------------
	function:     GetPacketMaxUsableArea
	description:  yields the maximal usable area of the packet, i.e. the
                number of bytes, that can be transferred via a packet
                containing one data part.
		   
	arguments:    none
	return value: the number of bytes, that can be transferred via a packet
                containing one data part.
  --------------------------------------------------------------------------*/

  virtual tsp00_Int4 GetPacketMaxUsableArea();
  
  /*!-------------------------------------------------------------------------
	function:     HeapThresholdReached
	description:  returns true, if the heap threshold has been reached, i.e.
                OMS_HEAP_THRESHOLD percent of the heap are in use 
		   
	arguments:    none
	return value: 0
  --------------------------------------------------------------------------*/

  virtual bool  HeapThresholdReached();

  /*!-------------------------------------------------------------------------
    function:     IncOmsVersionUnloadCounter
    description:  increments the counter counting number of oms version unloads
   
    arguments:    none
     
    return value: none
   --------------------------------------------------------------------------*/
 
  virtual void IncOmsVersionUnloadCounter();

  /*!
   * \brief check if OneDB system
   *
   * Tells whether liveCache is a OneDB system.
   *
   * \return true, if liveCache is a OneDB system, false otherwise
   */
  virtual bool IsOneDBSystem();

  /*!-----------------------------------------------------------------------
   function:     IsUnicodeInstance
   description:  tells whether liveCache is runnung in UNICODE mode
   arguments:    none
   return value: true, if parameter _UNICODE='YES', otherwise false
   -------------------------------------------------------------------------*/  
  
  virtual bool IsUnicodeInstance();

  /*!-------------------------------------------------------------------------
   function:     OmsAllocate
   description:  Allocates contiguous memory for 'count' bytes from oms allocator.
  
   arguments:    count [in] number of requested bytes
    
   return value: pointer to memory chunk
  --------------------------------------------------------------------------*/

  virtual void* OmsAllocate(SAPDB_ULong count);

  /*!-------------------------------------------------------------------------
   function:     Deallocate
   description:  Deallocates contiguous memory identified by p from LVCMem_Allocator.
  
   arguments:    p [in] memory to be deallocated
    
   return value: none
  --------------------------------------------------------------------------*/
  
  virtual void OmsDeallocate(void* p);

  /*!-------------------------------------------------------------------------
	function:     RegisterAllocator
	description:  registers an allocator in the kernel allocator registry.  
		   
	arguments:    allocatorInfo [in] a structure containing allocator info.
	return value: none
   --------------------------------------------------------------------------*/

  virtual void RegisterAllocator(RTEMem_AllocatorInfo& allocatorInfo);

  /*!-------------------------------------------------------------------------
	function:     Signal
	description:  noop method.  
		   
	arguments:    taskId [in] the task identification of the waiting task.
                timeout[in] if true, waitung task is timed out
	return value: none
  --------------------------------------------------------------------------*/

  virtual void Signal(tsp00_TaskId taskId, bool timeout);

  /*!-------------------------------------------------------------------------
	function:     SignalUnloadLiboms
	description:  noop method 
		   
	arguments:    none
	return value: none
  --------------------------------------------------------------------------*/

  virtual void SignalUnloadLiboms();

  /*!-------------------------------------------------------------------------
	function:     TestBadAlloc
	description:  noop method
		   
	arguments:    none
	return value: none
  --------------------------------------------------------------------------*/
  
  virtual void TestBadAlloc();

  /*!-------------------------------------------------------------------------
	function:	  SimCtlAttachSink
	description:  attach a new sink to a handle in simulator to ease
				  writing tests

	arguments:	  handle [in]	OMS handle to attach new sink to
	return value: none
  --------------------------------------------------------------------------*/

  virtual void SimCtlAttachSink(OmsHandle &handle);

  /*!-------------------------------------------------------------------------
	function:	  ConsoleMessage
	description:  print a console message, used in test programs in simulator

	arguments:	  msg [in]		message to print
	return value: none
  --------------------------------------------------------------------------*/

  virtual void ConsoleMessage(const char *msg);

  /*!-------------------------------------------------------------------------
	function:     AllocateArray
	description:  Allocates contiguous memory for 'count' bytes from LVCMem_Allocator
                for use by array new[].

  arguments:    count [in] number of requested bytes
    
	return value: pointer to memory chunk
  --------------------------------------------------------------------------*/
  
  virtual void* AllocateArray(SAPDB_ULong count);

  /*!-------------------------------------------------------------------------
	function:     DeallocateArray
	description:  Deallocates contiguous memory identified by p from LVCMem_Allocator
                for use by array delete[].
    
	return value: none
  --------------------------------------------------------------------------*/

  virtual void DeallocateArray(void* p);

  virtual void DumpStackTrace();

  virtual unsigned int SaveStackTrace(void **buffer, unsigned int levels);

  virtual void EvalStackTrace(void **buffer, unsigned int levels,
    OmsStackTraceCallbackInterface *cb);

  /// Get time in microseconds.
  virtual SAPDB_UInt8 GetMicroTime();

  /// Get file information.
  virtual bool StatFile(
    const char  *fn,
    SAPDB_UInt8 &ctime,
    SAPDB_UInt8 &fsize);

  /// Format time into ASCII.
  virtual void FormatTime(
    char        *buffer,
    size_t      bufsize,
    SAPDB_UInt8 time);
  /*-----------------------------------------------------*/
  
  virtual short LockRequest(LVC_LockRequest &lockReq);

  /*!-------------------------------------------------------------------------
	function:     UseReaderWriterLocks
	description:  Use readerWriterLocks or CriticalSections for synchonization
                of the version dictionary (new with PTS 1124170)
	   
	arguments:    none
	return value: none
  --------------------------------------------------------------------------*/
  virtual bool UseReaderWriterLocks();

  /*!-------------------------------------------------------------------------
	function:     Crash
	description:  writes msg into knldiag and crashes the kernel.  
		   
	arguments:    msg [in] msg to write into knldiag
	return value: none
  --------------------------------------------------------------------------*/
  virtual void Crash(const char* msg);

  /*!-------------------------------------------------------------------------
	function:     DumpChunk
	description:  dumps a memory chunk.  
		   
	arguments:    p [in] address of the chunk
                size [in] size of the chunk
	return value: none
  --------------------------------------------------------------------------*/
  virtual void DumpChunk (void* p, int size);

  /*!-------------------------------------------------------------------------
	function:     GetHeapCheckLevel
	description:  returns the value of the parameter HEAP_CHECK_LEVEL.  
		   
	arguments:    none
	return value: the value of the parameter HEAP_CHECK_LEVEL
  --------------------------------------------------------------------------*/
  virtual tsp00_Int4 GetHeapCheckLevel();

  /*!-------------------------------------------------------------------------
	function:     Trace
	description:  writes msg into knldiag.  
		   
	arguments:    msg [in] msg to write into knldiag
	return value: none
  --------------------------------------------------------------------------*/
  virtual void Trace(const char* msg);

  /*!-------------------------------------------------------------------------
	function:     Instance
	description:  returns a reference to the OMS_KernelDefaultInterface singleton instance.
		   
	arguments:    none
	return value: reference to OMS_KernelDefaultInterface instance
  --------------------------------------------------------------------------*/

  static LVC_KernelInterface& Instance();

  /*!-------------------------------------------------------------------------
	function:     SetErrorCallback
	description:  Set the error callback interface in the kernel (new with PTS 1133630)
	   
	arguments:    Pointer to DbpErrorCallbackInterface
	return value: none
  --------------------------------------------------------------------------*/

  virtual void SetErrorCallback(DbpErrorCallbackInterface *cb);

  /*!
   * \brief Get kernel runtime interface
   *
   * Returns a pointer to the liveCache kernel runtime interface
   *
   * \return pointer to the liveCache kernel runtime interface
   */
  virtual RTE_IInterface &GetKernelRunTimeInterface();

private :
  static OMS_KernelDefaultInterface* m_instance;
};

/*!*****************************************************************************

   endclass: OMS_KernelDefaultInterface

*******************************************************************************/

#endif
