/*!
  @file           Log_ActionSetFileState.hpp
  @author         UweH
  @ingroup        ActionHandling
  @brief          set the filestate

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/



#ifndef LOG_ACTION_SET_FILE_STATE_H
#define LOG_ACTION_SET_FILE_STATE_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_ReadWriteActionImage.hpp"

#include "ggg00.h"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

//===================================================================================
/*!
   @brief          defines a class to handle the after image of SetFileState
   @copydoc        Log_ReadWriteActionImage
 */

class Log_ActionSetFileState : public Log_ReadWriteActionImage
{

private:
    
    /// physical representation of this action
    struct Head
    {
        tgg00_Filename filename;      ///< filename (16 bytes)
        SAPDB_Bool1    readonly;      ///< set to readonly (yes/no) (1 byte)
        SAPDB_Bool1    filler1;       ///< filler (1 byte)
        SAPDB_UInt2    filler2;       ///< filler (2 byte)

        /// constructor
        Head()
        {
            memset (filename, 0, sizeof(filename));
            readonly = 0;
            filler1  = 0;
            filler2  = 0;
        }

        /// constructor
        Head( const tgg00_Filename &newfilename, bool newreadonly )
        {
            filename = newfilename;
            readonly = newreadonly?1:0;
            filler1  = 0;
            filler2  = 0;
        }
    };
    
    /// the in memory representation
    Head m_head;

public:
    
    /// This is used for reading persistent images.
    Log_ActionSetFileState()
    : Log_ReadWriteActionImage (Log_SetFileState)
    {}

    /// This is used for rollback of begin table load
    Log_ActionSetFileState( const tgg00_Filename &filename,
                            bool                  readonly )
    : Log_ReadWriteActionImage (Log_SetFileState),
      m_head (filename, readonly)
    {}
    
    /// called by command
    void ExecuteAction (tgg00_TransContext &TransContext,
                        tgg00_FileId       *pCommandFileId = NULL,
                        bool                readonly       = false) const;

    virtual void Execute (tgg00_TransContext &TransContext) const
    {
        ExecuteAction (TransContext);
    }

    virtual SAPDB_UInt GetPersistentLength () const
    {
        return Log_AlignedImageLen(sizeof(m_head));
    }

    /// reads the image  from the reader into memory
    void ReadImagePersistent (Log_ActionImageReader &reader,
                              bool                  &isOk)
    {
        reader.CopyImage( reinterpret_cast<SAPDB_Byte*>(&m_head),
                          sizeof(m_head),
                          isOk );
    }

    /// writes the memory from to the writer
    void WriteImagePersistent (Log_ActionImageWriter &writer,
                               bool                  &isOk) const
    {
        writer.PersistentWrite (&m_head, sizeof (m_head), isOk);
    }
    
};


/*==========================================================================*
*  DEFINITION OF INLINE METHODS                                             *
*==========================================================================*/

#endif  /* LOG_ACTION_SET_FILE_STATE_H */
