/*!**************************************************************************

  module      : Kernel_Dump.hpp

  -------------------------------------------------------------------------

  responsible : TorstenS

  special area: Kernel dump file handling
  description : This class is used to write main memory structures into
                the kernel dump file.


  last changed: 2001-06-07  18:54
  see also    : 

  -------------------------------------------------------------------------

  copyright:    (c) 2001-2004 SAP AG

    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*****************************************************************************/


#ifndef KERNEL_DUMP_HPP
#define KERNEL_DUMP_HPP



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "gsp00.h"

#include "KernelCommon/Kernel_Common.hpp"
#include "KernelCommon/Kernel_DumpPage.hpp"
#include "KernelCommon/Kernel_FileIO.hpp"
#include "KernelCommon/Kernel_IPage.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

static struct Kernel_DumpInfoType
{
    SAPDB_Char  dmpLabel[KERNEL_DUMP_LABEL_LENGTH+1];
    SAPDB_UInt2 dmpCode;
}
Kernel_DumpInfo[] =         // see also vxt08!!
{
    { "NIL     ", 2000 }, // => DmpNil
    { "CONVMGM ", 2001 }, // => DmpConverterManager
    { "CONVMAPC", 2002 }, // => DmpConverterMapControl
    { "CONVMAPS", 2003 }, // => DmpConverterMapSection
    { "CONVMAPE", 2004 }, // => DmpConverterMapEntry
    { "CONVIDX ", 2005 }, // => DmpConverterIndex
    { "FBMMGM  ", 2010 }, // => DmpFBMManager
    { "FBMDEV  ", 2011 }, // => DmpFBMDataDevice
    { "PAGERCNT", 2020 }  // => DmpPagerController
};

/*!*****************************************************************************

   class: Kernel_Dump

   description: This class is used to write main memory structures into
                the kernel dump file.
              
*******************************************************************************/

class Kernel_Dump
{
public:

    /* -----------------------------------------------------------------------*/
    /*! declarations: Type definitions                                             */
    /* -----------------------------------------------------------------------*/
    
    enum Kernel_DumpEntryId  // changes will affect Kernel_DumpInfo too!!
    {
        DmpNil = 0,
        DmpConverterManager,
        DmpConverterMapControl,
        DmpConverterMapSection,
        DmpConverterMapEntry,
        DmpConverterIndex,
        DmpFBMManager,
        DmpFBMDataDevice,
        DmpPagerController,
		DmpEnd
    };
    
    /* -----------------------------------------------------------------------*/
    /*! enddeclarations: Type definitions                                          */
    /* -----------------------------------------------------------------------*/

    /* -----------------------------------------------------------------------*/
    /*! chapter: Constructors and initialization                              */
    /* -----------------------------------------------------------------------*/

    /*!-------------------------------------------------------------------------
    function:     Kernel_Dump()
    description:  This is the constructor of the kernel dump file.
    arguments:    FileName  [in] Name of the kernel dump file
                  PageFrame [in] I/O capable buffer for the dump entries
    return value: none
    --------------------------------------------------------------------------*/
    
    Kernel_Dump( 
        const tsp00_VFilename           &FileName,
        const Kernel_IPage::PageFrame   &PageFrame )
        :
        m_File( FileName ),
        m_Page( PageFrame ),
        m_PageNo( 0 ),
        m_LastEntryId( DmpNil )
        {}

    /*!-------------------------------------------------------------------------
    function:     ~Kernel_Dump()
    description:  This is the destructor of the kernel dump file. The last 
                  updated dump page is filled and written into the kernel
                  dump file.
    arguments:    none
    return value: none
    --------------------------------------------------------------------------*/

    ~Kernel_Dump()
    {
        Flush();
    }

    /* -----------------------------------------------------------------------*/
    /*! endchapter: Constructors and initialization                           */
    /* -----------------------------------------------------------------------*/

    /* -----------------------------------------------------------------------*/
    /*! chapter: Update operations                                            */
    /* -----------------------------------------------------------------------*/
    
    /*!-------------------------------------------------------------------------
    function:     InsertEntry()
    description:  This method inserts a given dump entry in the kernel dump
                  file. Each dump entry belongs to a group of dump informations
                  identified by a DumpEntryId. Each dump page contains only the
                  dump entries of one DumpEntryId. If a new DumpEntryId is given
                  the old dump page is written into the file and the new entry
                  is stored on a fresh dump page together with the internal dump
                  label which is build by the given DumpEntryId. It is assumed
                  that dump entries of the same group are written stepwise,
                  for instance each entry of a data cache control block is written
                  with a single InsertEntry call! Dump entries are limited in size.
                  The upper bound is delimited by the page size less than about
                  20 bytes.
    arguments:    DumpEntryId [in] identification of the given entry
                  DumpEntry   [in] given dump entry
    return value: true if insert was successfully; else false
    --------------------------------------------------------------------------*/

    SAPDB_Bool  InsertEntry( 
        Kernel_DumpEntryId              DumpEntryId,
        const Kernel_DumpPage::Entry    &DumpEntry );
    
    /*!-------------------------------------------------------------------------
    function:     InsertPage()
    description:  This method inserts a given page in the kernel dump file.
                  If the current dump page is not completely filled with dump
                  entries this dump page will be filled with up with a special
                  token and then the page is flushed before the given page is
                  written into the kernel dump file.
    arguments:    Page [in] kernel page handler
    return value: true if insert was successfully; else false
    --------------------------------------------------------------------------*/

    SAPDB_Bool InsertPage( const Kernel_Page  &Page );

    /* -----------------------------------------------------------------------*/
    /*! endchapter: Update operations                                         */
    /* -----------------------------------------------------------------------*/

    /* -----------------------------------------------------------------------*/
    /*! chapter: Miscellaneous operations                                     */
    /* -----------------------------------------------------------------------*/

    /*!-------------------------------------------------------------------------
    function:     Flush()
    description:  The current dump page is filled with a special token and 
                  than it is written into the kernel dump file.
    arguments:    none
    return value: none
    --------------------------------------------------------------------------*/

    void Flush();
    
    /*!-------------------------------------------------------------------------
    function:     ErrorOccured()
    description:  This method is used to get the kernel dump file error state.
    arguments:    none
    return value: true if a error has occured
    --------------------------------------------------------------------------*/

    SAPDB_Bool  ErrorOccured()
    {
        return( m_File.ErrorOccured() );
    }

    /*!-------------------------------------------------------------------------
    function:     GetPageNo()
    description:  This method returns the current dump page number. It is
                  only used by the old pascal dump.
    arguments:    none
    return value: none
    --------------------------------------------------------------------------*/

    SAPDB_UInt4 GetPageNo() const
    {
        return( m_PageNo );
    }

    /*!-------------------------------------------------------------------------
    function:     GetFileNo()
    description:  This method returns the kernel dump file identifier. It is
                  only used by the old pascal dump.
    arguments:    none
    return value: none
    --------------------------------------------------------------------------*/

    SAPDB_UInt4 GetFileNo() const
    {
        return( m_File.GetFileNo() );
    }

    /* -----------------------------------------------------------------------*/
    /*! endchapter: Miscellaneous operations                                  */
    /* -----------------------------------------------------------------------*/


private:

    /*-------------------------------------------------------------------------
    declaration: m_File
    description: This is the handler for the kernel dump file.
    --------------------------------------------------------------------------*/

    Kernel_FileIO   m_File;

    /*-------------------------------------------------------------------------
    declaration: m_Page
    description: This is the current dump page on which the dump entries
                 will be stored.
    --------------------------------------------------------------------------*/

    Kernel_DumpPage m_Page;

    /*-------------------------------------------------------------------------
    declaration: m_PageNo
    description: This is the page number of the current dump page.
    --------------------------------------------------------------------------*/

    SAPDB_UInt4 m_PageNo;

    /*-------------------------------------------------------------------------
    declaration: m_LastEntryId
    description: This is the unique identifier for the dump entries stored 
                 at this time. 
    --------------------------------------------------------------------------*/

    Kernel_DumpEntryId  m_LastEntryId;
};

/*!*****************************************************************************

   endclass: Kernel_Dump

*******************************************************************************/



#endif  /* KERNEL_DUMP_HPP */
