/*!
  @file           FBM_Exception.hpp
  @author         TorstenS
  @author         AlexanderK
  @ingroup        FBM
  @brief          Error messages

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/



#ifndef FBM_EXCEPTION_HPP
#define FBM_EXCEPTION_HPP


#include "FreeBlockManagement/FBM_Messages.hpp"
#include "FreeBlockManagement/FBM_Types.hpp"

/*!
   @brief          defines common exception class of this component

   - class FBM_Exception
 */
SAPDBERR_DEFINE_COMPONENT_EXCEPTION_CLASS( FBM );

/*! enddeclaration */


/*!
  @interface      FBM_IllegalTransition
  @brief          FBM exception for illegal state transistions
 */

class FBM_IllegalTransition : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message displaying a wrong transition between block states
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] device number of the block address
       @param          BlockNo [in] block number of the block address
       @param          OldBlockState [in] current state of the block
       @param          OldBackUpBlockState [in] current back up state
       @param          NewBlockState [in] wanted state of the block
       @param          NewBackUpBlockState [in] wanted back up state
       @return         none

       - Constructor of an error message. The error occurs when the FBM is ask to
      change the state of a block in a way which is not permitted
     */
    FBM_IllegalTransition
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               BlockNo,
     const FBM_BlockState           OldBlockState,
     const FBM_BlockState           OldBackUpBlockState,
     const FBM_BlockState           NewBlockState,
     const FBM_BlockState           NewBackUpBlockState
    )
    :
    FBM_Exception
    (filename, linenumber,
     FBM_ILLEGAL_TRANSITION,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(BlockNo, _T_d),
     FBM_BlockStateStrings( OldBlockState ),
     FBM_BlockStateStrings( OldBackUpBlockState ),
     FBM_BlockStateStrings( NewBlockState ),
     FBM_BlockStateStrings( NewBackUpBlockState ))
    {}
};


/*!
  @interface      FBM_IllegalRestoreAfterCompression

 */

class FBM_IllegalRestoreAfterCompression : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message displaying a wrong transition between block states
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] device number of the block address
       @param          BlockNo [in] block number of the block address
       @param          BlockState [in] current state of the block
       @param          BackUpBlockState [in] current back up state
       @return         none


       - Constructor of an error message. The error  occurs when the FBM is ask to
      restore the original state of a block marked for compression, but the block
    is not at all marked for compression
     */
    FBM_IllegalRestoreAfterCompression
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               BlockNo,
     const FBM_BlockState           BlockState,
     const FBM_BlockState           BackUpBlockState
    )
    :
    FBM_Exception
    (filename, linenumber,
     FBM_ILLEGAL_RESTORE_AFTER_COMPRESSION,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(BlockNo, _T_d),
     FBM_BlockStateStrings( BlockState ),
     FBM_BlockStateStrings( BackUpBlockState ))
    {}
};



/*!
  @interface      FBM_IllegalTransitionToFreeAfterSVP

 */

class FBM_IllegalTransitionToFreeAfterSVP : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message displaying a wrong transition between block states
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] device number of the block address
       @param          BlockNo [in] block number of the block address
       @param          BlockState [in] current state of the block
       @param          BackUpBlockState [in] current back up state
       @return         none

       - Constructor of an error message. The error  occurs when the FBM is ask to
      change the state of a block to 'free after savepoint' but the current state
    does not allow this transition
     */
    FBM_IllegalTransitionToFreeAfterSVP
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               BlockNo,
     const FBM_BlockState           BlockState,
     const FBM_BlockState           BackUpBlockState
    )
    :
    FBM_Exception
    (filename, linenumber,
     FBM_ILLEGAL_TRANS_TO_FREE_AFTER_SVP,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(BlockNo, _T_d),
     FBM_BlockStateStrings( BlockState ),
     FBM_BlockStateStrings( BackUpBlockState ))
    {}
};


/*!
  @interface      FBM_IllegalRestoreAfterBackup

 */
class FBM_IllegalRestoreAfterBackup : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message displaying a wrong transition between block states
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] device number of the block address
       @param          BlockNo [in] block number of the block address
       @param          BlockState [in] current state of the block
       @param          BackUpBlockState [in] current back up state
       @return         none


       - Constructor of an error message. The error occurs when the FBM is ask to
      restore the original state of a block marked for backup, but the block
    is not at all marked for backup
     */
    FBM_IllegalRestoreAfterBackup
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               BlockNo,
     const FBM_BlockState           BlockState,
     const FBM_BlockState           BackUpBlockState
    )
    :
    FBM_Exception
    (filename, linenumber,
     FBM_ILLEGAL_RESTORE_AFTER_BACKUP,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(BlockNo, _T_d),
     FBM_BlockStateStrings( BlockState ),
     FBM_BlockStateStrings( BackUpBlockState ))
    {}
};



/*!
  @class          FBM_NoMoreMemoryForNewDevice

 */

class FBM_NoMoreMemoryForNewDevice : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message needed if there is no more meory to register a new device
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] number of the device which could not be registered
       @param          DevSize [in] size of the device  which could not be registered
       @return         none

     */

    FBM_NoMoreMemoryForNewDevice
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               DevSize)
    :
    FBM_Exception
    (filename, linenumber,
     FBM_NO_MORE_MEMORY_FOR_NEW_DEVICE,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(DevSize, _T_d))
    {}
};



/*!
  @class          FBM_WrongDevNo

 */

class FBM_WrongDevNo : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message needed if the number of a new device is not inthe correct range
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          DevNo [in] wrong device number
       @param          MinDevNo [in] MinDevNo and MaxDevNo define the range a valid devno can be in
       @param          MaxDevNo [in] MinDevNo &lt;= DevNo &lt; MaxDevNo
       @return         none

     */

    FBM_WrongDevNo
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               DevNo,
     const SAPDB_UInt               MinDevNo,
     const SAPDB_UInt               MaxDevNo)
    :
    FBM_Exception
    (filename, linenumber,
     FBM_WRONG_DEVNO,
     SAPDB_ToString(DevNo, _T_d),
     SAPDB_ToString(MinDevNo, _T_d),
     SAPDB_ToString(MaxDevNo, _T_d))
    {}
};



/*!
  @interface      FBM_IllegalState

 */

class FBM_IllegalState : public FBM_Exception
{
public:
    /*!
       @brief          Constructor of an error message needed if the number of a new device is not inthe correct range
       @param          filename [in] filename (to be generated)
       @param          linenumber [in] linenumber (to be generated)
       @param          BlockNo [in] number of the blocjk which is in the illegal state
       @param          BlockState [in] current state of the block
       @param          BackUpBlockState [in] current back up state
       @return         none

     */

    FBM_IllegalState
    (const SAPDB_Char*              filename,
     const SAPDB_UInt4              linenumber,
     const SAPDB_UInt               BlockNo,
     const FBM_BlockState           BlockState,
     const FBM_BlockState           BackUpBlockState)
    :
    FBM_Exception
    (filename, linenumber,
     FBM_ILLEGAL_STATE,
     SAPDB_ToString(BlockNo, _T_d),
     FBM_BlockStateStrings( BlockState ),
     FBM_BlockStateStrings( BackUpBlockState ))
    {}
};

/// block info message
class FBM_BlockNoConverter : public FBM_Exception
{
public:
	/*!
	   @brief          Constructor of an info message
	   @param          filename [in] filename (to be generated)
	   @param          linenumber [in] linenumber (to be generated)
	   @param          BlockNo [in] number of the blocjk which is in the illegal state
	   @param          BlockState [in] current state of the block
	   @return         none

	 */

	FBM_BlockNoConverter
	(const SAPDB_Char*              filename,
	 const SAPDB_UInt4              linenumber,
 	 const SAPDB_UInt               VolNo,
	 const SAPDB_UInt               BlockNo,
	 const FBM_BlockState           BlockState)
	:
	FBM_Exception
	(filename, linenumber,
	 FBM_INFO_NOT_IN_CONVERTER,
	 SAPDB_ToString( VolNo, _T_d ),
  	 SAPDB_ToString( BlockNo, _T_d ),
	 FBM_BlockStateStrings( BlockState ))
	{}
};



#endif  /* FBM_EXCEPTION_HPP */
