/****************************************************************************

  module      : Data_TraverseTreeAccess.cpp

  -------------------------------------------------------------------------

  responsible : TorstenS

  special area:
  description : 

  last changed: 2001-01-20
  see also    :

  -------------------------------------------------------------------------

  copyright:    (c) 2001-2004 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "ggg90.h"

#include "hbd30.h"
#include "hgg01_3.h"


#include "DataAccess/Data_TraverseTreeAccess.hpp"
#include "DataAccess/Data_Messages.hpp"
#include "DataAccess/Data_Exceptions.hpp"
#include "RunTime/RTE_Crash.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/



/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/



/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

Data_TraverseTreeAccess::Data_TraverseTreeAccess( tbd_current_tree    &current )
        :
        m_Current( current ),
        m_TrError( current.curr_trans->trError_gg00 ),
        m_PageAccessManager( current ),
        m_Page()
{}

/*---------------------------------------------------------------------------*/

Data_TraverseTreeAccess::~Data_TraverseTreeAccess()
{
    m_Page.Deassign( false ); // release the page for read
}

/*---------------------------------------------------------------------------*/

Data_PrimPage&
Data_TraverseTreeAccess::GetNode()
{
    SAPDBTRACE_ROUTINE_DEBUG( "Data_TraverseTreeAccess::GetNode", DataTree_Trace, 5 );

    if( ! m_Page.IsAssigned() )
    {
        SetToRootNode();
        return( m_Page );
    }

    const Data_PageNo currentPageNo = m_Page.PageNo();
    const Data_PageNo nextPageNo    = m_Page.Successor();
    const Data_Level  currentLevel  = m_Page.GetLevel();

    m_PageAccessManager.ReleasePage( m_Page );

    if( nextPageNo.IsValid() )
    {
        if( currentLevel == LEAF_LEVEL_BD00 )
        {
            ReplaceLeafNodeLock( currentPageNo );
        }
        else if( currentLevel == FIRST_INDEX_LEVEL_BD00 )
        {
            ReplaceFirstLevelIndexNodeLock( nextPageNo );
        }

        if( e_ok == m_TrError )
        {
            m_PageAccessManager.GetPage( m_Page, nextPageNo );
            return( m_Page );
        }
    }
    else if( LEAF_LEVEL_BD00 < currentLevel )
    {
        if( GetLeftMostNodeAtNextLevel( currentLevel -1 ))
            return( m_Page );
    }
    // error handling
    if( m_Page.IsAssigned() )
        m_PageAccessManager.ReleasePage( m_Page );

    return m_Page;
}

/*---------------------------------------------------------------------------*/
/*                            private methods                                */
/*---------------------------------------------------------------------------*/

SAPDB_Bool
Data_TraverseTreeAccess::SetToRootNode()
{
    SAPDBERR_ASSERT_STATE( ! m_Page.IsAssigned() );
    SAPDBERR_ASSERT_STATE( NULL != m_Current.currRootNptrs_bd00.np_ptr() );

    m_PageAccessManager.GetPage( m_Page, m_Current.curr_tree_id.fileRoot_gg00() );

    return( e_ok == m_TrError );
}

/*---------------------------------------------------------------------------*/

SAPDB_Bool
Data_TraverseTreeAccess::GetLeftMostNodeAtNextLevel( const Data_Level   stopLevel )
{
    SAPDBERR_ASSERT_STATE( ! m_Page.IsAssigned() );

    bd30ReleaseSubTree( m_Current );

    SetToRootNode();

    while( e_ok == m_TrError )
    {
        const tgg00_Rec     *pSeparator     = m_Page.GetFirstRecord();
        const Data_PageNo   childPagePageNo = pSeparator->recPno_gg00().gg90GetInt4();
        const Data_Level    parentLevel     = m_Page.GetLevel();

        m_PageAccessManager.ReleasePage( m_Page );

        if(
            ( m_Current.curr_tree_id.fileType_gg00().includes( ftsPerm_egg00 )) &&
            ( SECOND_INDEX_LEVEL_BD00 == parentLevel                                )
        )
        {
            bd30GetSubTree( m_Current, childPagePageNo );

            if( e_ok != m_TrError )
                return( false );
        }
        if( ! m_PageAccessManager.GetPage( m_Page, childPagePageNo ))
            return( false );

        if( stopLevel ==  m_Page.GetLevel() )
            return( true );
    }
    return( false );
}

/*---------------------------------------------------------------------------*/

void
Data_TraverseTreeAccess::ReplaceLeafNodeLock( const Data_PageNo currentPageNo )
{
    SAPDBERR_ASSERT_STATE( ! m_Page.IsAssigned() );

    if(
        ( m_Current.curr_tree_id.fileType_gg00().includes( ftsPerm_egg00 )) &&
        ( m_Current.currRightBound_bd00 == currentPageNo                  ) &&
        ( m_Current.currIndexNptrs_bd00.np_ptr() != NULL                  )
    )
    {
        const Data_PageNo nextIndexPage = m_Current.currIndexNptrs_bd00.np_ptr()->nd_right();

        ReplaceFirstLevelIndexNodeLock( nextIndexPage );
    }
}

/*---------------------------------------------------------------------------*/

void
Data_TraverseTreeAccess::ReplaceFirstLevelIndexNodeLock( const Data_PageNo nextPageNo )
{
    SAPDBERR_ASSERT_STATE( ! m_Page.IsAssigned() );
    SAPDBERR_ASSERT_STATE( nextPageNo.IsValid() );
    SAPDBERR_ASSERT_STATE( NULL != m_Current.currIndexNptrs_bd00.np_ptr() );
    SAPDBERR_ASSERT_STATE( FIRST_INDEX_LEVEL_BD00 == m_Current.currIndexNptrs_bd00.np_ptr()->nd_level() );

    bd30ReleaseSubTree( m_Current );
    bd30GetSubTree( m_Current, nextPageNo );
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
