/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.wsee;

import java.io.File;

import java.text.MessageFormat;
import org.apache.commons.logging.Log;

import xjavadoc.XClass;
import xjavadoc.XPackage;

import xdoclet.ConfigParamIntrospector;
import xdoclet.XDocletException;
import xdoclet.XmlSubTask;
import xdoclet.tagshandler.PackageTagsHandler;
import xdoclet.util.LogUtil;
import xdoclet.util.Translator;

/**
 * Subtask that generates a service.wsdl service descriptor.
 *
 * @author        Christoph G. Jung (christoph.jung@infor.de)
 * @author        Jason Essington (jasone@greenrivercomputing.com)
 * @created       23.12.03
 * @ant.element   display-name="service.wsdl" name="wsdl" parent="xdoclet.modules.wsee.WseeDocletTask"
 * @version       $Revision: 1.2 $
 */
public class WsdlSubTask extends XmlSubTask
{
    public final static String DEFAULT_WSDL_FILE_PATTERN = "wsdl/{0}.wsdl";
    /**
     * constants
     */
    private static String DEFAULT_TEMPLATE_FILE = "resources/wsdl.xdt";

    private boolean prefixWithPackageStructure = false;

    /**
     * sets template
     */
    public WsdlSubTask()
    {
        setTemplateURL(getClass().getResource(DEFAULT_TEMPLATE_FILE));
    }

    /**
     * Gets the PrefixWithPackageStructure attribute of the TemplateSubTask object
     *
     * @return   The PrefixWithPackageStructure value
     */
    public boolean isPrefixWithPackageStructure()
    {
        return prefixWithPackageStructure;
    }

    /**
     * Indicates whether or not to prefix with package structure.
     *
     * @param prefixWithPackageStructure  The new PrefixWithPackageStructure value
     * @ant.not-required                  No, default is "true"
     */
    public void setPrefixWithPackageStructure(boolean prefixWithPackageStructure)
    {
        this.prefixWithPackageStructure = prefixWithPackageStructure;
    }

    /**
     * run subtask
     *
     * @exception XDocletException
     */
    public void execute() throws XDocletException
    {
        validateOptions();
        startProcess();
    }

    /*
     * validate wsdl file parameter
     * @see xdoclet.SubTask#validateOptions()
     */
    public void validateOptions() throws XDocletException
    {
        Object wsdlFile = getContext().getConfigParam("wsdlFile");

        if (wsdlFile == ConfigParamIntrospector.NULL || "".equals(wsdlFile)) {
            wsdlFile = DEFAULT_WSDL_FILE_PATTERN;
        }
        setDestinationFile((String) wsdlFile);
        super.validateOptions();
    }

    /**
     * Returns class name for the generated file. {0} substituted by wsee.port-component name.
     *
     * @param clazz                 Description of Parameter
     * @return                      The GeneratedClassName value
     * @exception XDocletException  Description of Exception
     */
    protected String getGeneratedFileName(XClass clazz) throws XDocletException
    {
        Log log = LogUtil.getLog(WsdlSubTask.class, "getGeneratedFileName");

        XPackage pak = clazz.getContainingPackage();
        String package_structure = null;

        if (isPrefixWithPackageStructure() == true)
            // This will do package substitution too
            package_structure = PackageTagsHandler.packageNameAsPathFor(pak);
        else
            package_structure = null;

        String packageName = isPackageSubstitutionInheritanceSupported() == true ? package_structure : null;

        String serviceName = getCurrentClass().getDoc().getTagAttributeValue(WseeTagsHandler.PORT_COMPONENT, "name");
        String file = new File(packageName, serviceName).toString();

        String destinationFile = MessageFormat.format(getDestinationFile(), new Object[]{file});

        if (log.isDebugEnabled()) {
            log.debug("clazz.getName()=" + clazz.getName());
            log.debug("clazz.getQualifiedName()=" + clazz.getQualifiedName());
            log.debug("pak=" + pak);
            log.debug("packageName=" + packageName);
            log.debug("serviceName=" + serviceName);
            log.debug("destinationFile=" + destinationFile);
        }
        return destinationFile;
    }

    /**
     * notify start of task
     *
     * @exception XDocletException
     */
    protected void engineStarted() throws XDocletException
    {
        System.out.println(
            Translator.getString(
            XDocletModulesMessages.class,
            XDocletModulesMessages.GENERATING_WSDL_DESCRIPTOR,
            new String[]{getDestinationFile()}));
    }

    /**
     * Describe what the method does
     *
     * @param clazz                 Describe what the parameter does
     * @return                      Describe the return value
     * @exception XDocletException
     */
    protected boolean matchesGenerationRules(XClass clazz) throws XDocletException
    {
        Log log = LogUtil.getLog(WsdlSubTask.class, "matchesGenerationRules");

        if (super.matchesGenerationRules(clazz) == false) {
            log.debug("Skip bean " + clazz.getQualifiedName() + " because super.matchesGenerationRules() returned false.");
            return false;
        }

        // TODO improve this check to make sure that our class is also a service-endpoint (SLSB or servlet)
        boolean isPortComponent = getCurrentClass().getDoc().hasTag(WseeTagsHandler.PORT_COMPONENT, false);

        return isPortComponent;
    }

}
