/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIMMIOMATCHING_HPP
  #define TEPDIMMIOMATCHING_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIMtxDoubleAdpt.hpp"

  #include <TeThreadJobsManager.h>
  #include <TeMutex.h>

  #include <list>

  /*!
      \class TePDIMMIOMatching
      \brief Modified Moravec Interest Operator based image area matching.
      \author Emiliano F. Castejon <castejon@dpi.inpe.br>.
      \ingroup PDIMatchingAlgorithms
     
      \note The required parameters are:
     
      \param input_image1_ptr (TePDITypes::TePDIRasterPtrType) - 
      The input image 1.
      \param input_channel1 ( unsigned int ) - Band to process 
      from input_image1.
     
      \param input_image2_ptr (TePDITypes::TePDIRasterPtrType) - 
      The input image 2.
      \param input_channel2 ( unsigned int ) - Band to process 
      from input_image2.
     
      \param out_tie_points_ptr ( TeSharedPtr< TeCoordPairVect > ) - The 
      output tie- points correlation_method
      where TeCoordPair.pt1 are input_image1 matricial
      indexes and TeCoordPair.pt2 are input_image2 matricial
      indexes.
     
      \note The Optional parameters are:
      
      \param matching_method (FeatMatchingMethod) Features matching method
      (default: TePDIMMIOMatching::NormCrossCorrMethod).    
     
      \param pixel_x_relation (double) - The pixel resolution relation 
      pixel_x_relation = img1_pixel_res_x / img2_pixel_res_x (default=1.0).
     
      \param pixel_y_relation (double) - The pixel resolution relation 
      pixel_y_relation = img1_pixel_res_y / img2_pixel_res_y (default=1.0).   
      
      \param gt_params ( TeGTParams ) - The geometric transformation
      parameters to be used (if not supplied, the dafult TeGTParams
      will be used - affine transformation, max direct mapping
      error = 2 pixels, max inverse mapping error = 2 pixels,
      LWAOutRemotion outliers remotion ).
     
      \param out_gt_params_ptr ( TeGTParams::pointer ) - The output 
      parameters of internally generated geometric transformation 
      (only valid when geometrical filtering is enabled).
     
      \param input_box1 ( TeBox ) - Box (image matrix coords) to process 
      from input_image1 ( the entire image will be used if no box
      was supplied ).
     
      \param input_box2 ( TeBox ) - Box (image matrix coords) to process 
      from input_image2 ( the entire image will be used if no box
      was supplied ).
     
      \param enable_multi_thread (int) - If present (any value) 
      a multi-threaded processing will be used; Some TeDecoders do
      not support multi-thread so use it with cation
      (default: multi-thread disabled).
     
      \param max_tie_points (unsigned int) - The maximum number
      of generated tie-points (default=1000).
     
      \param skip_geom_filter (int) - If present (any value) 
      will disable the geometric filtering
      ( default: geometric filtering enabled ). 
  
      \param enable_upsampled_filter (int) - If present (any value) 
      will enable the upsampled image Moravec repeatability filter
      ( default: upsampled filtering disable - only the downsampled
      image filter will be applied ).  
     
      \param corr_window_width (unsigned int) - The correlation 
      window width used to correlate points between 
      images (Must be an odd number, minimum 13, default: 13).  
  
      \param moravec_window_width (unsigned int) - The Moravec
      window width used to locate canditate tie-points
      (Must be an odd number, minimum 10, default: 11 ).  
      
      \param variance_min_thrs (double) - Variance minimum value threshold( tie-points
      with directional variance values below this threshold will be eliminated ) - Allowed
      range  [0,3] - with higher values more points with low directional variance will be acquired - 
      default:1.
      
      \example TePDIMMIOMatching_test.cpp Shows how to use this class.
  */
  class PDI_DLL TePDIMMIOMatching : public TePDIAlgorithm {
    public :
      
      /*! Features matching methods. */
      enum FeatMatchingMethod
      {
        /*! Features nuclidian distance method */
        EuclidianDistMethod = 1,      
        /*! Features normalized cross-correlation method */
        NormCrossCorrMethod = 2
      };

      TePDIMMIOMatching();

      ~TePDIMMIOMatching();

      // Overloaded
      bool CheckParameters( const TePDIParameters& parameters ) const;

    protected :
      
      /*! Maximas list node type 
         \param unsigned int (first) Column - X.
         \param unsigned int (second) Line - Y.
       */
      typedef std::pair< unsigned int, unsigned int > MaximasListNodeT;      
      
      /*! Maximas list type 
       */
      typedef std::list< MaximasListNodeT > MaximasListT;
      
      /*! Ordered maximas map type 
         \param unsigned int (first) Column - X.
         \param unsigned int (second) Line - Y.
       */
      typedef std::multimap< double, std::pair< unsigned int, unsigned int > > 
        MaximasMapT;      
      
      struct LoadImageParams
      {
        TePDITypes::TePDIRasterPtrType input_image_ptr_;
        unsigned int input_image_channel_;
        TePDIMtxDoubleAdptInt* img_matrix_ptr_;
        TePDIMtxDoubleAdptInt* img_matrix_ds_ptr_;
        TePDIMtxDoubleAdptInt* img_matrix_us_ptr_;
        unsigned int in_box_x_off_;
        unsigned int in_box_y_off_;
        unsigned int in_box_nlines_;
        unsigned int in_box_ncols_;
        bool progress_enabled_;
        TeMutex* glb_mem_lock_ptr_;
        double origImgXRescFact_; // factor = rescaled_orignal_image / original_image
        double origImgYRescFact_; // factor = rescaled_orignal_image / original_image
        double moravecDownSampleFactor_; // factor = rescaled_original_img / ds_image
        double moravecUpSampleFactor_; // factor = rescaled_original_img / us_image
        bool returnValue_; // true if OK.
        bool generateUpsampledImage_;
      };
      
      struct ExtractLocalMaximasParams 
      {
        /*! Input image matrix */
        TePDIMtxDoubleAdptInt* imgMatrixPtr_;
        
        /*! The output list where the found maximas index coords will be 
            appended */
        MaximasListT* outMaximasListPtr_;
        
        unsigned int moravecWindowSize_;
        
        unsigned int maximasNumber_;
        
        double moravecMinVarianceThreshold_;
        
        /*! Number of sub-images to seach (power of 2). */
        unsigned seachSubImagesNmb_; // Number of sub-images to seach.
        
        //Job return value.
        bool returnValue_;
        
        /*! Is progress interface enabled ?? */
        bool progressEnabled_;
        
        /*! Global mutex pointer */
        TeMutex* glbMutexptr_;     
      };
      
      struct GenerateCorrelationFeaturesParams
      {
        /*! Input image matrix */
        TePDIMtxDoubleAdptInt const* imgMatrixPtr_;
        
        /*! Input maximas list */
        MaximasListT const* inMaximasListPtr_;
        
        /*! Output features matrix pointer */
        TePDIMatrix< double >* featMtxPtr_;
        
        /*! Execution return value (true if OK, false on errors).*/
        bool returnValue_;
        
        /* A mutex object pointer to sync large memory allocations */
        TeMutex* glbMemLockPtr_; // To sync large memory allocations
        
        /*! The generated correlation windows width.*/
        unsigned int corrWindowsWidth_;
        
        /*! Is progress interface enabled ?? */
        bool progressEnabled_;
        
        /*! If true, the generated windows will be normalized between -1 
         * and + 1. 
         */
        bool normalizeWindows_;
      };
      
      struct CalcEuclidianDistanceMtxParams
      {
        /*! Image 1 features matrix pointer. */
        TePDIMatrix< double > const* img1FeatMtxPtr_;
        
        /*! Image 2 features matrix pointer. */
        TePDIMatrix< double > const* img2FeatMtxPtr_;

        /*! Output matching matrix. */
        TePDIMatrix<double>* matchMatrixPtr_;
  
        /*! Is progress interface enabled ?? */
        bool progressEnabled_;
        
        /*! Global mutex pointer */
        TeMutex* glbMutexptr_;
  
        /*! Execution return value (true if OK, false on errors).*/
        bool returnValue_;
      };
      
      struct CalcCCorrelationMtxParams
      {
        /*! Image 1 features matrix pointer. */
        TePDIMatrix< double > const* img1FeatMtxPtr_;
        
        /*! Image 2 features matrix pointer. */
        TePDIMatrix< double > const* img2FeatMtxPtr_;

        /*! Output matching matrix. */
        TePDIMatrix<double>* matchMatrixPtr_;
  
        /*! Is progress interface enabled ?? */
        bool progressEnabled_;
        
        /*! Global mutex pointer */
        TeMutex* glbMutexptr_;
  
        /*! Execution return value (true if OK, false on errors).*/
        bool returnValue_;
      };      
      
      /*! Internal threaded jobs manager. */
      TeThreadJobsManager jobsMan_;
      
      /*! A global syncronization mutex. */
      TeMutex globalMutex_;
      
      /*! A pointer to the input image 1. */
      TePDITypes::TePDIRasterPtrType input_image1_ptr_;
      
      /*! Input image 1 channel. */
      unsigned int input_channel1_;
      
      /*! Input image 1 box (indexed). */
      TeBox input_box1_;
      
      /*! A pointer to the input image 2. */
      TePDITypes::TePDIRasterPtrType input_image2_ptr_;
      
      /*! Input image 2 channel. */
      unsigned int input_channel2_;
      
      /*! Input image 2 box (indexed). */
      TeBox input_box2_;
      
      /*! Image 1 features matrix (each matrix line is a stacked version of one 
       * correlation window
       */ 
      TePDIMatrix< double > img1featMtx_;
      
      /*! Image 2 features matrix (each matrix line is a stacked version of one 
       * correlation window
       */ 
      TePDIMatrix< double > img2featMtx_;      

      /*! Image 1 maximas list. */       
      MaximasListT img1_maximas_list_;

      /*! Image 2 maximas list. */       
      MaximasListT img2_maximas_list_;
      
      /*! Feature matching mathod. */
      FeatMatchingMethod matching_method_;
      
      // Overloaded
      bool RunImplementation();

      // Overloaded
      void ResetState( const TePDIParameters& params );
      
      /**
        \brief A job function load raster data into
        a simple matrix.
        \param paramsPtr The parameters needed for the thread 
        execution (a pointer to a LoadImageParams struct).
       */
      static void loadImage( void * paramsPtr ); 
      
      /*! Instantiate an image matrix capable of store pixel data
         of the supplied type.
        \param dataType Pixel type.
        \param matrixPtr A pointer for the new created matrix.
       */
      static void createImgMatrix( TeDataType dataType, 
        TeSharedPtr< TePDIMtxDoubleAdptInt >& matrixPtr );
     
      /*!
        \brief Extract local interest maxima points using a Modified
        Moravec Interest Operator aproach.
        \param paramsPtr The parameters needed for the thread 
        execution (a pointer to a ExtractLocalMaximasParams struct).
       */
      static void extractLocalMaximas( void * paramsPtr );
      
      /*!
        \brief Extract local interest maxima points from an image
        sub-region using a Modified Moravec Interest Operator aproach.
        \param paramsPtr The parameters needed for the thread
        execution (a pointer to a ExtractLocalMaximasParams struct).
        \param xStart Image sub-region initial X coordinate. 
        \param yStart Image sub-region initial y coordinate.
        \param width Image sub-region width.
        \param height Image sub-region height.
        \param progres Progress interface manager reference.
        \param return true if OK, false on errors.
       */      
      static bool extractLocalMaximas( ExtractLocalMaximasParams& params,
        unsigned int xStart, unsigned int yStart, unsigned int width,
        unsigned int height, MaximasMapT& outMaximasMap,
        TePDIPIManager& progress );
      
      /*! 
        \brief Applies a Moravec repeatability constraint over an image
        maximas list.
        \param inputList Original image maximas list.
        \param constraintList Constraint list.
        \param constraintImageWidth Constraint image width.
        \param constraintImageHeight Constraint image height.
        \param moravecRepeatabilityMinTolerance Error tolerance (pixels)
        - constraint image reference.
        \param moravecReSampleFactor Moravec resample factor 
        (factor = inputList_scal / constraintList_scale).
        \param outputList Output maximas list from input list.
       */
      void moravecRepeatability( const MaximasListT& inputList,
        const MaximasListT& constraintList, 
        unsigned int constraintImageWidth,
        unsigned int constraintImageHeight,
        double moravecRepeatabilityMinTolerance, 
        double moravecReSampleFactor, 
        MaximasListT& outputList );      
      
      /*!
        \brief Generate correlation features.
        \param paramsPtr The parameters needed for the thread 
        execution (a pointer to a ExtractLocalMaximasParams struct).
       */
      static void generateCorrelationFeatures( void * paramsPtr );    
      
      /*! 
        \brief Features matching and tie-points generation.
        \param matching_method Features matching method.
        \param img1featMtx Image 1 features matrix.
        \param img1_maximas_list Image 1 maximas list.
        \param img2featMtx Image 2 features matrix.
        \param img2_maximas_list Image 2 maximas list.
        \param tiePointsVec Output tie-points vector.
        \return true if OK, false on errors.
       */      
      bool matchFeatures( FeatMatchingMethod matching_method,
        const TePDIMatrix< double >& img1featMtx,
        const MaximasListT& img1_maximas_list,
        const TePDIMatrix< double >& img2featMtx,
        const MaximasListT& img2_maximas_list,
        TeCoordPairVect& tiePointsVec );
      
      /*!
        \brief Calc euclidian distance matrix for the supplied features.
        \param paramsPtr The parameters needed for the thread 
        execution (a pointer to a CalcEuclidianDistanceMtxParams struct).
       */
      static void calcEuclidianDistanceMtx( void * paramsPtr );   
      
      /*!
        \brief cross-correlation matrix calcule (absolute values).
        \param paramsPtr The parameters needed for the thread 
        execution (a pointer to a CalcInvCCorrelationMtxParams struct).
       */
      static void calcCCorrelationMtx( void * paramsPtr );      
      
      /*!
        \brief Save the output matrix to a geotiff file.
        \param input_matrix Input matrix.
        \param out_file_name Output file name.
        \param maxima_points The maxima points to be
        filled with 255
       */      
      static void matrix2Tiff( 
        const TePDIMtxDoubleAdptInt& input_matrix,
        const std::string& out_file_name,
        const MaximasListT& maxima_points = MaximasListT() );
      
      /*!
        \brief Save the output matrix to a geotiff file.
        \param input_matrix Input matrix.
        \param out_file_name Output file name.
        \param tiepoints The tie oints to be
        filled with 255
        \param usePt1 Use pt1 from tiepoints, otherwise use pt1
       */      
      static void matrix2Tiff( 
        const TePDIMtxDoubleAdptInt& input_matrix,
        const std::string& out_file_name,
        const TeCoordPairVect& tiepoints,
        bool usePt1 );      
      
      /*!
        \brief Save the generated features to tif files.
        \param corr_window_width The correlation windows width.
        \param img_maxima_points The image maxima points
        (these are the correlation matrix centers).
        \param img_features_matrix The output image
        features matrix ( each line is a stacked version
        of one rotated correlation window ).
        \filenameaddon A string to be appended to the
        file name of each feature file.
        \return true if OK, false on errors.
       */          
      static void features2Tiff( 
        unsigned int corr_window_width,
        const MaximasListT& img_maxima_points,
        TePDIMatrix< double >& img_features_matrix,
        const std::string& filenameaddon );      

      /*!
        \brief Input matrix bicubic resample.
        \param inputMatrix Input matrix.
        \param outLines Output number of lines.
        \param outCols Output number of columns.
        \param progressEnabled Progress enabled flag.
        \param globalMutex Global mutex.
        \param outMinValue The allowed output min value.
        \param outMaxValue The allowed output max value.
        \param outputMatrix Output matrix.
        \return true if OK, false on errors.
       */      
      static bool bicubicResampleMatrix( const TePDIMtxDoubleAdptInt& inputMatrix,
        unsigned int outLines, unsigned int outCols, bool progressEnabled,
        TeMutex& globalMutex, double outMinValue, double outMaxValue,
        TePDIMtxDoubleAdptInt& outputMatrix );
      
  };


#endif
