#include "TePDIBatchGeoMosaic.hpp"
#include "TePDIGeoMosaic.hpp"
#include "TePDIUtils.hpp"
#include <TeAgnostic.h>

#include <TeVectorRemap.h>


TePDIBatchGeoMosaic::TePDIBatchGeoMosaic()
{
}


TePDIBatchGeoMosaic::~TePDIBatchGeoMosaic()
{
}


void TePDIBatchGeoMosaic::ResetState( const TePDIParameters& )
{
}


bool TePDIBatchGeoMosaic::CheckParameters( const TePDIParameters& parameters ) const
{
  unsigned int index = 0;

  /* Checking input_rasters */
  
  TePDITypes::TePDIRasterVectorType input_rasters;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_rasters", 
    input_rasters ),
    "Missing parameter: input_rasters" );
  TEAGN_TRUE_OR_RETURN( ( input_rasters.size() > 1 ),
    "Invalid parameter: input_rasters" );
    
  for( index = 0 ; index < input_rasters.size() ; ++index ) {
    TEAGN_TRUE_OR_RETURN( input_rasters[ index ].isActive(),
      "Invalid parameter: input_rasters[" +  Te2String( index ) + 
      "] inactive" );
      
    TEAGN_TRUE_OR_RETURN( input_rasters[ index ]->params().status_ != 
      TeRasterParams::TeNotReady, "Invalid parameter: input_rasters[" +
      Te2String( index ) + "] not ready" );
      
    TEAGN_TRUE_OR_RETURN( ( input_rasters[ index ]->params().projection() != 0 ),
      "Missing input_rasters[" + Te2String( index ) + "] projection" );
      
    TEAGN_TRUE_OR_RETURN( 
      ( input_rasters[ index ]->params().projection()->name() != "NoProjection" ),
      "Invalid input_rasters[" + Te2String( index ) + "] projection" );        
  }
  
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );

  /* Checking for blending type */
  
  std::string blending_type;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "blending_type", 
    blending_type ), "Missing parameter: blending_type" );
    
  /* Checking bands */
  
  std::vector< int > bands;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "bands", 
    bands ), "Missing parameter: bands" );
  TEAGN_TRUE_OR_RETURN( ( bands.size() >= input_rasters.size() ),
    "Invalid parameter: bands" );
  TEAGN_TRUE_OR_RETURN( ( ( bands.size() % input_rasters.size() ) == 0 ),
    "Invalid paramter: bands" );
    
  unsigned int bands_per_raster = bands.size() / input_rasters.size();
    
  for( unsigned int bands_index = 0 ; bands_index < bands.size() ; 
    ++bands_index ) {
    
    unsigned int current_raster_index = (unsigned int) floor( 
      ( (double)bands_index ) / ( (double)bands_per_raster ) );
    TEAGN_TRUE_OR_RETURN( (
      ( bands[ bands_index ] >= 0 ) &&
      ( bands[ bands_index ] < input_rasters[ current_raster_index ]->nBands() 
      ) ), "Invalid parameter: bands[" + Te2String( bands_index ) + "]" );
  }
    
  return true;
}


bool TePDIBatchGeoMosaic::RunImplementation()
{
  TePDITypes::TePDIRasterVectorType input_rasters;
  params_.GetParameter( "input_rasters", input_rasters );

  TePDITypes::TePDIRasterPtrType output_raster;
  params_.GetParameter( "output_raster", output_raster );
  
  std::vector< int > bands;
  params_.GetParameter( "bands", bands );
  
  /* Dumyy value definition */
 
  const bool output_raster_uses_dummy = output_raster->params().useDummy_;
  double output_raster_dummy = 0;
  if( output_raster_uses_dummy ) {
    output_raster_dummy = output_raster->params().dummy_[ 0 ];
  }
  
  bool user_required_dummy = false;
  double user_dummy = 0;
  if( params_.CheckParameter< double >( "dummy_value" ) ) {
    
    params_.GetParameter( "dummy_value", user_dummy );
    user_required_dummy = true;
  }
  
  bool must_use_dummy = false;
  double output_dummy_value = 0;
  if( user_required_dummy ) {
    must_use_dummy = true;
    output_dummy_value = user_dummy;
  } else if( output_raster_uses_dummy ) {
    must_use_dummy = true;
    output_dummy_value = output_raster_dummy;
  }  
  
  /* Building a internal rasters list */
  
  std::list< TePDITypes::TePDIRasterPtrType > rasters_list;
  
  for( unsigned int index = 0 ; index < input_rasters.size() ; ++index ) {
    rasters_list.push_back( input_rasters[ index ] );
  }
  
  /* Processing each pair of rasters */
  
  TePDITypes::TePDIRasterPtrType result_raster; 
    /* The final raster will be pointed by this */
  
  while( rasters_list.size() > 0 ) {
    TePDITypes::TePDIRasterPtrType current_input_raster1;
    TePDITypes::TePDIRasterPtrType current_input_raster2;
    
    std::vector< int > channels1;
    std::vector< int > channels2;
   
    if( rasters_list.size() == input_rasters.size() ) {
      TePDITypes::TePDIRasterPtrType found_raster1;
      TePDITypes::TePDIRasterPtrType found_raster2;
        
      FindClosestRasters( rasters_list, found_raster1, found_raster2 );
        
      current_input_raster1 = found_raster1;
      current_input_raster2 = found_raster2;
      
      findRastersChannels( current_input_raster1, params_, channels1 );
      findRastersChannels( current_input_raster2, params_, channels2 );
      
      rasters_list.remove( found_raster1 );
      rasters_list.remove( found_raster2 );
    } else {
      TePDITypes::TePDIRasterPtrType found_raster;
        
      FindClosestRaster( rasters_list, 
        result_raster->params().box(),
        result_raster->params().projection(),
        found_raster );
        
      current_input_raster1 = result_raster;
      current_input_raster2 = found_raster;
      
      /* building the channels list for the input mosaic */
      
      channels1.clear();
        
      for( int channels1_index = 0 ; 
        channels1_index < result_raster->nBands() ; ++channels1_index ) {
        
        channels1.push_back( channels1_index );
      }
      
      /* Retriving the channels vector for the new input rasters */      
      
      findRastersChannels( current_input_raster2, params_, channels2 );
      
      /* done */
      
      rasters_list.remove( found_raster );
    }
    
    /* Guessing the output data type */
    
    TeDataType output_datatype = TeUNSIGNEDCHAR;
    {
      double r1_min = 0;
      double r1_max = 0;
      double r2_min = 0;
      double r2_max = 0;

      TePDIUtils::TeGetRasterMinMaxBounds( current_input_raster1, 0, r1_min,
        r1_max );
      TePDIUtils::TeGetRasterMinMaxBounds( current_input_raster2, 0, r2_min,
        r2_max );
        
      if( ( r2_min < r1_min ) || ( r2_max > r1_max ) ) {
        output_datatype = current_input_raster2->params().dataType_[ 0 ];
      } else {
        output_datatype = current_input_raster1->params().dataType_[ 0 ];
      }
    }
    
    /* Setting the output raster */
    
    TePDITypes::TePDIRasterPtrType current_output_raster;
    
    if( rasters_list.size() == 0 ) {
      /* writing directly to the output raster */
      
      current_output_raster = output_raster;
    } else {
      /* writing to a internal buffered raster */
      
      TeRasterParams current_output_raster_params;
      current_output_raster_params.nBands( 1 );
      current_output_raster_params.nlines_ = 10;
      current_output_raster_params.ncols_ = 10;
      current_output_raster_params.dataType_[ 0 ] = output_datatype;
      current_output_raster_params.setDummy( output_dummy_value, -1 );
      if( must_use_dummy ) {
        current_output_raster_params.useDummy_ = true;
      } else {
        current_output_raster_params.useDummy_ = false;
      }
      current_output_raster_params.setPhotometric( TeRasterParams::TeMultiBand,
        -1 );
        
      TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( 
        current_output_raster, current_output_raster_params, 
        TePDIUtils::TePDIUtilsAutoMemPol ), 
        "Unable to allocate temporary mem raster" );
    }
    
    /* Building geomosaic parameters */
    
    TePDIParameters geo_mos_params = params_;
    geo_mos_params.SetParameter( "mosaic_type", std::string( "geo_mosaic" ) );  
    geo_mos_params.SetParameter( "input_raster1", current_input_raster1 );
    geo_mos_params.SetParameter( "input_raster2", current_input_raster2 );
    geo_mos_params.SetParameter( "output_raster", current_output_raster );
    geo_mos_params.SetParameter( "keep_best_res", (int)1 );
    geo_mos_params.SetParameter( "channels1", channels1 );
    geo_mos_params.SetParameter( "channels2", channels2 );
          
    TePDIGeoMosaic geo_mos_instance;
    
    TEAGN_TRUE_OR_RETURN( geo_mos_instance.Reset( geo_mos_params ), 
      "Geo-mosaic instance reset error" );
      
    TEAGN_TRUE_OR_RETURN( geo_mos_instance.Apply(),
      "Unable to apply geographic mosaic strategy" );
      
    result_raster = current_output_raster;
  } /* while */        
  
  return true;
}


void TePDIBatchGeoMosaic::FindClosestRaster( 
  std::list< TePDITypes::TePDIRasterPtrType >& input_rasters,
  const TeBox& reference_raster_box,
  TeProjection* reference_raster_projection,
  TePDITypes::TePDIRasterPtrType& closest_raster )
{
  TEAGN_TRUE_OR_THROW( ( input_rasters.size() > 0 ), 
    "input_rasters.size() < 1" );
  TEAGN_TRUE_OR_THROW( ( reference_raster_projection != 0 ),
    "Invalid reference_raster_projection" );
  
  std::list< TePDITypes::TePDIRasterPtrType >::iterator it = 
    input_rasters.begin();
  
  double best_distance = 0;
  bool first_best_distance_found = false;
  
  double current_distance = 0;
  TeBox current_box;
  TeBox remapped_box;
  
  while( it != input_rasters.end() ) {
    current_box = (*it)->params().box();
    remapped_box = TeRemapBox( current_box,
      (*it)->params().projection(),
      reference_raster_projection );
    
    if( first_best_distance_found ) {
      current_distance = 
        TeDistance( remapped_box.center(), 
        reference_raster_box.center() );
          
      if( current_distance < best_distance ) {
        best_distance = current_distance;
        closest_raster = *it;
      }
    } else {
      best_distance = 
        TeDistance( remapped_box.center(), 
        reference_raster_box.center() );
      
      first_best_distance_found = true;
      closest_raster = *it;
    }  
    
    ++it;
  }
  
  TEAGN_DEBUG_CONDITION( first_best_distance_found, "Best distance not found" );
}

void TePDIBatchGeoMosaic::FindClosestRasters( 
  std::list< TePDITypes::TePDIRasterPtrType >& input_rasters,
  TePDITypes::TePDIRasterPtrType& raster1,
  TePDITypes::TePDIRasterPtrType& raster2 )
{
  TEAGN_TRUE_OR_THROW( ( input_rasters.size() > 1 ), 
    "input_rasters.size() < 2" );
    
  double best_distance = 0;
  bool best_distance_found = false;
  double current_distance = 0;
  TeBox current_box;
  TeBox closest_raster_box;
  TeBox closest_raster_box_reproj;
  TePDITypes::TePDIRasterPtrType current_raster;
  TePDITypes::TePDIRasterPtrType closest_raster;
  std::list< TePDITypes::TePDIRasterPtrType > input_rasters_copy;
  std::list< TePDITypes::TePDIRasterPtrType >::iterator input_rasters_it = 
    input_rasters.begin();
  
  while( input_rasters_it != input_rasters.end() ) {
    current_raster = *input_rasters_it;
    current_box = current_raster->params().box();
    
    input_rasters_copy = input_rasters;
    input_rasters_copy.remove( current_raster );
    
    FindClosestRaster( input_rasters_copy, current_box, 
      current_raster->params().projection(),
      closest_raster );
      
    closest_raster_box = closest_raster->params().box();  
    closest_raster_box_reproj = TeRemapBox( closest_raster_box,
      closest_raster->projection(), current_raster->projection() );
    
    current_distance = TeDistance( current_box.center(), 
      closest_raster_box_reproj.center() );
      
    if( best_distance_found ) {
      if( current_distance < best_distance ) {
        best_distance = current_distance;
        raster1 = current_raster;
        raster2 = closest_raster;
      }
    } else {
      best_distance = current_distance;
      raster1 = current_raster;
      raster2 = closest_raster;
    }
    
    ++input_rasters_it;
  }  
}


void TePDIBatchGeoMosaic::findRastersChannels( 
  const TePDITypes::TePDIRasterPtrType& raster_ptr, 
  const TePDIParameters& params,
  std::vector< int >& channels )
{
  channels.clear();
  
  std::vector< int > bands;
  TEAGN_TRUE_OR_THROW( params.GetParameter( "bands", bands ),
    "Unable to extract parameter: bands" );
    
  TePDITypes::TePDIRasterVectorType input_rasters;
  TEAGN_TRUE_OR_THROW( params.GetParameter( "input_rasters", input_rasters ),
    "Unable to extract parameter: input_rasters" );
    
  unsigned int bands_per_raster = bands.size() / input_rasters.size();
  
  for( unsigned int rasters_index = 0 ; rasters_index < input_rasters.size() ;
    ++rasters_index ) {
    if( input_rasters[ rasters_index ] == raster_ptr ) {
      unsigned int bands_start_index = ( rasters_index * bands_per_raster );
      
      for( unsigned int offset = 0 ; offset < bands_per_raster ; ++offset ) {
        channels.push_back(  bands[ bands_start_index + offset ] );
      }
      
      return;
    }
  }
  
  TEAGN_LOG_AND_THROW( "Unable to find the required rasters bands" );
}
