/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifdef WIN32 
#pragma warning ( disable: 4786 ) 
#endif

#include "TeRasterParams.h"
#include "TeUtils.h"

// copy constructor
TeRasterParams::TeRasterParams(const TeRasterParams& other)
{
	resolution_ = other.resolution_;
	subBand_  = other.subBand_;
	nlines_		= other.nlines_;
	ncols_		= other.ncols_;
	nbands_		= other.nbands_;
	lutName_	= other.lutName_;
	swap_		= other.swap_;
	useDummy_	= other.useDummy_;

	nbitsperPixel_.resize(nbands_);
	vmin_.resize(nbands_);
	vmax_.resize(nbands_);
	bands_.resize(nbands_);
	photometric_.resize(nbands_);
	dataType_.resize(nbands_);
	dummy_.resize(nbands_);
	compression_.resize(nbands_);

	int i;
	for (i=0; i<nbands_; i++)
	{
		nbitsperPixel_[i] = other.nbitsperPixel_[i];
		vmin_[i] = other.vmin_[i];
		vmax_[i] = other.vmax_[i];
		bands_[i] = other.bands_[i];
		photometric_[i] = other.photometric_[i];
		dataType_[i] = other.dataType_[i];
		dummy_[i] = other.dummy_[i];
		compression_[i] = other.compression_[i];
	}

	resx_ = other.resx_;
	resy_ = other.resy_;
	fileName_ = other.fileName_;
	mode_ = other.mode_;
	status_ = other.status_;
	offset_ = other.offset_;
	tiling_type_ = other.tiling_type_;
	blockHeight_ = other.blockHeight_;
	blockWidth_ = other.blockWidth_;
	decoderIdentifier_ = other.decoderIdentifier_;
	nTilesInMemory_ = other.nTilesInMemory_;
	if (other.projection_)
		projection_ = TeProjectionFactory::make(other.projection_->params());
	else
		projection_=0;
	box_ = other.box_;
	database_ = other.database_;
	interleaving_ = other.interleaving_;

	lutr_ = other.lutr_;
	lutg_ = other.lutg_;
	lutb_ = other.lutb_;
	dyJ_ = other.dyJ_;
	dxI_ = other.dxI_;
	dyI_ = other.dyI_;
	dxJ_ = other.dxJ_;
	x0_ = other.x0_;
	y0_ = other.y0_;
}

// pperator =
TeRasterParams& 
TeRasterParams::operator=(const TeRasterParams& rhs)
{
	if ( this != &rhs )
	{
		resolution_ = rhs.resolution_;
		subBand_  = rhs.subBand_;
		nlines_		= rhs.nlines_;
		ncols_		= rhs.ncols_;
		nbands_		= rhs.nbands_;
		swap_		= rhs.swap_;
		useDummy_	= rhs.useDummy_;
		lutName_	= rhs.lutName_;

		nbitsperPixel_.resize(nbands_);
		vmin_.resize(nbands_);
		vmax_.resize(nbands_);
		bands_.resize(nbands_);
		photometric_.resize(nbands_);
		dataType_.resize(nbands_);
		dummy_.resize(nbands_);
		compression_.resize(nbands_);

		for (int i=0; i<nbands_; i++)
		{
			nbitsperPixel_[i] = rhs.nbitsperPixel_[i];
			vmin_[i] = rhs.vmin_[i];
			vmax_[i] = rhs.vmax_[i];
			bands_[i] = rhs.bands_[i];
			photometric_[i] = rhs.photometric_[i];
			dataType_[i] = rhs.dataType_[i];
			dummy_[i] = rhs.dummy_[i];
			compression_[i] = rhs.compression_[i];
		}

		resx_ = rhs.resx_;
		resy_ = rhs.resy_;
		fileName_ = rhs.fileName_;
		mode_ = rhs.mode_;
		status_ = rhs.status_;
		offset_ = rhs.offset_;
		tiling_type_ = rhs.tiling_type_;
		blockHeight_ = rhs.blockHeight_;
		blockWidth_ = rhs.blockWidth_;
		decoderIdentifier_ = rhs.decoderIdentifier_;
		nTilesInMemory_ = rhs.nTilesInMemory_;
		if (projection_)
		{
			delete projection_;
			projection_ = 0;
		}
		if (rhs.projection_)
			projection_ = TeProjectionFactory::make(rhs.projection_->params());
		box_ = rhs.box_;
		database_ = rhs.database_;			
		interleaving_ = rhs.interleaving_;

		lutr_ = rhs.lutr_;
		lutg_ = rhs.lutg_;
		lutb_ = rhs.lutb_;
		dyJ_ = rhs.dyJ_;
		dxI_ = rhs.dxI_;
		x0_ = rhs.x0_;
		y0_ = rhs.y0_;
		dyI_ =  rhs.dyI_;
		dxJ_ =  rhs.dxJ_;
	}
	return *this;
}

void
TeRasterParams::resizeToTiling(TeBox& bb, int bWidth, int bHeight)
{
	blockHeight_ = bHeight;
	blockWidth_ = bWidth;

	double bXSize = bWidth*resx_;
	double bYSize = bHeight*resy_;
	
	TeBox baux = adjustToCut(bb, bXSize, bYSize);
	this->boundingBoxResolution(baux.x1_,baux.y1_,baux.x2_,baux.y2_,resx_,resy_);
}

//! Calculates the *bounding box* when box coordinates are relative to the center of the pixel
TeBox	
TeRasterParams::boundingBox ()
{ return TeBox (box_.x1()-resx_/2.,
				box_.y1()-resy_/2.,
				box_.x2()+resx_/2.,
				box_.y2()+resy_/2.);
}

void	
TeRasterParams::boundingBoxResolution (double x1, double y1, double x2, double y2, 
									   double resx, double resy, TeBox::TeBoxCorner lock)
{
	resx_ = resx;
	resy_ = resy;
	ncols_  = (int)((x2-x1)/resx_+0.5);
	nlines_ = (int)((y2-y1)/resy_+0.5);
	
	if (lock == TeBox::TeUPPERLEFT)
		box_ = TeBox(x1+0.5*resx_, y2-nlines_*resy_+0.5*resy_,
		             x1+ncols_*resx_-0.5*resx_, y2-0.5*resy_);

	else if (lock == TeBox::TeLOWERLEFT)
		box_ = TeBox(x1+0.5*resx_, y1+0.5*resy_,
		             x1+ncols_*resx_-0.5*resx_, y1+nlines_*resy_-0.5*resy_);

	else if (lock == TeBox::TeUPPERRIGHT)
		box_ = TeBox(x2-ncols_*resx_+0.5*resx_, y2-nlines_*resy_+0.5*resy_,
			         x2-0.5*resx_,y2-0.5*resy_);

	else		// TeLOWERRIGHT
		box_ = TeBox(x2-ncols_*resx_+0.5*resx_,y1+0.5*resy_,
		             x2-0.5*resx_,y1+nlines_*resy_-0.5*resy_);
}

void	
TeRasterParams::boundingBoxLinesColumns (double x1, double y1, double x2, double y2, 
										 int nlines, int ncols, TeBox::TeBoxCorner lock)
{
	nlines_ = nlines;
	ncols_ = ncols;
	
	resx_ = (x2-x1)/ncols_;
	resy_ = (y2-y1)/nlines_;

	if (lock == TeBox::TeUPPERLEFT)
		box_ = TeBox(x1+0.5*resx_, y2-nlines_*resy_+0.5*resy_,
		             x1+ncols_*resx_-0.5*resx_, y2-0.5*resy_);

	else if (lock == TeBox::TeLOWERLEFT)
		box_ = TeBox(x1+0.5*resx_, y1+0.5*resy_,
		             x1+ncols_*resx_-0.5*resx_, y1+nlines_*resy_-0.5*resy_);

	else if (lock == TeBox::TeUPPERRIGHT)
		box_ = TeBox(x2-ncols_*resx_+0.5*resx_, y2-nlines_*resy_+0.5*resy_,
			         x2-0.5*resx_,y2-0.5*resy_);

	else		// TeLOWERRIGHT
		box_ = TeBox(x2-ncols_*resx_+0.5*resx_,y1+0.5*resy_,
		             x2-0.5*resx_,y1+nlines_*resy_-0.5*resy_);
}

void	
TeRasterParams::boxResolution (double x1, double y1, double x2, double y2, 
							   double resx, double resy, TeBox::TeBoxCorner lock)
{
	resx_ = resx;
	resy_ = resy;

	ncols_  = (int)((x2-x1)/resx_+0.45)+1;
	nlines_ = (int)((y2-y1)/resy_+0.45)+1;

	if (lock == TeBox::TeUPPERLEFT)
		box_ = TeBox(x1, y2-(nlines_-1)*resy_,x1+(ncols_-1)*resx_,y2);

	else if (lock == TeBox::TeLOWERLEFT)
		box_ = TeBox(x1, y1,x1+(ncols_-1)*resx_,y1+(nlines_-1)*resy_);

	else if (lock == TeBox::TeUPPERRIGHT)
		box_ = TeBox(x2-(ncols_-1)*resx_, y2-(nlines_-1)*resy_,x2,y2);

	else		// TeLOWERRIGHT
		box_ = TeBox(x2-(ncols_-1)*resx_,y1,x2,y1+(nlines_-1)*resy_);
}

void	
TeRasterParams::boxLinesColumns (double x1, double y1, double x2, double y2, 
								 int nlines, int ncols, TeBox::TeBoxCorner lock)
{
	nlines_ = nlines;
	ncols_ = ncols;

	resx_ = (x2-x1)/(ncols_-1);
	resy_ = (y2-y1)/(nlines_-1);

	if (lock == TeBox::TeUPPERLEFT)
		box_ = TeBox(x1, y2-(nlines_-1)*resy_,x1+(ncols_-1)*resx_,y2);

	else if (lock == TeBox::TeLOWERLEFT)
		box_ = TeBox(x1, y1,x1+(ncols_-1)*resx_,y1+(nlines_-1)*resy_);

	else if (lock == TeBox::TeUPPERRIGHT)
		box_ = TeBox(x2-(ncols_-1)*resx_, y2-(nlines_-1)*resy_,x2,y2);

	else		// TeLOWERRIGHT
		box_ = TeBox(x2-(ncols_-1)*resx_,y1,x2,y1+(nlines_-1)*resy_);
}

void	
TeRasterParams::topLeftResolutionSize (double left, double top, 
									   double resx, double resy, 
									   int ncol, int nlin, 
									   bool coordIsCentrePixel)
{
	resx_ = resx;
	resy_ = resy;
	ncols_  = ncol;
	nlines_ = nlin;
	
	if (coordIsCentrePixel)
		box_ = TeBox(left,top-(nlin-1)*resy_,left+(ncol-1)*resx_,top);
	else
	{
		TeBox bb(left,top-nlines_*resy_,left+ncols_*resx_,top);
		box_ = TeBox(bb.x1()+resx_/2.,
					 bb.y1()+resy_/2.,
					 bb.x2()-resx_/2.,
					 bb.y2()-resy_/2.);
	}
}

void	
TeRasterParams::lowerLeftResolutionSize (double left, double lower, 
										 double resx, double resy, 
										 int ncol, int nlin, 
										 bool coordIsCentrePixel)
{
	resx_ = resx;
	resy_ = resy;
	ncols_  = ncol;
	nlines_ = nlin;
	
	if (coordIsCentrePixel)
		box_ = TeBox(left,lower,left+(ncol-1)*resx_,lower+(nlin-1)*resy_);
	else
	{
		TeBox bb(left,lower,left+ncols_*resx_,lower+nlines_*resy_);
		box_ = TeBox(bb.x1()+resx_/2.,
					 bb.y1()+resy_/2.,
					 bb.x2()-resx_/2.,
					 bb.y2()-resy_/2.);
	}
}

void 
TeRasterParams::nBands(int n)
{
	if (nbands_ != n)
	{
		if (n < 1)
			n = 1;
		nbands_ = n;
		nbitsperPixel_.resize(nbands_);
		vmin_.resize(nbands_);
		vmax_.resize(nbands_);
		bands_.resize(nbands_);
		photometric_.resize(nbands_);
		dataType_.resize(nbands_);
		dummy_.resize(nbands_);
		compression_.resize(nbands_);
		double d = dummy_[0];
		for (int i = 0; i<n; i++)
		{
			bands_[i] = i;
			vmin_[i] = TeMAXFLOAT;
			vmax_[i] = -TeMAXFLOAT;
			dataType_[i] = TeUNSIGNEDCHAR;
			photometric_[i] = TeRASTERUNKNOWN;
			dummy_[i] = d;
		}
	}
}

void 
TeRasterParams::projection(TeProjection* proj)
{
	if (projection_)
	{
		delete projection_;
		projection_ = 0;
	}
	if (proj)
		projection_ = TeProjectionFactory::make(proj->params());
}

void 
TeRasterParams::setDataType(TeDataType type, int band)
{
	int nb = 8;
	switch (type) {
	case TeUNSIGNEDCHAR:
		nb = sizeof(unsigned char)*8;
		break;
	case TeCHAR :
		nb = sizeof(char)*8;
		break;
	case TeUNSIGNEDSHORT:
		nb = sizeof(unsigned short)*8;
		break;
	case (TeSHORT):
		nb = sizeof(short)*8;
		break;
	case (TeUNSIGNEDLONG):
		nb = sizeof(unsigned long)*8;
		break;
	case (TeLONG):
		nb = sizeof(long)*8;
		break;
	case (TeFLOAT):
		nb = sizeof(float)*8;
		break;
	case (TeDOUBLE):
		nb = sizeof(double)*8;
        break;
    default:
        break;
	}
	if (band<0 || band>= (int)dataType_.size())
		for (unsigned int i=0; i<dataType_.size(); dataType_[i]=type,nbitsperPixel_[i]=nb,i++);
	else
	{
		dataType_[band]=type;
		nbitsperPixel_[band]=nb;
	}
}

int TeRasterParams::elementSize(int band)
{
	switch (dataType_[band]) {
	case TeUNSIGNEDCHAR:
		return (sizeof(unsigned char));
	case TeCHAR :
		return (sizeof(char));
	case TeUNSIGNEDSHORT:
		return (sizeof(unsigned short));
	case (TeSHORT):
		return (sizeof(short));
	case (TeUNSIGNEDLONG):
		return (sizeof(unsigned long));
	case (TeLONG):
		return (sizeof(long));
	case (TeFLOAT):
		return (sizeof(float));
	default:
		return (sizeof(double));
	}
}


void 
TeRasterParams::setPhotometric(short photom, int band)
{
	if (band<0 || band>= (int)photometric_.size())
		for (unsigned int i=0; i<photometric_.size(); photometric_[i]=photom,i++);
	else
		photometric_[band]=photom;
}

void 
TeRasterParams::setCompressionMode(TeCompressionMode cmode, int band)
{
	if (band<0 || band>= (int)compression_.size())
		for (unsigned int i=0; i<compression_.size(); compression_[i]=cmode,i++);
	else
		compression_[band] = cmode;
}

void 
TeRasterParams::setDummy(double dummy, int band)
{
	if (band<0 || band>= (int)dummy_.size())
		for (unsigned int i=0; i<dummy_.size(); dummy_[i]=dummy,i++);
	else
		dummy_[band] = dummy;
	this->useDummy_ = true;
}

TeCoord2D 
TeRasterParams::coord2Index (TeCoord2D& pt)
{
	double i, j;
	if (dxI_ != 0 || dyJ_ != 0 || dxJ_ != 0 || dyI_ != 0)
	{
		double x = pt.x();
		double y = pt.y();
		
		i = (dyJ_*(x-x0_)-dxJ_*(y-y0_))/(dxI_*dyJ_-dxJ_*dyI_);
		j = (x-x0_-i*dxI_)/dxJ_;
		return TeCoord2D (j,i);
	}
	else
	{
		i = (pt.x() - box_.x1()) / resx_;
		j = (box_.y2() - pt.y()) / resy_;
		return TeCoord2D (i,j);
	}
}

TeCoord2D 
TeRasterParams::index2Coord (TeCoord2D& pt)
{
	double x,y;
	double i = pt.y();
	double j = pt.x();
		
	if (dxI_ != 0 || dyJ_ != 0 || dxJ_ != 0 || dyI_ != 0)
	{
		// X= x0 + J*Dx_j + I*Dx_i
		// Y= Y0 + J*Dy_j + I*Dy_i 

		x = x0_ + i*dxI_ + j*dxJ_;
		y = y0_ + i*dyI_ + j*dyJ_;
	}
	else
	{
		x = box_.x1() + pt.x() * resx_;
		y = box_.y2() - pt.y() * resy_;
	}
	return TeCoord2D (x,y);
}

