/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeDecoderMemory.h"

TeDecoderMemory::~TeDecoderMemory ()
{
	if (data_ != 0)
		clear();
}

void*	
TeDecoderMemory::data (int band)
{
	if (band==0)
		return data_;
	if (params_.interleaving_ != TePerBand)
		return 0;

	void* pt = 0;
	switch (params_.dataType_[band]) {
	case TeUNSIGNEDCHAR:
		pt =  reinterpret_cast<unsigned char*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case TeCHAR :
		pt =  reinterpret_cast<char*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case TeUNSIGNEDSHORT:
		pt = reinterpret_cast<unsigned short*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case (TeSHORT):
		pt = reinterpret_cast<short*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case (TeUNSIGNEDLONG):
		pt = reinterpret_cast<unsigned long*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case (TeLONG):
		pt = reinterpret_cast<long*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case (TeFLOAT):
		pt = reinterpret_cast<float*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	case (TeDOUBLE):
		pt = reinterpret_cast<double*>(data_) + (params_.ncols_ * params_.nlines_*band);
		break;
	default:
		break;
	}
	return (void*) pt;
}

void 
TeDecoderMemory::init()
{
	params_.status_= TeNOTREADY;
	if ( !data_ ||	nelems_ != (params_.ncols_ * params_.nlines_ * params_.nBands()))
	{
		clear();
		nelems_ = params_.ncols_ * params_.nlines_ * params_.nBands();
		if (!allocateMemory() || !resetMemory())
			return;
	}

	if (params_.mode_ == 'c' || params_.mode_ == 'w')	// creating a new file
	{
		if (resetMemory())
			params_.status_ = TeREADYTOWRITE;
	}
	else if (params_.mode_ == 'r')
		params_.status_ = TeREADYTOREAD;		
}


bool 
TeDecoderMemory::allocateMemory()
{
	// Allocate necessary memory
	switch (params_.dataType_[0]) {
	case (TeUNSIGNEDCHAR):
		data_ = new unsigned char [nelems_];
		if (!data_)
			return false;
		break;
	case (TeCHAR) :
		data_ = new char [nelems_];
		if (!data_)
			return false;
		break;
	case (TeUNSIGNEDSHORT):
		data_ = new unsigned short [nelems_];
		if (!data_)
			return false;
		break;
	case (TeSHORT):
		data_ = new short [nelems_];
		if (!data_)
			return false;
		break;
	case (TeUNSIGNEDLONG):
		data_ = new unsigned long [nelems_];
		if (!data_)
			return false;
		break;
	case (TeLONG):
		data_ = new long [nelems_];
		if (!data_)
			return false;
		break;
	case (TeFLOAT):
		data_ = new float [nelems_];
		if (!data_)
			return false;
		break;
	case (TeDOUBLE):
		data_ = new double [nelems_];
		if (!data_)
			return false;
		break;
	default:
		break;
	}
	if (data_)
		return true;
	else 
		return false;
}


bool 
TeDecoderMemory::resetMemory()
{
	if (!data_)
		return false;

	int i;
	switch (params_.dataType_[0]) {
	case (TeUNSIGNEDCHAR):
		for(i=0; i<nelems_; reinterpret_cast<unsigned char*>(data_)[i]=static_cast<unsigned char>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(unsigned char);
		break;
	case (TeCHAR) :
		for(i=0; i<nelems_; reinterpret_cast<char*>(data_)[i]=static_cast<char>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(char);
		break;
	case (TeUNSIGNEDSHORT):
		for(i=0; i<nelems_; reinterpret_cast<unsigned short*>(data_)[i]=static_cast<unsigned short>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(unsigned short);
		break;
	case (TeSHORT):
		for(i=0; i<nelems_; reinterpret_cast<short*>(data_)[i]=static_cast<short>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(short);
		break;
	case (TeUNSIGNEDLONG):
		for(i=0; i<nelems_; reinterpret_cast<unsigned long*>(data_)[i]=static_cast<unsigned long>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(unsigned long);
		break;
	case (TeLONG):
		for(i=0; i<nelems_; reinterpret_cast<long*>(data_)[i]=static_cast<long>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(long);
		break;
	case (TeFLOAT):
		for(i=0; i<nelems_; reinterpret_cast<float*>(data_)[i]=static_cast<float>(params_.dummy_[0]),++i);
		size_ = nelems_ * sizeof(float);
		break;
	case (TeDOUBLE):
		for(i=0; i<nelems_; reinterpret_cast<double*>(data_)[i]=params_.dummy_[0],++i);
		size_ = nelems_ * sizeof(double);
		break;
	default:
		break;
	}
	return true;
}

bool
TeDecoderMemory::clear()
{
	if (data_ == 0)
	    return true;
	switch (params_.dataType_[0])
	{
	case (TeUNSIGNEDCHAR):
		delete [](unsigned char*)data_;
		break;
	case (TeCHAR) :
		delete (char*)data_;
		break;
	case (TeUNSIGNEDSHORT):
		delete [](unsigned short*)data_;
		break;
	case (TeSHORT):
		delete (short*) data_;
		break;
	case (TeUNSIGNEDLONG):
		delete [](unsigned long*)data_;
		break;
	case (TeLONG):
		delete [](long*) data_;
		break;
	case (TeFLOAT):
		delete [](float*)data_;
		break;
	case (TeDOUBLE):
		delete [](double*) data_;
		break;
	default:
		break;
	}
	data_ = 0;
	params_.status_ = TeNOTREADY;
	return true;
}

bool 
TeDecoderMemory::getElement (int col,int lin, double& val,int band)
{
	unsigned int position = 0;

	switch (params_.interleaving_)
	{
	case TePerPixel:
		position = params_.nBands()*(params_.ncols_*lin+col)+band;
		break;
	case TePerLine:
		position = (params_.nBands()*params_.ncols_*lin)+band*params_.ncols_+col;
		break;
	case TePerBand:
		position = band*(params_.ncols_*params_.nlines_)+(params_.ncols_*lin+col);
		break;
	}
	switch (params_.dataType_[0]) 
	{
	case (TeUNSIGNEDCHAR):
		val = reinterpret_cast<unsigned char*>(data_)[position];
		break;
	case (TeCHAR) :
		val = reinterpret_cast<char*>(data_)[position];
		break;
	case (TeUNSIGNEDSHORT):
		val = reinterpret_cast<unsigned short*>(data_)[position];
		break;
	case (TeSHORT):
		val = reinterpret_cast<short*>(data_)[position];
		break;
	case (TeUNSIGNEDLONG):
		val = reinterpret_cast<unsigned long*>(data_)[position];
		break;
	case (TeLONG):
		val = reinterpret_cast<long*>(data_)[position];
		break;
	case (TeFLOAT):
		val = reinterpret_cast<float*>(data_)[position];
		break;
	case (TeDOUBLE):
		val = reinterpret_cast<double*>(data_)[position];
		break;
	default:
		break;
	}
	return true;
}

bool 
TeDecoderMemory::setElement (int col, int lin, double val,int band)
{
	unsigned int position = 0;
	switch (params_.interleaving_)
	{
	case TePerPixel:
		position = params_.nBands()*(params_.ncols_*lin+col)+band;
		break;
	case TePerLine:
		position = (params_.nBands()*params_.ncols_*lin)+band*params_.ncols_+col;
		break;
	case TePerBand:
		position = band*(params_.ncols_*params_.nlines_)+(params_.ncols_*lin+col);
		break;
	}

	switch (params_.dataType_[0]) 
	{
	case (TeUNSIGNEDCHAR):
		reinterpret_cast<unsigned char*>(data_)[position] = static_cast<unsigned char>(val);
		break;
	
	case (TeCHAR) :
		reinterpret_cast<char*>(data_)[position] = static_cast<char>(val);
		break;
	
	case (TeUNSIGNEDSHORT):
		reinterpret_cast<unsigned short*>(data_)[position] = static_cast<unsigned short>(val);
		break;
	
	case (TeSHORT):
		reinterpret_cast<short*>(data_)[position] = static_cast<short>(val);
		break;
	
	case (TeUNSIGNEDLONG):
		reinterpret_cast<unsigned long*>(data_)[position] = static_cast<unsigned long>(val);
		break;
	
	case (TeLONG):
		reinterpret_cast<long*>(data_)[position] = static_cast<long>(val);
		break;
	
	case (TeFLOAT):
		reinterpret_cast<float*>(data_)[position] = static_cast<float>(val);
		break;
	
	case (TeDOUBLE):
		reinterpret_cast<double*>(data_)[position] = val;

	default:
		break;
	}
	isModified_ = true;
	return true;
}

