/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

/**
 *   This file contains the class of structures and definitions
 *   to deal with parameters used by all PDI classes.
 */

#ifndef TEPDIPARAMETERS_HPP
  #define TEPDIPARAMETERS_HPP

  #include "TePDIBox.hpp"
  #include "TePDITypes.hpp"
  #include "TePDIRgbPalette.hpp"
  #include "TePDIFilterMask.hpp"

  #include "TePDISharedPtr.cpp"

  #include <TeRaster.h>

  #include <map>
  #include <string>
  #include <vector>

  class TePDIParameters {
    public :
      typedef TePDISharedPtr< TePDIParameters > pointer;
      typedef const TePDISharedPtr< TePDIParameters > const_pointer;

      /**
       * Suported parameters.
       */
      enum TePDIParamTypes{
        TePDIIntParam = 1,
        TePDIIntVectorParam = 2,
        TePDIRasterParam = 3,
        TePDIDoubleParam = 4,
        TePDIStringParam = 5,
        TePDIBoxParam = 6,
        TePDIRgbPaletteParam = 7,
        TePDIFilterMaskParam = 8
      };

      /**
       * Default Constructor
       */
      TePDIParameters();

      /**
       *  Default Destructor
       */
      ~TePDIParameters();

      /**
       * Sets a TeRaster parameter reference.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         const TePDITypes::TePDIRasterPtrType& data );

      /**
       * Gets a TeRaster parameter reference.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was not found or error.
       */
      bool GetParameter( const std::string& pname,
                         TePDITypes::TePDIRasterPtrType& data ) const;

      /**
       * Sets a TePDIPalette parameter reference.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         TePDIRgbPalette::const_pointer& data );

      /**
       * Gets a TePDIPalette parameter reference.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was not found or error.
       */
      bool GetParameter( const std::string& pname,
                         TePDIRgbPalette::pointer& data ) const;

      /**
       * Sets a Integer parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         int data );

      /**
       * Gets a Integer parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         int& data ) const;

      /**
       * Sets a Integer vector parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         std::vector< int >& data );

      /**
       * Gets a Integer vector parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         std::vector< int >& data ) const;

      /**
       * Sets a String parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         const std::string& data );

      /**
       * Gets a String parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         std::string& data ) const;

      /**
       * Sets a Double parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         double data );

      /**
       * Gets a Double parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         double& data ) const;

      /**
       * Sets a PDIBox parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         const TePDIBox::pointer& data );

      /**
       * Gets a PDIBox parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         TePDIBox::pointer& data ) const;

      /**
       * Sets a FilterMask parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       */
      void SetParameter( const std::string& pname,
                         const TePDIFilterMask::pointer& data );

      /**
       * Gets a FilterMask parameter.
       *
       * @param pname Parameter name.
       * @param data Parameter data.
       * @return true if OK, false if the parameter was no found or error.
       */
      bool GetParameter( const std::string& pname,
                         TePDIFilterMask::pointer& data ) const;

      /**
       * Checks if the parameter was set.
       *
       * @param pname Parameter name.
       * @param type Parameter type.
       * @return true if OK, false if the parameter was not set.
       */
      bool CheckParameter( const std::string& pname,
                           TePDIParamTypes type ) const;

      /**
       * Operator = overload.
       *
       * @param ext_params External instance TePDIParameters reference.
       */
      void operator=( const TePDIParameters& ext_params );

    protected :
      /**
       * Internal Type definitions of suported parameters.
       */
      typedef std::map< std::string, std::string >
        TePDIStringMap;
      typedef std::map< std::string, int >
        TePDIIntegerMap;
      typedef std::map< std::string, std::vector< int > >
        TePDIIntegerVectorMap;
      typedef std::map< std::string, double >
        TePDIDoubleMap;
      typedef std::map< std::string, TePDITypes::TePDIRasterPtrType >
        TePDIRasterMap;
      typedef std::map< std::string, TePDIBox::pointer >
        TePDIBoxMap;
      typedef std::map< std::string, TePDIRgbPalette::pointer >
        TePDIRgbPaletteMap;
      typedef std::map< std::string, TePDIFilterMask::pointer >
        TePDIFilterMaskMap;

      /**
       * Stored TeRaster references Map.
       */
      mutable TePDIRasterMap RasterMap_;

      /**
       * Stored integer references Map.
       */
      mutable TePDIIntegerMap IntegerMap_;

      /**
       * Stored integer vector references Map.
       */
      mutable TePDIIntegerVectorMap IntegerVectorMap_;

      /**
       * Stored double references Map.
       */
      mutable TePDIDoubleMap DoubleMap_;

      /**
       * Stored string references Map.
       */
      mutable TePDIStringMap StringMap_;

      /**
       * Stored PDIBox references Map.
       */
      mutable TePDIBoxMap PDIBoxMap_;

      /**
       * Stored PDIPalette references Map.
       */
      mutable TePDIRgbPaletteMap PDIRgbPaletteMap_;

      /**
       * Stored PDIFilterMask references Map.
       */
      mutable TePDIFilterMaskMap PDIFilterMaskMap_;
};

#endif //TEPDIPARAMETERS_HPP
