/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include <libsyncml/objects/sml_ds_server.h>
#include <libsyncml/objects/sml_auth.h>

#include <libsyncml/transports/http_client.h>
#include <libsyncml/transports/http_server.h>

#include <libsyncml/transports/obex_client.h>
#include <libsyncml/transports/obex_server.h>

#include <libsyncml/objects/sml_devinf_obj.h>

#define NUM_SESSIONS 30

typedef struct managerTracker {
	SmlManager *manager;
	SmlSession *session;
	SmlDsSession *dssessions[NUM_SESSIONS];
	SmlDsServer *ds[3];
} managerTracker;

unsigned int init_alerts_received = 0;
unsigned int alerts_received = 0;
unsigned int syncs_received = 0;
unsigned int sans_received = 0;
unsigned int sync_ends_received = 0;
unsigned int changes_received = 0;
unsigned int replies_received = 0;
unsigned int adds_received = 0;
unsigned int add_replies_received = 0;
unsigned int deletes_received = 0;
unsigned int delete_replies_received = 0;
unsigned int modifies_received = 0;
unsigned int modify_replies_received = 0;
unsigned int session_ends = 0;

unsigned int transport_errors = 0;
unsigned int num_sessions = 0;
unsigned int num_finals = 0;
unsigned int num_end = 0;
unsigned int session_errors = 0;

static void _manager_event(SmlManager *manager, SmlManagerEventType type, SmlSession *session, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p)", __func__, manager, type, session, error, userdata);
	managerTracker *tracker = userdata;
	smlAssert(manager);
	smlAssert(userdata);
	
	switch (type) {
		case SML_MANAGER_SESSION_FLUSH:
			break;
		case SML_MANAGER_CONNECT_DONE:
		case SML_MANAGER_DISCONNECT_DONE:
//			fail();
			break;
		case SML_MANAGER_TRANSPORT_ERROR:
			transport_errors++;
			break;
		case SML_MANAGER_SESSION_NEW:
			smlAssert(session);
			tracker->session = session;
			num_sessions++;
			smlSessionRef(session);
			break;
		case SML_MANAGER_SESSION_FINAL:
			num_finals++;
			break;
		case SML_MANAGER_SESSION_END:
			num_end++;
			break;
		case SML_MANAGER_SESSION_ERROR:
		case SML_MANAGER_SESSION_WARNING:
			session_errors++;
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _recv_init_alert(SmlDsSession *dsession, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, dsession, userdata);
	managerTracker *tracker = userdata;
	
	if (tracker->ds[0] == smlDsSessionGetServer(dsession))
		tracker->dssessions[0] = dsession;
	else if (tracker->ds[1] == smlDsSessionGetServer(dsession))
		tracker->dssessions[1] = dsession;
	else if (tracker->ds[2] == smlDsSessionGetServer(dsession))
		tracker->dssessions[2] = dsession;
	
	smlDsSessionRef(dsession);
	
	init_alerts_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static SmlBool _recv_alert(SmlDsSession *dsession, SmlAlertType type, const char *last, const char *next, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %s, %s, %p)", __func__, dsession, type, last, next, userdata);
	
	fail_unless(last != NULL, NULL);
	fail_unless(next != NULL, NULL);
	
	alerts_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void _alert_reply(SmlSession *dsession, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
	
	fail_unless(status != NULL, NULL);
	
	if (GPOINTER_TO_INT(userdata) == 1) {
		fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	} else {
		fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_FATAL, NULL);
	}
	
	replies_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _sync_reply(SmlSession *dsession, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
	
	fail_unless(status != NULL, NULL);
	
	if (GPOINTER_TO_INT(userdata) == 1) {
		fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	} else {
		fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_FATAL, NULL);
	}
	
	replies_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _add_reply(SmlDsSession *session, SmlStatus *status, const char *newuid, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %s, %p)", __func__, session, status, newuid, userdata);
	
	fail_unless(status != NULL, NULL);
	fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	fail_unless(GPOINTER_TO_INT(userdata) == 1 || GPOINTER_TO_INT(userdata) == 2, NULL);

	if (GPOINTER_TO_INT(userdata) == 1)
		fail_unless(!strcmp(newuid, "newuid"), NULL);
	else
		fail_unless(newuid == NULL, NULL);
	
	add_replies_received++;

	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _modify_reply(SmlDsSession *session, SmlStatus *status, const char *newuid, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %s, %p)", __func__, session, status, newuid, userdata);
	
	fail_unless(status != NULL, NULL);
	fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	fail_unless(GPOINTER_TO_INT(userdata) == 1, NULL);

	fail_unless(newuid == NULL, NULL);
	
	modify_replies_received++;

	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _delete_reply(SmlDsSession *session, SmlStatus *status, const char *newuid, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %s, %p)", __func__, session, status, newuid, userdata);
	
	fail_unless(status != NULL, NULL);
	fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	fail_unless(GPOINTER_TO_INT(userdata) == 1, NULL);

	fail_unless(newuid == NULL, NULL);
	
	delete_replies_received++;

	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _map_reply(SmlSession *dsession, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
	
	fail_unless(status != NULL, NULL);
	fail_unless(smlStatusGetClass(status) == SML_ERRORCLASS_SUCCESS, NULL);
	
	replies_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static SmlErrorType _recv_san(SmlDsServer *server, SmlAlertType type, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, server, type, userdata);
	fail_unless(GPOINTER_TO_INT(userdata) == 1, NULL);
	
	fail_unless(type == SML_ALERT_TWO_WAY_BY_SERVER, NULL);
	sans_received++;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return SML_NO_ERROR;
}

static void _recv_sync(SmlDsSession *dsession, unsigned int numChanges, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
	
	fail_unless(numChanges != 87, NULL);
	syncs_received++;
		
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static SmlBool _recv_changes(SmlDsSession *dsession, SmlChangeType type, const char *uid, char *data, unsigned int size, const char *contenttype, void *userdata, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %s, %s, %i, %s, %p, %p)", __func__, dsession, type, uid, data, size, contenttype, userdata, error);

	if (type == SML_CHANGE_ADD) {
		adds_received++;
		fail_unless(!strcmp(uid, "uid"), NULL);
		fail_unless(!strncmp(data, "data", size), NULL);
		fail_unless(size == 4, NULL);
	} else if (type == SML_CHANGE_REPLACE) {
		modifies_received++;
		fail_unless(!strcmp(uid, "uid"), NULL);
		fail_unless(!strncmp(data, "newdata", size), NULL);
		fail_unless(size == 7, NULL);
	} else if (type == SML_CHANGE_DELETE) {
		deletes_received++;
		fail_unless(!strcmp(uid, "uid"), NULL);
		fail_unless(!data, NULL);
		fail_unless(size == 0, NULL);
	} else
		fail(NULL);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

START_TEST (sync_multi_start)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	SmlSession *session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	
	fail_unless(smlManagerSessionAdd(clientmanager, session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	
	while (num_end != 2 || replies_received != 3 || num_finals != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 3, NULL);
	fail_unless(syncs_received == 0, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 3, NULL);
	fail_unless(adds_received == 0, NULL);
	fail_unless(add_replies_received == 0, NULL);
	fail_unless(deletes_received == 0, NULL);
	fail_unless(delete_replies_received == 0, NULL);
	fail_unless(modifies_received == 0, NULL);
	fail_unless(modify_replies_received == 0, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 2, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

/* client -- 3 * alert ---> server
 * server -- statuses, 3 * alert ---> client
 * client -- statuses, num * add, num * replace, num * delete ---> server
 * server -- statuses ---> client
 */
START_TEST (sync_multi_stress)
{
	int num_items = 10;
	
	init_alerts_received = 0;
	alerts_received = 0;
	syncs_received = 0;
	sync_ends_received = 0;
	changes_received = 0;
	replies_received = 0;
	adds_received = 0;
	add_replies_received = 0;
	deletes_received = 0;
	delete_replies_received = 0;
	modifies_received = 0;
	modify_replies_received = 0;
	session_ends = 0;
	
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	clienttracker->session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	
	fail_unless(smlManagerSessionAdd(clientmanager, clienttracker->session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	smlDsSessionSendAlert(servertracker->dssessions[0], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[1], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[2], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetAlert(clienttracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(clienttracker->dssessions[0], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[1], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[2], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	
	int i = 0;
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(clienttracker->dssessions[0], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[1], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(servertracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 3 || adds_received != num_items * 3 || deletes_received != num_items * 3 || modifies_received != num_items * 3) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	while (num_end != 2 || replies_received != 9 || num_finals != 4) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 6, NULL);
	fail_unless(syncs_received == 3, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 9, NULL);
	fail_unless(adds_received == num_items * 3, NULL);
	fail_unless(add_replies_received == num_items * 3, NULL);
	fail_unless(deletes_received == num_items * 3, NULL);
	fail_unless(delete_replies_received == num_items * 3, NULL);
	fail_unless(modifies_received == num_items * 3, NULL);
	fail_unless(modify_replies_received == num_items * 3, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 4, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	
	smlDsSessionUnref(clienttracker->dssessions[0]);
	smlDsSessionUnref(clienttracker->dssessions[1]);
	smlDsSessionUnref(clienttracker->dssessions[2]);
	
	smlDsSessionUnref(servertracker->dssessions[0]);
	smlDsSessionUnref(servertracker->dssessions[1]);
	smlDsSessionUnref(servertracker->dssessions[2]);
	
	smlDsServerFree(clienttracker->ds[0]);
	smlDsServerFree(clienttracker->ds[1]);
	smlDsServerFree(clienttracker->ds[2]);
	
	smlDsServerFree(servertracker->ds[0]);
	smlDsServerFree(servertracker->ds[1]);
	smlDsServerFree(servertracker->ds[2]);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

/* client -- 3 * alert ---> server
 * server -- statuses, 3 * alert ---> client
 * client -- statuses, num * add, num * replace, num * delete ---> server
 * server -- statuses, num * add, num * replace, num * delete  ---> client
 * client -- statuses, map ---> server
 * server -- statuses ---> client
 */
START_TEST (sync_multi_stress2)
{
	int num_items = 100;
	
	init_alerts_received = 0;
	alerts_received = 0;
	syncs_received = 0;
	sync_ends_received = 0;
	changes_received = 0;
	replies_received = 0;
	adds_received = 0;
	add_replies_received = 0;
	deletes_received = 0;
	delete_replies_received = 0;
	modifies_received = 0;
	modify_replies_received = 0;
	session_ends = 0;
	
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	clienttracker->session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	
	fail_unless(smlManagerSessionAdd(clientmanager, clienttracker->session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	smlDsSessionSendAlert(servertracker->dssessions[0], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[1], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[2], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetAlert(clienttracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(clienttracker->dssessions[0], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[1], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[2], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	
	int i = 0;
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(clienttracker->dssessions[0], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[1], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(servertracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 3 || adds_received != num_items * 3 || deletes_received != num_items * 3 || modifies_received != num_items * 3) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(servertracker->dssessions[0], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(servertracker->dssessions[1], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(servertracker->dssessions[2], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(servertracker->dssessions[0], &error);
	smlDsSessionCloseSync(servertracker->dssessions[1], &error);
	smlDsSessionCloseSync(servertracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(clienttracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(clienttracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(clienttracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 6 || adds_received != num_items * 6 || deletes_received != num_items * 6 || modifies_received != num_items * 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[0], "uid", "newuid", &error), NULL);
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[1], "uid", "newuid", &error), NULL);
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[2], "uid", "newuid", &error), NULL);
	}
	
	smlDsSessionCloseMap(clienttracker->dssessions[0], _map_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionCloseMap(clienttracker->dssessions[1], _map_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionCloseMap(clienttracker->dssessions[2], _map_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	while (delete_replies_received != num_items * 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	while (num_end != 2 || replies_received != 15 || num_finals != 6) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 6, NULL);
	fail_unless(syncs_received == 6, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 15, NULL);
	fail_unless(adds_received == num_items * 6, NULL);
	fail_unless(add_replies_received == num_items * 6, NULL);
	fail_unless(deletes_received == num_items * 6, NULL);
	fail_unless(delete_replies_received == num_items * 6, NULL);
	fail_unless(modifies_received == num_items * 6, NULL);
	fail_unless(modify_replies_received == num_items * 6, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 6, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	
	smlDsSessionUnref(clienttracker->dssessions[0]);
	smlDsSessionUnref(clienttracker->dssessions[1]);
	smlDsSessionUnref(clienttracker->dssessions[2]);
	
	smlDsSessionUnref(servertracker->dssessions[0]);
	smlDsSessionUnref(servertracker->dssessions[1]);
	smlDsSessionUnref(servertracker->dssessions[2]);
	
	smlDsServerFree(clienttracker->ds[0]);
	smlDsServerFree(clienttracker->ds[1]);
	smlDsServerFree(clienttracker->ds[2]);
	
	smlDsServerFree(servertracker->ds[0]);
	smlDsServerFree(servertracker->ds[1]);
	smlDsServerFree(servertracker->ds[2]);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

/* client -- 3 * alert ---> server
 * server -- statuses, 3 * alert ---> client
 * client -- statuses, num * add, num * replace, num * delete ---> server
 * server -- statuses ---> client
 */
START_TEST (sync_multi_conf_from_devinf)
{
	int num_items = 1;
	
	init_alerts_received = 0;
	alerts_received = 0;
	syncs_received = 0;
	sync_ends_received = 0;
	changes_received = 0;
	replies_received = 0;
	adds_received = 0;
	add_replies_received = 0;
	deletes_received = 0;
	delete_replies_received = 0;
	modifies_received = 0;
	modify_replies_received = 0;
	session_ends = 0;
	
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 12004;
	clientConfig.url = "http://127.0.0.1:12004";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 12004;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	/* The devinf obj */
	SmlDevInf *devinf = smlDevInfNew("LibSyncmML", SML_DEVINF_DEVTYPE_WORKSTATION, &error);
	smlDevInfSetSupportsNumberOfChanges(devinf, FALSE);
	smlDevInfSetSupportsLargeObjs(devinf, FALSE);
	smlDevInfSetSupportsUTC(devinf, FALSE);
	SmlDevInfAgent *clientagent = smlDevInfAgentNew(devinf, &error);
	smlDevInfAgentRegister(clientagent, clientmanager, &error);

	/* And we also add the devinfo to the devinf agent */
	SmlDevInfDataStore *datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc1), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc2), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCAL, "2.0");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCAL, "2.0");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc3), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_PLAIN, "1.0");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_PLAIN, "1.0");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	devinf = smlDevInfNew("LibSyncmML", SML_DEVINF_DEVTYPE_SERVER, &error);
	smlDevInfSetSupportsNumberOfChanges(devinf, FALSE);
	smlDevInfSetSupportsLargeObjs(devinf, FALSE);
	smlDevInfSetSupportsUTC(devinf, FALSE);
	SmlDevInfAgent *serveragent = smlDevInfAgentNew(devinf, &error);
	smlDevInfAgentRegister(serveragent, servermanager, &error);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc1), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCARD, "2.1");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc2), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_VCAL, "2.0");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_VCAL, "2.0");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	datastore = smlDevInfDataStoreNew(smlLocationGetURI(loc3), &error);
	smlDevInfDataStoreSetRxPref(datastore, SML_ELEMENT_TEXT_PLAIN, "1.0");
	smlDevInfDataStoreSetTxPref(datastore, SML_ELEMENT_TEXT_PLAIN, "1.0");
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_TWO_WAY, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SLOW_SYNC, TRUE);
	smlDevInfDataStoreSetSyncCap(datastore, SML_DEVINF_SYNCTYPE_SERVER_ALERTED_SYNC, TRUE);
	smlDevInfAddDataStore(devinf, datastore);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	clienttracker->session = smlSessionNew(SML_SESSION_TYPE_CLIENT, SML_MIMETYPE_XML, SML_VERSION_12, SML_PROTOCOL_SYNCML, loc, loc, NULL, 0, &error);
	smlSessionSetReceivingLimit(clienttracker->session, 76400);
	fail_unless(smlSessionGetReceivingMaxObjSize(clienttracker->session) == 0, NULL);
	smlSessionSetReceivingMaxObjSize(clienttracker->session, 76500);
	fail_unless(smlSessionGetReceivingMaxObjSize(clienttracker->session) == 76500, NULL);
	
	smlDevInfConfigureSession(devinf, clienttracker->session);
	
	fail_unless(smlManagerSessionAdd(clientmanager, clienttracker->session, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlDevInfAgentSendDevInf(clientagent, clienttracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(smlSessionGetSendingLimit(servertracker->session) == 76400, NULL);
	fail_unless(smlSessionGetSendingMaxObjSize(servertracker->session) == -1, NULL);
	smlSessionSetReceivingLimit(servertracker->session, 76400);
	smlSessionSetReceivingMaxObjSize(servertracker->session, 76500);
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 3 || !smlDevInfAgentGetDevInf(serveragent)) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		usleep(100);
	}
	
	SmlDevInf *recvClientDevInf = smlDevInfAgentGetDevInf(serveragent);
	fail_unless(recvClientDevInf != NULL, NULL);
	fail_unless(smlDevInfNumDataStores(recvClientDevInf) == 3, NULL);
	fail_unless(smlDevInfSupportsNumberOfChanges(recvClientDevInf) == FALSE, NULL);
	fail_unless(smlDevInfSupportsLargeObjs(recvClientDevInf) == FALSE, NULL);
	fail_unless(smlDevInfSupportsUTC(recvClientDevInf) == FALSE, NULL);
	
	fail_unless(smlDevInfAgentSendDevInf(serveragent, servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlDsSessionSendAlert(servertracker->dssessions[0], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[1], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[2], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetAlert(clienttracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(smlSessionGetSendingLimit(clienttracker->session) == 76400, NULL);
	fail_unless(smlSessionGetSendingMaxObjSize(servertracker->session) == -1, NULL);
	
	SmlDevInf *recvServerDevInf = smlDevInfAgentGetDevInf(clientagent);
	fail_unless(recvServerDevInf != NULL, NULL);
	fail_unless(smlDevInfNumDataStores(recvServerDevInf) == 3, NULL);
	fail_unless(smlDevInfSupportsNumberOfChanges(recvServerDevInf) == FALSE, NULL);
	fail_unless(smlDevInfSupportsLargeObjs(recvServerDevInf) == FALSE, NULL);
	fail_unless(smlDevInfSupportsUTC(recvServerDevInf) == FALSE, NULL);
	
	smlDsSessionSendSync(clienttracker->dssessions[0], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[1], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[2], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	
	int i = 0;
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(clienttracker->dssessions[0], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[1], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(servertracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 3 || adds_received != num_items * 3 || deletes_received != num_items * 3 || modifies_received != num_items * 3) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(servertracker->dssessions[0], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(servertracker->dssessions[1], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(servertracker->dssessions[2], 87, _sync_reply, GINT_TO_POINTER(1), &error);
	
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(servertracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(servertracker->dssessions[0], &error);
	smlDsSessionCloseSync(servertracker->dssessions[1], &error);
	smlDsSessionCloseSync(servertracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(clienttracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(clienttracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(clienttracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(clienttracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 6 || adds_received != num_items * 6 || deletes_received != num_items * 6 || modifies_received != num_items * 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[0], "uid", "newuid", &error), NULL);
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[1], "uid", "newuid", &error), NULL);
		fail_unless(smlDsSessionQueueMap(clienttracker->dssessions[2], "uid", "newuid", &error), NULL);
	}
	
	smlDsSessionCloseMap(clienttracker->dssessions[0], _map_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionCloseMap(clienttracker->dssessions[1], _map_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionCloseMap(clienttracker->dssessions[2], _map_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	while (delete_replies_received != num_items * 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	
	fail_unless(smlSessionGetSendingLimit(servertracker->session) == 76400, NULL);
	fail_unless(smlSessionGetSendingMaxObjSize(servertracker->session) == -1, NULL);
	
	fail_unless(smlSessionGetSendingLimit(clienttracker->session) == 76400, NULL);
	fail_unless(smlSessionGetSendingMaxObjSize(clienttracker->session) == -1, NULL);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	while (num_end != 2 || replies_received != 15 || num_finals != 6) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 6, NULL);
	fail_unless(syncs_received == 6, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 15, NULL);
	fail_unless(adds_received == num_items * 6, NULL);
	fail_unless(add_replies_received == num_items * 6, NULL);
	fail_unless(deletes_received == num_items * 6, NULL);
	fail_unless(delete_replies_received == num_items * 6, NULL);
	fail_unless(modifies_received == num_items * 6, NULL);
	fail_unless(modify_replies_received == num_items * 6, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 6, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	
	smlDsSessionUnref(clienttracker->dssessions[0]);
	smlDsSessionUnref(clienttracker->dssessions[1]);
	smlDsSessionUnref(clienttracker->dssessions[2]);
	
	smlDsSessionUnref(servertracker->dssessions[0]);
	smlDsSessionUnref(servertracker->dssessions[1]);
	smlDsSessionUnref(servertracker->dssessions[2]);
	
	smlDsServerFree(clienttracker->ds[0]);
	smlDsServerFree(clienttracker->ds[1]);
	smlDsServerFree(clienttracker->ds[2]);
	
	smlDsServerFree(servertracker->ds[0]);
	smlDsServerFree(servertracker->ds[1]);
	smlDsServerFree(servertracker->ds[2]);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (sync_send_san)
{
	int num_items = 10;
	
	init_alerts_received = 0;
	alerts_received = 0;
	syncs_received = 0;
	sync_ends_received = 0;
	changes_received = 0;
	replies_received = 0;
	adds_received = 0;
	add_replies_received = 0;
	deletes_received = 0;
	delete_replies_received = 0;
	modifies_received = 0;
	modify_replies_received = 0;
	session_ends = 0;
	
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_OBEX_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_OBEX_CLIENT, &error);
	
	SmlTransportObexClientConfig clientConfig;
	clientConfig.type = SML_OBEX_TYPE_NET;
	clientConfig.url = "127.0.0.1";
	clientConfig.port = 10110;
	
	SmlTransportObexServerConfig serverConfig;
	serverConfig.type = SML_OBEX_SERVER_TYPE_NET;
	serverConfig.path = NULL;
	serverConfig.port = 10110;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerSetSanCallback(clienttracker->ds[0], _recv_san, GINT_TO_POINTER(1));
	smlDsServerSetSanCallback(clienttracker->ds[1], _recv_san, GINT_TO_POINTER(1));
	smlDsServerSetSanCallback(clienttracker->ds[2], _recv_san, GINT_TO_POINTER(1));
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	fail_unless(smlTransportConnect(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_11, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "test", "/", SML_MIMETYPE_XML, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlDsServerAddSan(servertracker->ds[0], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlDsServerAddSan(servertracker->ds[1], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlDsServerAddSan(servertracker->ds[2], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationSend(san, client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (sans_received != 3 || num_sessions != 1) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(clienttracker->session != NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	smlDsSessionSendAlert(servertracker->dssessions[0], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[1], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[2], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetAlert(clienttracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(clienttracker->dssessions[0], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[1], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[2], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	
	int i = 0;
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(clienttracker->dssessions[0], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[1], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(servertracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 3 || adds_received != num_items * 3 || deletes_received != num_items * 3 || modifies_received != num_items * 3) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	while (num_end != 2 || replies_received != 9 || num_finals != 5) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 6, NULL);
	fail_unless(syncs_received == 3, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 9, NULL);
	fail_unless(adds_received == num_items * 3, NULL);
	fail_unless(add_replies_received == num_items * 3, NULL);
	fail_unless(deletes_received == num_items * 3, NULL);
	fail_unless(delete_replies_received == num_items * 3, NULL);
	fail_unless(modifies_received == num_items * 3, NULL);
	fail_unless(modify_replies_received == num_items * 3, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 5, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	
	smlDsSessionUnref(clienttracker->dssessions[0]);
	smlDsSessionUnref(clienttracker->dssessions[1]);
	smlDsSessionUnref(clienttracker->dssessions[2]);
	
	smlDsSessionUnref(servertracker->dssessions[0]);
	smlDsSessionUnref(servertracker->dssessions[1]);
	smlDsSessionUnref(servertracker->dssessions[2]);
	
	smlDsServerFree(clienttracker->ds[0]);
	smlDsServerFree(clienttracker->ds[1]);
	smlDsServerFree(clienttracker->ds[2]);
	
	smlDsServerFree(servertracker->ds[0]);
	smlDsServerFree(servertracker->ds[1]);
	smlDsServerFree(servertracker->ds[2]);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (sync_send_san12)
{
	int num_items = 10;
	
	init_alerts_received = 0;
	alerts_received = 0;
	syncs_received = 0;
	sync_ends_received = 0;
	changes_received = 0;
	replies_received = 0;
	adds_received = 0;
	add_replies_received = 0;
	deletes_received = 0;
	delete_replies_received = 0;
	modifies_received = 0;
	modify_replies_received = 0;
	session_ends = 0;
	
	transport_errors = 0;
	num_sessions = 0;
	num_finals = 0;
	num_end = 0;
	session_errors = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_OBEX_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_OBEX_CLIENT, &error);
	
	SmlTransportObexClientConfig clientConfig;
	clientConfig.type = SML_OBEX_TYPE_NET;
	clientConfig.url = "127.0.0.1";
	clientConfig.port = 10110;
	
	SmlTransportObexServerConfig serverConfig;
	serverConfig.type = SML_OBEX_SERVER_TYPE_NET;
	serverConfig.path = NULL;
	serverConfig.port = 10110;
	
	managerTracker *clienttracker = g_malloc0(sizeof(managerTracker));
	SmlManager *clientmanager = clienttracker->manager = smlManagerNew(server, &error);
	smlManagerSetEventCallback(clienttracker->manager, _manager_event, clienttracker);
	managerTracker *servertracker = g_malloc0(sizeof(managerTracker));
	SmlManager *servermanager = servertracker->manager = smlManagerNew(client, &error);
	smlManagerSetEventCallback(servertracker->manager, _manager_event, servertracker);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	fail_unless(smlManagerStart(clientmanager, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlManagerStart(servermanager, &error), NULL);
	fail_unless(error == NULL, NULL);

	SmlLocation *loc = smlLocationNew("test", NULL, &error);
	fail_unless(loc != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc1 = smlLocationNew("test1", NULL, &error);
	fail_unless(loc1 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc2 = smlLocationNew("test2", NULL, &error);
	fail_unless(loc2 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlLocation *loc3 = smlLocationNew("test3", NULL, &error);
	fail_unless(loc3 != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	clienttracker->ds[0] = smlDsClientNew(SML_ELEMENT_TEXT_VCARD, loc1, loc1, &error);
	clienttracker->ds[1] = smlDsClientNew(SML_ELEMENT_TEXT_VCAL, loc2, loc2, &error);
	clienttracker->ds[2] = smlDsClientNew(SML_ELEMENT_TEXT_PLAIN, loc3, loc3, &error);
	smlDsServerSetSanCallback(clienttracker->ds[0], _recv_san, GINT_TO_POINTER(1));
	smlDsServerSetSanCallback(clienttracker->ds[1], _recv_san, GINT_TO_POINTER(1));
	smlDsServerSetSanCallback(clienttracker->ds[2], _recv_san, GINT_TO_POINTER(1));
	smlDsServerRegister(clienttracker->ds[0], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[1], clientmanager, &error);
	smlDsServerRegister(clienttracker->ds[2], clientmanager, &error);
				
	servertracker->ds[0] = smlDsServerNew(SML_ELEMENT_TEXT_VCARD, loc1, &error);
	servertracker->ds[1] = smlDsServerNew(SML_ELEMENT_TEXT_VCAL, loc2, &error);
	servertracker->ds[2] = smlDsServerNew(SML_ELEMENT_TEXT_PLAIN, loc3, &error);
	smlDsServerSetConnectCallback(servertracker->ds[0], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[1], _recv_init_alert, servertracker);
	smlDsServerSetConnectCallback(servertracker->ds[2], _recv_init_alert, servertracker);
	smlDsServerRegister(servertracker->ds[0], servermanager, &error);
	smlDsServerRegister(servertracker->ds[1], servermanager, &error);
	smlDsServerRegister(servertracker->ds[2], servermanager, &error);
	
	smlLocationUnref(loc1);
	smlLocationUnref(loc2);
	smlLocationUnref(loc3);
	
	/* The authenticator */
	SmlAuthenticator *auth = smlAuthNew(&error);
	smlAuthSetEnable(auth, FALSE);
	smlAuthRegister(auth, clientmanager, &error);
	
	/* The authenticator */
	SmlAuthenticator *auth2 = smlAuthNew(&error);
	smlAuthSetEnable(auth2, FALSE);
	smlAuthRegister(auth2, servermanager, &error);
	
	fail_unless(smlTransportConnect(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	SmlNotification *san = smlNotificationNew(SML_SAN_VERSION_12, SML_SAN_UIMODE_USER, SML_SAN_INITIATOR_SERVER, 65535, "test", "/", SML_MIMETYPE_XML, &error);
	fail_unless(san != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlDsServerAddSan(servertracker->ds[0], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlDsServerAddSan(servertracker->ds[1], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlDsServerAddSan(servertracker->ds[2], san, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlNotificationSend(san, client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (sans_received != 3 || num_sessions != 1) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(clienttracker->session != NULL, NULL);
	
	smlLocationUnref(loc);
	
	clienttracker->dssessions[0] = smlDsServerSendAlert(clienttracker->ds[0], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[1] = smlDsServerSendAlert(clienttracker->ds[1], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	clienttracker->dssessions[2] = smlDsServerSendAlert(clienttracker->ds[2], clienttracker->session, SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (init_alerts_received != 3 || num_sessions != 2) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionGetAlert(servertracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(servertracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	smlDsSessionSendAlert(servertracker->dssessions[0], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[1], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendAlert(servertracker->dssessions[2], SML_ALERT_TWO_WAY, "last", "next", _alert_reply, GINT_TO_POINTER(1), &error);
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionFlush(servertracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetAlert(clienttracker->dssessions[0], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[1], _recv_alert, GINT_TO_POINTER(1));
	smlDsSessionGetAlert(clienttracker->dssessions[2], _recv_alert, GINT_TO_POINTER(1));
	
	while (alerts_received != 6) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	smlDsSessionSendSync(clienttracker->dssessions[0], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[1], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	smlDsSessionSendSync(clienttracker->dssessions[2], num_items, _sync_reply, GINT_TO_POINTER(1), &error);
	
	int i = 0;
	for (i = 0; i < num_items; i++) {
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCARD, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCARD, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[0], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCARD, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_VCAL, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_VCAL, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[1], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_VCAL, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_ADD, "uid", "data", 4, SML_ELEMENT_TEXT_PLAIN, _add_reply, GINT_TO_POINTER(2), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_REPLACE, "uid", "newdata", 7, SML_ELEMENT_TEXT_PLAIN, _modify_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlDsSessionQueueChange(clienttracker->dssessions[2], SML_CHANGE_DELETE, "uid", NULL, 0, SML_ELEMENT_TEXT_PLAIN, _delete_reply, GINT_TO_POINTER(1), &error), NULL);
		fail_unless(error == NULL, NULL);
	}
	
	smlDsSessionCloseSync(clienttracker->dssessions[0], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[1], &error);
	smlDsSessionCloseSync(clienttracker->dssessions[2], &error);
	
	fail_unless(smlSessionFlush(clienttracker->session, TRUE, &error), NULL);
	
	smlDsSessionGetSync(servertracker->dssessions[0], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[1], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetSync(servertracker->dssessions[2], _recv_sync, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[0], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[1], _recv_changes, GINT_TO_POINTER(1));
	smlDsSessionGetChanges(servertracker->dssessions[2], _recv_changes, GINT_TO_POINTER(1));
	
	while (syncs_received != 3 || adds_received != num_items * 3 || deletes_received != num_items * 3 || modifies_received != num_items * 3) {
		smlDsSessionDispatch(servertracker->dssessions[0]);
		smlDsSessionDispatch(servertracker->dssessions[1]);
		smlDsSessionDispatch(servertracker->dssessions[2]);
		
		smlDsSessionDispatch(clienttracker->dssessions[0]);
		smlDsSessionDispatch(clienttracker->dssessions[1]);
		smlDsSessionDispatch(clienttracker->dssessions[2]);
		
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(servertracker->session != NULL, NULL);
	fail_unless(smlSessionEnd(servertracker->session, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlSessionUnref(servertracker->session);
	smlSessionUnref(clienttracker->session);
	
	while (num_end != 2 || replies_received != 9 || num_finals != 5) {
		smlManagerDispatch(servermanager);
		smlManagerDispatch(clientmanager);
		usleep(100);
	}
	
	fail_unless(init_alerts_received == 3, NULL);
	fail_unless(alerts_received == 6, NULL);
	fail_unless(syncs_received == 3, NULL);
	fail_unless(sync_ends_received == 0, NULL);
	fail_unless(changes_received == 0, NULL);
	fail_unless(replies_received == 9, NULL);
	fail_unless(adds_received == num_items * 3, NULL);
	fail_unless(add_replies_received == num_items * 3, NULL);
	fail_unless(deletes_received == num_items * 3, NULL);
	fail_unless(delete_replies_received == num_items * 3, NULL);
	fail_unless(modifies_received == num_items * 3, NULL);
	fail_unless(modify_replies_received == num_items * 3, NULL);
	fail_unless(transport_errors == 0, NULL);
	fail_unless(num_sessions == 2, NULL);
	fail_unless(num_finals == 5, NULL);
	fail_unless(num_end == 2, NULL);
	fail_unless(session_errors == 0, NULL);
	
	
	smlDsSessionUnref(clienttracker->dssessions[0]);
	smlDsSessionUnref(clienttracker->dssessions[1]);
	smlDsSessionUnref(clienttracker->dssessions[2]);
	
	smlDsSessionUnref(servertracker->dssessions[0]);
	smlDsSessionUnref(servertracker->dssessions[1]);
	smlDsSessionUnref(servertracker->dssessions[2]);
	
	smlDsServerFree(clienttracker->ds[0]);
	smlDsServerFree(clienttracker->ds[1]);
	smlDsServerFree(clienttracker->ds[2]);
	
	smlDsServerFree(servertracker->ds[0]);
	smlDsServerFree(servertracker->ds[1]);
	smlDsServerFree(servertracker->ds[2]);
	
	smlAuthFree(auth);
	smlAuthFree(auth2);
	
	g_free(clienttracker);
	g_free(servertracker);
	
	smlManagerStop(clientmanager);
	smlManagerStop(servermanager);
	
	smlManagerFree(clientmanager);
	smlManagerFree(servermanager);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

Suite *sync_suite(void)
{
	Suite *s = suite_create("Sync");
	Suite *s2 = suite_create("Sync");
	
	create_case(s, "sync_multi_start", sync_multi_start);
	create_case(s, "sync_multi_stress", sync_multi_stress);
	create_case(s, "sync_multi_stress2", sync_multi_stress2);
	create_case(s, "sync_multi_conf_from_devinf", sync_multi_conf_from_devinf);
	create_case(s, "sync_send_san", sync_send_san);
	create_case(s2, "sync_send_san12", sync_send_san12);
	
	return s2;
}

int main(void)
{
	int nf;

	Suite *s = sync_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
