/* CellList.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.lists;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.dnd.DragSource;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetAdapter;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.util.TooManyListenersException;
import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.DefaultListSelectionModel;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JViewport;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.TransferHandler;
import javax.swing.UIManager;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.MouseInputAdapter;
import javax.swing.event.MouseInputListener;

import static java.awt.dnd.DnDConstants.ACTION_NONE;
import static java.awt.dnd.DnDConstants.ACTION_MOVE;
import static java.awt.dnd.DnDConstants.ACTION_COPY;
import static java.awt.dnd.DnDConstants.ACTION_LINK;

/**
 * List component in which list items are layed out as cells in a grid.
 * The functionality is the same
 * as that of {@link javax.swing.JList} except for the following:
 * <ul>
 * <li>We use a different cell layout and visual representation.</li>
 * <li>Selection properties should be obtained directly from the selection
 * model. We do not provide proxy methods except {@link #getSelectedValue}
 * and {@link #getSelectedValues}.</li>
 * <li>Although drag and drop is enabled, we do not provide a default
 * transfer handler.</li>
 * </ul>
 * This list component uses a <code>cellRenderer</code> delegate of type
 * {@link CellListCellRenderer} to paint the visible cells in the list
 * and to obtain the correct tool tip information for that cell.
 */
public class CellList extends JComponent
        implements Scrollable, ListDataListener {
    
    //
    private ListSelectionModel selectionModel;
    
    //
    private ListModel dataModel;
    
    //
    private CellListCellRenderer cellRenderer;
    
    /**
     * Holds value of property cellWidth.
     */
    private int cellWidth;
    
    /**
     * Holds value of property cellHeight.
     */
    private int cellHeight;
    
    /**
     * Holds value of property numberOfColumns.
     */
    private int numberOfColumns;
    
    /**
     * Set the number of columns to use in this list.
     * @param numberOfColumns New value of property numberOfColumns.
     */
    public void setNumberOfColumns(int numberOfColumns) {
        if (this.numberOfColumns != numberOfColumns) {
            this.numberOfColumns = numberOfColumns;
            repaint();
        }
    }
    
    //
    private final Handler handler;
    
    //
    private final MouseHandler mouseHandler;
    
    //
    private final MouseHandlerDragEnabled mouseHandlerDragEnabled;
    
    //
    private MouseInputListener currentMouseHandler;
    
    /**
     * Construct a list of this type for the given model, with the given
     * cell size.
     * @param cellWidth width of every grid cell
     * @param cellHeight height of every grid cell
     * @param numberOfColumns initial number of columns
     */
    public CellList(ListModel dataModel, int cellWidth, int cellHeight, int numberOfColumns) {
        this.dataModel = dataModel;
        this.cellWidth = cellWidth;
        this.cellHeight = cellHeight;
        this.numberOfColumns = numberOfColumns;
        
        ToolTipManager.sharedInstance().registerComponent(this);
        
        dataModel.addListDataListener(this);
        
        dropInProgress = false;
        dropLocation = new Point();
        
        handler = new Handler();
        addFocusListener(handler);
        setSelectionModel(new DefaultListSelectionModel()); // not overridable (PMD)
        
        setOpaque(true);
        setFocusable(true);
        setAutoscrolls(true);
        
        this.cellRenderer = new DefaultCellListCellRenderer();
        
        setBackground(UIManager.getColor("List.background"));
        
        mouseHandlerDragEnabled = new MouseHandlerDragEnabled();
        mouseHandler = new MouseHandler();
        this.dragEnabled = false;
        currentMouseHandler = mouseHandler;
        addMouseListener(currentMouseHandler);
        addMouseMotionListener(currentMouseHandler);
        
        setInputMap(WHEN_FOCUSED, (InputMap)UIManager.get("List.focusInputMap"));
        initActionMap(); // not overridable (PMD)
    }
    
    /**
     * Width of a grid cell.
     * @return Value of property cellWidth.
     */
    public int getCellWidth() {
        
        return this.cellWidth;
    }
    
    /**
     * Set the width of a grid cell.
     * @param cellWidth New value of property cellWidth.
     */
    public void setCellWidth(int cellWidth) {
        if (cellWidth != this.cellWidth) {
            this.cellWidth = cellWidth;
            repaint();
        }
    }
    
    /**
     * Height of a grid cell.
     * @return Value of property cellHeight.
     */
    public int getCellHeight() {
        
        return this.cellHeight;
    }
    
    /**
     * Set the height of a grid cell.
     * @param cellHeight New value of property cellHeight.
     */
    public void setCellHeight(int cellHeight) {
        if (cellHeight != this.cellHeight) {
            this.cellHeight = cellHeight;
            repaint();
        }
    }
    
    /**
     * Return the current selection model.
     */
    public ListSelectionModel getSelectionModel() {
        
        return this.selectionModel;
    }
    
    /**
     * Set width and height of a grid cell.
     * @param cellWidth New value of property cellWidth.
     * @param cellHeight New value of property cellHeight.
     */
    public void setCellSize(int cellWidth, int cellHeight) {
        this.cellWidth = cellWidth;
        this.cellHeight = cellHeight;
        repaint();
    }
    
    /**
     * Set or change the selection model.
     */
    public final void setSelectionModel(ListSelectionModel selectionModel) {
        if (this.selectionModel != null)
            this.selectionModel.removeListSelectionListener(handler);
        this.selectionModel = selectionModel;
        if (selectionModel != null)
            selectionModel.addListSelectionListener(handler);
    }
    
    /**
     * Return the current data model.
     */
    public ListModel getModel() {
        return this.dataModel;
    }
    
    /**
     * Sets a new data model and clears the selection.
     */
    public void setModel(ListModel model) {
        if (model == null) {
            throw new IllegalArgumentException("model must be non null");
        }
        model.removeListDataListener(this);
        this.dataModel = model;
        this.dataModel.addListDataListener(this);
    }
    
    /**
     * Return the cell renderer used by this list.
     */
    public CellListCellRenderer getCellRenderer() {
        return this.cellRenderer;
    }
    
    /**
     * Set the cell renderer.
     */
    public void setCellRenderer(CellListCellRenderer cellRenderer) {
        if (cellRenderer != this.cellRenderer) {
            this.cellRenderer = cellRenderer;
            repaint();
        }
    }
    
    /**
     * Overrides <code>JComponent</code>'s <code>getToolTipText</code>
     * method in order to allow the renderer's tips to be used
     * if it has text set.
     * @see JComponent#getToolTipText
     */
    public String getToolTipText(MouseEvent event) {
        if(event != null) {
            Point p = event.getPoint();
            int index = locationToIndex(p);
            
            if (index != -1 && cellRenderer != null) {
                Rectangle cellBounds = getCellBounds(index, index);
                if (cellBounds != null)
                    return cellRenderer.getToolTipText(this, dataModel.getElementAt(index), index);
            }
        }
        return super.getToolTipText();
    }
    
    /**
     * Convert a point in component coordinates to the index of the
     * cell at that location, or -1 if there is no such cell.
     */
    public int locationToIndex(Point location) {
        Insets insets = getInsets();
        int col = (location.x - insets.left)/ cellWidth;
        if (col < 0 || col >= numberOfColumns)
            return -1;
        int row = (location.y - insets.top) / cellHeight;
        int index = row*numberOfColumns + col;
        return index < dataModel.getSize() ? index : -1;
    }
    
    /**
     * Return the top left coordinate of the cell with given index,
     * or <code>null</code> if the index is not valid.
     */
    public Point indexToLocation(int index) {
        if (index < 0 || index >= dataModel.getSize())
            return null;
        else {
            Insets insets = getInsets();
            int row = index / numberOfColumns;
            int col = index % numberOfColumns;
            return new Point(insets.left + col*cellWidth,
                    insets.top + row*cellHeight);
        }
    }
    
    /**
     * Returns the bounds of the specified range of items in component
     * coordinates.Returns <code>null</code> if index isn't valid.
     *
     * @param index0  the index of the first cell in the range
     * @param index1  the index of the last cell in the range
     * @return the bounds of the indexed cells in pixels
     */
    public Rectangle getCellBounds(int index0, int index1) {
        if (index0 > index1 || index0 < 0 || index1 >= dataModel.getSize())
            return null;
        int row0 = index0 / numberOfColumns;
        int col0 = index0 % numberOfColumns;
        int row1 = index1 / numberOfColumns;
        int col1 = index1 % numberOfColumns;
        Insets insets = getInsets();
        return new Rectangle
                (insets.left + col0*cellWidth,
                insets.top + row0*cellHeight,
                (col1 - col0 + 1)*cellWidth,
                (row1 - row0 + 1)*cellHeight );
    }
    
    // extends JComponent
    @Override protected void paintComponent(Graphics g) {
        Rectangle clip = g.getClipBounds();
        
        // paint background
        if (isOpaque()) {
            g.setColor(getBackground());
            g.fillRect(clip.x, clip.y, clip.width, clip.height);
        }
        
        int count = dataModel.getSize();
        if (count <= 0)
            return ;
        Insets insets = getInsets();
        
        // determine which cells should be repainted
        int firstcol = (clip.x - insets.left) / cellWidth;
        if (firstcol < 0)
            firstcol = 0;
        else if (firstcol >= numberOfColumns)
            return;
        int firstrow = (clip.y - insets.top) / cellHeight;
        if (firstrow < 0)
            firstrow = 0;
        
        int lastcol = (clip.x + clip.width - insets.left + cellWidth - 1) / cellWidth;
        if (lastcol > numberOfColumns)
            lastcol = numberOfColumns;
        int lastrow = (clip.y + clip.height - insets.top + cellHeight - 1) / cellHeight;
        
        // uncomment below for debugging purposes
        /*
        java.awt.Color randomColor = new java.awt.Color
                (0.5f + (float)Math.random()/2.0f,
                 0.5f + (float)Math.random()/2.0f,
                 0.5f + (float)Math.random()/2.0f);
         */
        for (int row = firstrow; row < lastrow; row++) {
            // paint a single row
            for (int index = row*numberOfColumns+firstcol, col = firstcol;
            col < lastcol && index < count;
            col++, index++) {
                Graphics2D g2 = (Graphics2D)g.create();
                g2.translate(insets.left + cellWidth*col, insets.top + cellHeight*row);
                
                // uncomment below for debugging purposes
                /*
                g2.setColor (randomColor);
                g2.fillRect(0, 0,  cellWidth, cellHeight);
                 */
                cellRenderer.paintElement
                        (g2, this, dataModel.getElementAt(index), index,
                        selectionModel.isSelectedIndex(index),
                        isFocusOwner() &&
                        selectionModel.getLeadSelectionIndex() == index);
                
            }
        }
        
        
    }
    
    // overrides Component
    @Override public void doLayout() {
        // compute new number of columns
        int count = dataModel.getSize();
        if (count > 0) {
            Insets insets = getInsets();
            int cols = (getWidth() - insets.left - insets.right) / cellWidth;
            if (cols <= 0)
                cols = 1;
            setNumberOfColumns(cols);
        }
        super.doLayout();
    }
    
    // extends JComponent
    public Dimension getPreferredSize() {
        int count = dataModel.getSize();
        int rows = (count + numberOfColumns - 1) / numberOfColumns;
        Insets insets = getInsets();
        int width = cellWidth*numberOfColumns + insets.left + insets.right;
        int height = cellHeight*rows + insets.bottom + insets.top;
        Component parent = getParent();
        
        // adjust height to full viewport, when in scroll pane
        if (parent instanceof JViewport) {
            JViewport vp = (JViewport)parent;
            int vpheight = vp.getSize().height;
            if (vpheight > height)
                height = vpheight;
        }
        
        return new Dimension(width, height);
    }
    
    // extends JComponent
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }
    
    /**
     * Repaints the cell with given index.
     */
    private void repaintCell(int index) {
        Point pt = indexToLocation(index);
        if (pt != null) {
            repaint(pt.x, pt.y, cellWidth, cellHeight);
        }
    }
    
    /**
     * Repaint empty area.
     */
    private void repaintEmptyArea() {
        //int lastRow = (dataModel.getSize() + 1) / numberOfColumns;
        // bug, only apparent when numberOfColumns = 1
        int lastRow = dataModel.getSize() / numberOfColumns;
        int y = lastRow * cellHeight + getInsets().top;
        repaint(0, y, getWidth(), getHeight() - y);
    }
    
    /**
     * Scrolls the viewport to make the specified cell completely visible.
     * Only works when the list is in a scroll pane.
     *
     * @param index  the index of the cell to make visible
     */
    public void ensureIndexIsVisible(int index) {
        Rectangle cellBounds = getCellBounds(index, index);
        if (cellBounds != null) {
            scrollRectToVisible(cellBounds);
        }
    }
    
    /*============================================================
     * SELECTION PROXIES
     *============================================================*/
    
    
    /**
     * Returns an array of the values for the selected cells.
     * The returned values are sorted in increasing index order.
     */
    public Object[] getSelectedValues() {
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < 0) {
            return new Object[0];
        }
        
        Object[] res = new Object[iMax - iMin + 1];
        int count = 0;
        for(int i = iMin; i <= iMax; i++) {
            if (selectionModel.isSelectedIndex(i)) {
                res[count] = dataModel.getElementAt(i);
                count ++;
            }
        }
        if (res.length == count) {
            return res;
        } else {
            Object[] nres = new Object[count];
            System.arraycopy(res, 0, nres, 0, count);
            return nres;
        }
    }
    
    /**
     * Returns the first selected value, or <code>null</code> if the
     * selection is empty.
     */
    public Object getSelectedValue() {
        int i = selectionModel.getMinSelectionIndex();
        if (i < 0)
            return null;
        else
            return dataModel.getElementAt(i);
    }
    
    
    /*============================================================
     * SCROLLABLE
     *============================================================*/
    
    /**
     * Returns the distance to scroll vertically to expose the next or previous
     * row. If the top (resp. bottom) row was only partially exposed
     * it now becomes fully exposed.
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) {
        if (orientation == SwingConstants.HORIZONTAL) {
            // should not happen
            return 0;
        } else if (direction > 0) {
            // Scroll down
            int visibleY = visibleRect.y + visibleRect.height;
            int bottomRow = visibleY / cellHeight;
            int bottomY = bottomRow * cellHeight;
            if (bottomY < visibleY) {
                // bottom row not completely visible
                return cellHeight + bottomY - visibleY;
            } else {
                return cellHeight;
            }
        } else {
            // Scroll up
            int topRow = visibleRect.y / cellHeight;
            int topY = topRow * cellHeight;
            if (topY < visibleRect.y) {
                // top row not completely visible
                return visibleRect.y - topY;
            } else {
                return cellHeight;
            }
        }
    }
    
    
    /**
     * Returns the distance to scroll vertically to expose the next or previous block.
     * We are using the follows rules:
     * <ul>
     * <li>If scrolling down,
     * the last visible row should become the first completely
     * visible row.
     * <li>If scrolling up, the first visible row should become the last
     * completely visible row.
     * </ul>
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction) {
        if (orientation == SwingConstants.HORIZONTAL) {
            // should not happen
            return 0;
        } else if (direction > 0) {
            // Scroll down
            int visibleY = visibleRect.y + visibleRect.height;
            int bottomRow = visibleY / cellHeight;
            return bottomRow * cellHeight;
        } else {
            // Scroll up
            int topRow = visibleRect.y / cellHeight;
            int topY = topRow * cellHeight;
            return visibleRect.y + visibleRect.height - topY - cellHeight;
        }
    }
    
    // implements Scrollable
    public boolean getScrollableTracksViewportWidth() {
        return true;
    }
    
    // implements Scrollable
    public boolean getScrollableTracksViewportHeight() {
        return false;
    }
    
    // implements Scrollable
    public Dimension getPreferredScrollableViewportSize() {
        return getPreferredSize();
    }
    
    /*============================================================
     * LISTDATALISTENER
     *============================================================*/
    
    
    // implements ListDataListener
    public void intervalRemoved(ListDataEvent e) {
        if (selectionModel != null) {
            int index0 = e.getIndex0();
            selectionModel.removeIndexInterval(index0, e.getIndex1());
            revalidate(); // may change scroll bars
            for (int i = index0; i < dataModel.getSize(); i++)
                repaintCell(i);
            repaintEmptyArea();
        }
    }
    
    // implements ListDataListener
    public void intervalAdded(ListDataEvent e) {
        if (selectionModel != null) {
            int index0 = e.getIndex0();
            int index1 = e.getIndex1();
            selectionModel.insertIndexInterval(index0,  index1 - index0 - 1, true);
            revalidate(); // may change scroll bars
            for (int i = index0; i <= index1; i++)
                repaintCell(i);
        }
    }
    
    // implements ListDataListener
    public void contentsChanged(ListDataEvent e) {
        int index0 = e.getIndex0();
        int index1 = e.getIndex1();
        for (int i = index0; i <= index1; i++)
            repaintCell(i);
    }
    
    /*============================================================
     * FOCUS and SELECTION
     *============================================================*/
    
    /**
     * Inner class which handles focus events and selection events.
     */
    private class Handler implements FocusListener,
            ListSelectionListener {
        
        Handler  () {
            // avoids creation of accessor class
        }
        
        // implements FocusListener
        public void focusLost(FocusEvent e) {
            if (selectionModel != null)
                repaintCell(selectionModel.getLeadSelectionIndex());
        }
        
        // implements FocusListener
        public void focusGained(FocusEvent e) {
            if (selectionModel != null)
                repaintCell(selectionModel.getLeadSelectionIndex());
        }
        
        // implements ListSelectionListener
        public void valueChanged(ListSelectionEvent e) {
            int from = e.getFirstIndex();
            int to = e.getLastIndex();
            for (int index = from; index <= to; index++)
                repaintCell(index);
        }
        
    }
    
    /*============================================================
     * MOUSE and DRAG
     *============================================================*/
    
    /**
     * Inner class which handles mouse events (when not drag enabled).
     */
    private class MouseHandler extends MouseInputAdapter {
        
        MouseHandler  () {
            // avoids creation of accessor class
        }
        
        // implements MouseListener
        public void mousePressed(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e)) {
                selectionModel.setValueIsAdjusting(true);
                adjustFocus();
                adjustSelection(e);
            }
        }
        
        //
        public void mouseDragged(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e) &&
                    !e.isShiftDown() && ! e.isControlDown()) {
                int index = locationToIndex(e.getPoint());
                if (index >= 0)
                    ensureIndexIsVisible(index);
            }
        }
        
        // implements MouseListener
        public void mouseReleased(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e))
                selectionModel.setValueIsAdjusting(false);
        }
        
        // implements MouseListener
        public void mouseExited(MouseEvent e) {
            // do nothing
        }
        
        // implements MouseListener
        public void mouseEntered(MouseEvent e) {
            // do nothing
        }
        
        // implements MouseListener
        public void mouseClicked(MouseEvent e) {
            // do nothing
        }
        
        // implements MouseMotionListener
        public void mouseMoved(MouseEvent e) {
            // do nothing
        }
        
        /**
         * Request focus on the given component if it doesn't already have it
         * and <code>isRequestFocusEnabled()</code> returns true.
         */
        protected void adjustFocus() {
            if (!isFocusOwner() && isRequestFocusEnabled())
                requestFocusInWindow();
        }
        
        /**
         * Helper method with common code for {@code mousePressed} and
         * {@code mouseReleased}.
         */
        protected void adjustSelection(MouseEvent e) {
            
            int index = locationToIndex(e.getPoint());
            if (index >= 0)
                ensureIndexIsVisible(index);
            
            if (selectionModel == null)
                return;
            
            if (index < 0) {
                if (! e.isShiftDown() ||
                        selectionModel.getSelectionMode() == ListSelectionModel.SINGLE_SELECTION) {
                    selectionModel.clearSelection();
                }
            } else {
                
                int anchorIndex = selectionModel.getAnchorSelectionIndex();
                
                if (e.isControlDown()) {
                    if (e.isShiftDown() && anchorIndex >= 0) {
                        if (selectionModel.isSelectedIndex(anchorIndex)) {
                            selectionModel.addSelectionInterval(anchorIndex, index);
                        } else {
                            selectionModel.removeSelectionInterval(anchorIndex, index);
                            selectionModel.addSelectionInterval(index, index);
                            selectionModel.setAnchorSelectionIndex(anchorIndex);
                        }
                    } else if (selectionModel.isSelectedIndex(index)) {
                        selectionModel.removeSelectionInterval(index, index);
                    } else {
                        selectionModel.addSelectionInterval(index, index);
                    }
                } else {
                    if (e.isShiftDown() && anchorIndex >= 0) {
                        selectionModel.setSelectionInterval(anchorIndex, index);
                    } else {
                        selectionModel.setSelectionInterval(index, index);
                    }
                }
            }
        }
    }
    
    /**
     * Inner class which handles mouse events (when drag enabled).
     */
    private class MouseHandlerDragEnabled extends MouseHandler {
        
        MouseHandlerDragEnabled  () {
            // avoids creation of accessor class
        }
        
        // True if the start of a drag already selected the element pointed at
        private boolean dragPressDidSelection;
        
        // Mouse event which initiated dragging
        private MouseEvent dragStartEvent;
        
        
        // implements MouseListener
        public void mousePressed(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e)) {
                
                int index = locationToIndex(e.getPoint());
                
                dragStartEvent = null;
                if (index < 0)
                    adjustFocus();
                else {
                    int action = getDnDAction(e);
                    if (action == ACTION_NONE) {
                        adjustFocus();
                    } else {
                        dragPressDidSelection = false;
                        dragStartEvent = e;
                        if (e.isControlDown())
                            return;
                        else if (!e.isShiftDown() && selectionModel.isSelectedIndex(index)) {
                            selectionModel.addSelectionInterval(index, index);
                            return;
                        }
                        dragPressDidSelection = true;
                    }
                }
                
                adjustSelection(e);
            }
        }
        
        //
        public void mouseDragged(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e) &&
                    dragStartEvent != null) {
                int epsilon = DragSource.getDragThreshold();
                if (Math.abs(e.getX() - dragStartEvent.getX()) > epsilon ||
                        Math.abs(e.getY() - dragStartEvent.getY()) > epsilon) {
                    int action = getDnDAction(e);
                    if (action != ACTION_NONE) {
                        // this initiates a drag gesture!
                        if (e.isControlDown()) {
                            int index = locationToIndex(e.getPoint());
                            selectionModel.addSelectionInterval(index, index);
                        }
                        
                        getTransferHandler().exportAsDrag(CellList.this, dragStartEvent, action);
                        dragStartEvent = null;
                    }
                }
            }
        }
        
        // implements MouseListener
        public void mouseReleased(MouseEvent e) {
            if (isEnabled() && SwingUtilities.isLeftMouseButton(e) &&
                    dragStartEvent != null) {
                // mouse was pressed but not dragged
                adjustFocus();
                if (!dragPressDidSelection) {
                    // selection should be performed now
                    adjustSelection(dragStartEvent);
                }
            }
        }
    }
    
    /*============================================================
     * KEYBOARD ACTIONS
     *============================================================*/
    
    /**
     * Initialize the action map.
     */
    public final void initActionMap() {
        ActionMap actionMap = new ActionMap();
        actionMap.setParent(getActionMap());
        
        actionMap.put("selectPreviousRow",
                new ChangeSelection(-1, 0));
        actionMap.put("selectPreviousRowExtendSelection",
                new ExtendSelection(-1, 0));
        actionMap.put("selectPreviousRowChangeLead",
                new ChangeLead(-1, 0));
        actionMap.put("selectNextRow",
                new ChangeSelection(1, 0));
        actionMap.put("selectNextRowExtendSelection",
                new ExtendSelection(1, 0));
        actionMap.put("selectNextRowChangeLead",
                new ChangeLead(1, 0));
        actionMap.put("selectPreviousColumn",
                new ChangeSelection(0, -1));
        actionMap.put("selectPreviousColumnExtendSelection",
                new ExtendSelection(0, -1));
        actionMap.put("selectPreviousColumnChangeLead",
                new ChangeLead(0, -1));
        actionMap.put("selectNextColumn",
                new ChangeSelection(0, 1));
        actionMap.put("selectNextColumnExtendSelection",
                new ExtendSelection(0, 1));
        actionMap.put("selectNextColumnChangeLead",
                new ChangeLead(0, 1));
        actionMap.put("selectFirstRow",
                new ChangeSelection(-1, 0, MoveSelection.ABSOLUTE));
        actionMap.put("selectFirstRowExtendSelection",
                new ExtendSelection(-1, 0, MoveSelection.ABSOLUTE));
        actionMap.put("selectFirstRowChangeLead",
                new ChangeLead(-1, 0, MoveSelection.ABSOLUTE));
        actionMap.put("selectLastRow",
                new ChangeSelection(1, 0, MoveSelection.ABSOLUTE));
        actionMap.put("selectLastRowExtendSelection",
                new ExtendSelection(1, 0, MoveSelection.ABSOLUTE));
        actionMap.put("selectLastRowChangeLead",
                new ChangeLead(1, 0, MoveSelection.ABSOLUTE));
        
        actionMap.put("selectAll",
                new SelectAllAction());
        actionMap.put("clearSelection",
                new ClearSelectionAction());
        actionMap.put("addToSelection",
                new AddToSelection());
        actionMap.put("toggleAndAnchor",
                new ToggleAndAnchor());
        actionMap.put("extendTo",
                new ExtendSelection(0, 0));
        actionMap.put("moveSelectionTo",
                new ChangeSelection(0, 0));
        
        // SCROLL_ROW not yet implemented
        actionMap.put("scrollUp",
                new ChangeSelection(-1, 0, MoveSelection.SCROLL));
        actionMap.put("scrollUpExtendSelection",
                new ExtendSelection(-1, 0, MoveSelection.SCROLL));
        actionMap.put("scrollUpChangeLead",
                new ChangeLead(-1, 0, MoveSelection.SCROLL));
        actionMap.put("scrollDown",
                new ChangeSelection(1, 0, MoveSelection.SCROLL));
        actionMap.put("scrollDownExtendSelection",
                new ExtendSelection(1, 0, MoveSelection.SCROLL));
        actionMap.put("scrollDownChangeLead",
                new ChangeLead(1, 0, MoveSelection.SCROLL));
        
        actionMap.put("copy", TransferHandler.getCopyAction());
        actionMap.put("paste", TransferHandler.getPasteAction());
        actionMap.put("cut", TransferHandler.getCutAction());
        
        setActionMap(actionMap);
        
    }
    
    /**
     * Selects every element, except in single
     * selection mode, where it selects the lead element.
     */
    private class SelectAllAction extends AbstractAction {
        
        SelectAllAction  () {
            // avoids creation of accessor class
        }
        
        // implements Action
        public void actionPerformed(ActionEvent e) {
            if (selectionModel == null)
                return;
            int size = dataModel.getSize();
            if (size > 0) {
                if (selectionModel.getSelectionMode() == ListSelectionModel.SINGLE_SELECTION) {
                    int leadIndex = selectionModel.getLeadSelectionIndex();
                    if (leadIndex == -1)
                        leadIndex = 0;
                    ensureIndexIsVisible(leadIndex);
                    selectionModel.setSelectionInterval(leadIndex, leadIndex);
                } else {
                    selectionModel.setValueIsAdjusting(true);
                    int anchor = selectionModel.getAnchorSelectionIndex();
                    int lead = selectionModel.getLeadSelectionIndex();
                    selectionModel.setSelectionInterval(0, size - 1);
                    // restore anchor and lead
                    selectionModel.addSelectionInterval(anchor, lead);
                    selectionModel.setValueIsAdjusting(false);
                }
            }
        }
        
    }
    
    /**
     * Clears the selection.
     */
    private class ClearSelectionAction extends AbstractAction {
        
        ClearSelectionAction  () {
            // avoids creation of accessor class
        }
        
        // implements Action
        public void actionPerformed(ActionEvent e) {
            if (selectionModel != null)
                selectionModel.clearSelection();
        }
    }
    
    /**
     * Adds lead to the current selection.
     */
    private class AddToSelection extends AbstractAction {
        
        AddToSelection() {
            // avoids creation of accessor class
        }
        
        // implements Action
        public void actionPerformed(ActionEvent e) {
            if (selectionModel == null)
                return;
            int index = selectionModel.getLeadSelectionIndex();
            if (!selectionModel.isSelectedIndex(index)) {
                int oldAnchor = selectionModel.getAnchorSelectionIndex();
                selectionModel.setValueIsAdjusting(true);
                selectionModel.addSelectionInterval(index, index);
                selectionModel.setAnchorSelectionIndex(oldAnchor);
                selectionModel.setValueIsAdjusting(false);
            }
        }
        
    }
    
    /**
     * Toggles selection and anchors it.
     */
    private class ToggleAndAnchor extends AbstractAction {
        
        ToggleAndAnchor() {
            // avoids creation of accessor class
        }
        
        // implements Action
        public void actionPerformed(ActionEvent e) {
            if (selectionModel == null)
                return;
            int index = selectionModel.getLeadSelectionIndex();
            if (selectionModel.isSelectedIndex(index)) {
                selectionModel.removeSelectionInterval(index, index);
            } else {
                selectionModel.addSelectionInterval(index, index);
            }
        }
    }
    
    /**
     * Common super class of actions that move the selection.
     */
    private abstract class MoveSelection extends AbstractAction {
        
        public static final int RELATIVE = 0;
        
        public static final int ABSOLUTE = 1;
        
        public static final int SCROLL = 2;
        
        //
        private final int rowIncrement;
        
        //
        private final int colIncrement;
        
        //
        private final int type;
        
        //
        protected MoveSelection(int rowIncrement, int colIncrement, int type) {
            this.rowIncrement = rowIncrement;
            this.colIncrement = colIncrement;
            this.type = type;
        }
        
        // implements Action
        public void actionPerformed(ActionEvent e) {
            if (selectionModel == null)
                return;
            int index = selectionModel.getLeadSelectionIndex();
            int size = dataModel.getSize();
            switch (type) {
                case ABSOLUTE:
                    if (colIncrement < 0)
                        index -= index % numberOfColumns;
                    else if (colIncrement > 0)
                        index += numberOfColumns - 1 - index % numberOfColumns;
                    if (rowIncrement < 0) {
                        index = index % numberOfColumns;
                    } else if (rowIncrement > 0) {
                        index = size - 1 - (size - 1 - index) % numberOfColumns;
                    }
                    break;
                case SCROLL:
                    // TODO: implement this case
                    return ;
                default:
                    index += colIncrement;
                    index += numberOfColumns*rowIncrement;
            }
            if (index >= 0 && index < size) {
                if (rowIncrement != 0)
                    ensureIndexIsVisible(index);
                doSelection(index);
            }
        }
        
        // must be overridden
        protected abstract void doSelection(int index);
        
    }
    
    /**
     * Moves the selection by the given amount.
     */
    private class ChangeSelection extends MoveSelection {
        
        //
        public ChangeSelection(int rowIncrement, int colIncrement) {
            super(rowIncrement, colIncrement, RELATIVE);
        }
        
        //
        public ChangeSelection(int rowIncrement, int colIncrement, int type) {
            super(rowIncrement, colIncrement, type);
        }
        
        // implements Action
        @Override public void doSelection(int index) {
            selectionModel.setSelectionInterval(index, index);
        }
        
    }
    
    /**
     * Extends the selection by the given amount.
     */
    private class ExtendSelection extends MoveSelection {
        
        //
        public ExtendSelection(int rowIncrement, int colIncrement) {
            super(rowIncrement, colIncrement, RELATIVE);
        }
        
        //
        public ExtendSelection(int rowIncrement, int colIncrement, int type) {
            super(rowIncrement, colIncrement, type);
        }
        
        // implements Action
        @Override public void doSelection(int index) {
            int anchor = selectionModel.getAnchorSelectionIndex();
            if (anchor == -1)
                anchor = index;
            selectionModel.setSelectionInterval(anchor, index);
        }
        
    }
    
    /**
     * Moves the selection lead by the given amount.
     */
    private class ChangeLead extends MoveSelection {
        
        //
        public ChangeLead(int rowIncrement, int colIncrement) {
            super(rowIncrement, colIncrement, RELATIVE);
        }
        
        //
        public ChangeLead(int rowIncrement, int colIncrement, int type) {
            super(rowIncrement, colIncrement, type);
        }
        
        // implements Action
        @Override public void doSelection(int index) {
            if (selectionModel instanceof DefaultListSelectionModel &&
                    selectionModel.getSelectionMode() ==
                    ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
                ((DefaultListSelectionModel)selectionModel).moveLeadSelectionIndex(index);
            else
                selectionModel.setSelectionInterval(index, index);
        }
        
    }
    
    
    /*============================================================
     * DRAG AND DROP
     *============================================================*/
    
    //
    private boolean dragEnabled;
    
    /**
     * Indicate whether dragging on this component should be enabled.
     */
    public void setDragEnabled(boolean dragEnabled) {
        this.dragEnabled = dragEnabled;
        MouseInputListener newMouseHandler = dragEnabled ? mouseHandlerDragEnabled : mouseHandler;
        if (currentMouseHandler != newMouseHandler) { // NOPMD
            if (currentMouseHandler != null) {
                removeMouseListener(currentMouseHandler);
                removeMouseMotionListener(currentMouseHandler);
            }
            currentMouseHandler = newMouseHandler;
            addMouseListener(currentMouseHandler);
            addMouseMotionListener(currentMouseHandler);
        }
    }
    
    /**
     * Is dragging on this component enabled?
     */
    public boolean isDragEnabled() {
        return this.dragEnabled;
    }
    
    //
    private boolean dropInProgress;
    
    //
    private Point dropLocation;
    
    /**
     * Is a drag-and-drop operation in progress where
     * this component is a target?
     */
    public boolean isDropInProgress() {
        return dropInProgress;
    }
    
    /**
     * Return the last drag-and-drop location where this component is a target.
     */
    public Point getDropLocation() {
        return dropLocation;
    }
    
    //
    private DropListener dropListener; // NOPMD (bug?)
    
    //
    public void setDropTarget(DropTarget dt) {
        DropTarget oldDropTarget = getDropTarget();
        if (oldDropTarget != null) {
            oldDropTarget.removeDropTargetListener(dropListener);
        }
        if (dt != null) {
            if (dropListener == null)
                dropListener = new DropListener(); // lazy init
            try {
                dt.addDropTargetListener(dropListener);
            } catch (TooManyListenersException ex) {
                assert false : "Unexpected exception: " + ex;
            }
        }
        super.setDropTarget(dt);
    }
    
    /**
     * Follows movement of the drop location during drag-and-drop.
     */
    private class DropListener extends DropTargetAdapter {
        
        DropListener() {
            // avoids creation of accessor class
        }
        
        public void dragOver(DropTargetDragEvent dtde) {
            dropInProgress = true;
            dropLocation = dtde.getLocation();
        }
        
        public void drop(DropTargetDropEvent dtde) {
            dropInProgress = false;
        }
        
        public void dragEnter(DropTargetDragEvent dtde) {
            dropInProgress = true;
            dropLocation = dtde.getLocation();
        }
        
        public void dragExit(DropTargetEvent dte) {
            dropInProgress = false;
        }
    }
    
    /**
     * Translate mouse event into a dnd action constant;
     */
    private int getDnDAction(MouseEvent event) {
        TransferHandler transferHandler = getTransferHandler();
        if (transferHandler == null || !SwingUtilities.isLeftMouseButton(event))
            return ACTION_NONE;
        
        int sourceActions = transferHandler.getSourceActions(this);
        // look at mouse modifiers, if no modifiers present,
        // look at first source action allowed of MOVE, COPY and LINK
        if (event.isShiftDown()) {
            if (event.isControlDown())
                return ACTION_LINK & sourceActions;
            else
                return ACTION_MOVE & sourceActions;
        } else if (event.isControlDown()) {
            return ACTION_COPY & sourceActions;
        } else if ((sourceActions & ACTION_MOVE) == ACTION_MOVE)
            return ACTION_MOVE;
        else if ((sourceActions & ACTION_COPY) == ACTION_COPY)
            return ACTION_COPY;
        else if ((sourceActions & ACTION_LINK) == ACTION_LINK)
            return ACTION_LINK;
        else
            return ACTION_NONE;
    }
    
}
