/* DialogPanel.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */
         
package be.ugent.caagt.swirl.dialogs;

import java.awt.Component;
import java.awt.LayoutManager;
import java.util.ResourceBundle;
import javax.swing.Icon;
import javax.swing.JPanel;

/**
 * Common super class of all panels that are meant to be used as central panels
 * in a {@link SimpleDialog}. Extends <code>javax.swing.JPanel</code> with
 * several <code>createDialog</code> methods which create simple
 * dialogs with this panel as its central panel.
 * 
 * <p>
 * Clients will probably override the constructors to populate the dialog and
 * the method {@link #getTitleResourceKey} to provide an internationalized
 * dialog title
 * </p>
 * 
 * <p>
 * <a name="i18n"><b>Internationalization:</b></a> When the client does not
 * override <code>getTitleResourceKey</code>, the dialog title uses the
 * resource key <code>dialog.title</code>.
 * </p>
 */
public class DialogPanel extends JPanel {

    //
    private static final String RESOURCE_KEY = "dialog.title";

    /**
     * Create a dialog panel with the given layout manager.
     */
    public DialogPanel (LayoutManager a) {
        super(a);
    }

    /**
     * Return the resource key for the dialog title.
     */
    public String getTitleResourceKey () {
        return DialogPanel.RESOURCE_KEY;
    }

    /**
     * Create a new dialog, centered on the given component, with this as its
     * central panel.
     */
    public SimpleDialog createDialog (Component parent, String title, boolean modal) {
        return createDialog (parent, title, null, modal);
    }

    /**
     * Create a new dialog, centered on the given component, with this as its
     * central panel, displaying the given icon.
     */
    public SimpleDialog createDialog (Component parent, String title, Icon icon,
        boolean modal) {
        SimpleDialog dialog = SimpleDialog.createSimpleDialog (parent, title,
                modal);
        dialog.setCentralComponent (this);
        dialog.init ();
        dialog.setIcon (icon);
        return dialog;
    }

    /**
     * Create a new dialog, centered on the given component, with this as its
     * central panel.
     */
    public SimpleDialog createDialog (Component parent, ResourceBundle bundle,
        boolean modal) {
        return createDialog (parent, bundle, null, modal);
    }
    
    /**
     * Create a new dialog, centered on the given component, with this as its
     * central panel, displaying the given icon.
     */
    public SimpleDialog createDialog (Component parent, ResourceBundle bundle, Icon icon,
        boolean modal) {
        return createDialog (parent, bundle.getString (getTitleResourceKey ()),
            modal);
    }
}
