/* RepeaterCommand.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.commands;

import be.ugent.caagt.swirl.ActionRepeater;

import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.event.ChangeListener;

import org.pietschy.command.ActionCommand;
import org.pietschy.command.CommandManager;

/**
 * Extension of {@link ActionCommand} which creates 'repeater buttons'.
 * Buttons of this type execute some action repeatedly
 * while the button is being held down.<p>
 * The action to be repeated
 * is specified in {@link #handleRepeat}. When the button is first pressed
 * {@link #handleFirst} is called, before {@code handleRepeat}. Finally, if
 * the button is released
 * within the button boundaries {@link #handleExecute} is
 * called, as with {@link ActionCommand}, and if the mouse button is released
 * outside the button boundaries, {@link #handleCancel} is invoked.<p>
 * Only has any effect on buttons of type {@link JButton}, not on toggle buttons or
 * menu items.<p>
 *
 * <b>Important note:</b> work on this part of the Swirl library will be 
 * discontinued because unfortunately newer versions
 * of the <i>GUI commands</i> framework will no longer be available as <i>open source</i>.
 * In due course this class will become deprecated and will eventually be
 * removed from the library.
 */
public abstract class RepeaterCommand extends ActionCommand {
    
    private class Repeater extends ActionRepeater {
        
        public Repeater() {
            super();
            setInterval(interval); // interval of outer class
        }
        
        //
        public void doAction() {
            handleRepeat();
        }
        
        //
        public void buttonFirstPressed() {
            handleFirst();
        }
        
        //
        public void buttonPressCanceled() {
            handleCancel();
        }
        
        //
        public boolean isEnclosedBy(Object obj) {
            return obj == RepeaterCommand.this;
        }
    }
    
    //
    public RepeaterCommand() {
        this.interval = 150;
    }
    
    //
    public RepeaterCommand(String commandId) {
        super(commandId);
        this.interval = 150;
    }
    
    //
    public RepeaterCommand(CommandManager commandManager, String commandId) {
        super(commandManager, commandId);
        this.interval = 150;
    }
    
    //
    private int interval;
    
    /**
     * Set the interval between repeats. Only affects buttons which are still
     * to be created.
     */
    public void setInterval(int interval) {
        this.interval = interval;
    }
    
    /**
     * Method to be executed repeatedly while the button is pressed.
     */
    public abstract void handleRepeat() ;
    
    /**
     * Method to be executed when the button is first pressed.
     */
    public abstract void handleFirst();
    
    /**
     * Method to be executed when the button press is cancelled.
     */
    public abstract void handleCancel();
    
    //
    public void attach(AbstractButton button, String faceName) {
        super.attach(button, faceName);
        if (button instanceof JButton)
            new Repeater().registerWith(button);
    }
    
    //
    public void detach(AbstractButton button) {
        if (button instanceof JButton) {
            // search for a change listener of the correct type and with
            // the correct enclosing instance
            for (ChangeListener listener : button.getChangeListeners()) {
                if (listener instanceof Repeater && ((Repeater)listener).isEnclosedBy(this)) {
                    button.removeChangeListener(listener);
                    break; // at most one
                }
            }
            
            super.detach(button);
        }
        
    }
}
