/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  File:   spfquery.c
*  Desc:   SPF Query Tool (an example implementation of libSPF)
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/

#include "spfquery.h"


/* SPF_usage
*
*  Author: James Couzens <jcouzens@6o4.ca>
*
*  Date:   07/04/04
*
*  Desc:
*         Main function, allocates memory and makes calls to the libSPF 
*  library which parses the "fake" query.
*
*/
int main(int argc, char *argv[])
{
  u_int8_t i         = 0;       /* utility */
  u_int8_t res       = 0;       /* libSPF result code */
  
  char     *margv    = NULL;    /* pointer to current argv element */
  char     *ip       = NULL;    /* ip address to test connecting from */
  char     *address  = NULL;    /* email address to test sending from */
  char     *helo     = NULL;    /* helo hostname to test sending from */

  char     *tmp      = NULL;    /* utility pointer */
  
  peer_info_t *pinfo = NULL;    /* libSPF peer_info structure */

  if (argc <= 1)
  {
    SPF_usage();
    return(FALSE);
  }

  for (i = 1; i < argc; i++)
  {
    tmp = argv[i];

    if (*tmp == '-')
    {
      margv = (tmp + 3);
      
      switch (*(tmp + 1))
      {
        case 'd' :
          confg.level = atoi(margv);
          break;
        case 'i' :
          ip  = strdup(margv);
          break;
        case 's' :
          address = strdup(margv);
          break;
        case 'h' :
          helo  = strdup(margv);
          break;
      }
    }
  }
 
  if (ip == NULL)
  {
    printf("You need to specify an IP Address to test against\n\n");
    SPF_usage();
    return(FALSE);
  }
  else if (address == NULL)
  {
    printf("You need to specify a from email address\n\n");
    SPF_usage();
    return(FALSE);
  }
  else if (helo == NULL)
  {
    helo = strdup(HELO_HOST);
    printf("You didn't give me a helo host, using (%s)\n", helo);
    return(FALSE);
  }

  if (confg.level >= 1)
  {
    printf("SPF Query v%s - James Couzens <jcouzens@codeshare.ca>\n\n", 
      SPFQUERY_VERSION);
   
    printf("DEBUG: %u\n", confg.level);
    printf("IP address: %s\n", ip);
    printf("MAIL FROM: %s\n", address);
    printf("HELO: %s\n", helo);
  }
  
  if ((pinfo = SPF_init(helo, ip, NULL, NULL, NULL, FALSE, FALSE)) != NULL)
  {
    /* perform fake HELO */
    SPF_smtp_helo(pinfo, helo);
    
    /* perform fake MAIL FROM */
    SPF_smtp_from(pinfo, address);
    
    /* perform SPF parse */
    pinfo->RES = SPF_policy_main(pinfo);
    
    /* assign result of SPF parse */
    res = pinfo->RES;

    /* print the results */
    printf( "%s\n%s\n%s\n",
      pinfo->rs ? pinfo->rs : "NULL",
      pinfo->error ? pinfo->error : "NULL",
      pinfo->explain ? pinfo->explain : "NULL");

    /* close SPF session (free memory associated with parse) */
    SPF_close(pinfo);

  }   
  
  free(ip);
  free(address);
  free(helo);
  
  return(FALSE); 
}


/* SPF_usage
*
*  Author: James Couzens <jcouzens@6o4.ca>
*
*  Date:   12/25/03
*
*  Desc:
*         Displays usage help information when the binary is called with
*  no arguments.
*
*/
void SPF_usage()
{
  printf("Usage:\n");
  printf("\n");
  printf("spfquery [dish]\n");
  printf("\n");
  printf("-d [x]     - DEBUG where x is a number between 1 and 255\n");
  printf("-i [addr]  - IP Address where the fake connection will come from\n");
  printf("-s [email] - What email address to test with\n");
  printf("-h [host]  - HELO hostname to test with\n");
  printf("\n");
  printf("spfquery -i 10.0.0.2 -s jcouzens@6o4.ca -h spftools.net\n");

  return;
}

/* end spfquery.c */
