/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Author: Sean Comeau   <scomeau@obscurity.org>
*
*  File:   util.c
*  Desc:   Utility functions
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/

#include "macro.h"
#include "util.h"

/* MACRO_expand
*
*  Author: Sean Comeau <scomeau@obscurity.org>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:    01/18/04
*  Updated: 06/29/04 Roger Moser <Roger.Moser@pamho.net>
*
*  Desc:
*        Walks a string of macros tokenizing and expanding them along the
*  way, each token being inserted as a node into a linked list.  Once the
*  walk is complete the list is walked and the nodes are copied back out
*  in the order they were added as expanded macros into a string which is
*  then passed back to the calling function.
*
*  MACRO_freebuf is removed and I moved the code inside of this function since
*  before calling it, it was being walked anyway to copy out each string into
*  the return buffer, so now it just free's directly after and then destructs.
*
*/
char *MACRO_expand(peer_info_t *peer_info, const char *s)
{
  const char * const  macro_p = "%";     /* % (literal) */
  const char * const  macro_d = "%20";   /* "URL" space */
  const char * const  macro_u = " ";     /* whitespace */

  char                *buf;              /* return buffer */
  char                *p;                /* working pointer */
  char                *cp;               /* working pointer */
  char                *macro;
  char                *s_macro;          /* single char macro */

  strbuf_t            *master;           /* list pointers */
  strbuf_node_t       *c_node;           /* c_node node */
  strbuf_node_t       *kill_node;        /* temp node used in destruction */

  size_t              len;               /* length of passed string */
  size_t              i = 0;             /* for obtaining an index on a string */
  size_t              length = 0;        /* overall length of expanded string */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  len = strlen(s);
  p = cp = xstrndup(s, (len + 1));

  master = xmalloc(SIZEOF(strbuf_t));
  master->head      = NULL;
  master->elements  = 0;
  length            = 0;

  while (*p)
  {
     /*
     * This works by moving through the string and replacing non-essential
     * elements with NULL chars until the character designating an expansion
     * mechanism is found.  The character is then sent off to MACRO_process
     * for expansion
    */
    if (*p == '%') /* start of macro */
    {
      switch (*(p + 1))
      {
        case '%':

          /* convert %% into % */
          if (MACRO_addbuf(master, (char *)macro_p, 1) == FALSE)
          {
            xvprintf("Unable to allocate list node with (%c)!\n", macro_p);
            return(NULL);
          }

          p += 2;
          length++;
          break;

        case '_':

          /* convert %_ to a white space */
          if (MACRO_addbuf(master, (char *)macro_u, 1) == FALSE)
          {
            xvprintf("Unable to allocate list node with (%c)!\n", macro_u);
            return(NULL);
          }

          p += 2;
          length++;
          break;

        case '-':

          /* convert %- into URL encoded '%20' */
          if (MACRO_addbuf(master, (char *)macro_d, 3) == FALSE)
          {
            xvprintf("Unable to allocate list node with (%s)!\n", macro_d);
            return(NULL);
          }

          p += 2;
          length += 3;
          break;

        case '{':

          *p++ = '\0'; /* % */
          *p++ = '\0'; /* { */

          if ((i = UTIL_index(p, '}')) == 0)
          {
            xvprintf("'}' Invalid Macro (%c)\n", *(s + 1));
            return(NULL);  /* not closed, invalid macro */
          }

          *(p + i) = '\0'; /* } */

          xvprintf("Actual macro (%s)\n", p);
          if ((macro = MACRO_process(peer_info, p, (i + 1))) == NULL)
          {
            xeprintf("macro process returned null!\n");
          }
          else
          {
            length += strlen(macro);
            xvprintf("Macro expanded to: (%s) %i bytes\n", macro,
              strlen(macro));

            if (MACRO_addbuf(master, macro, strlen(macro)) == FALSE)
            {
              xvprintf("Unable to allocate list node with (%s)!\n", macro);
              xfree(macro);
              return(NULL);  /* not closed, invalid macro */
            }
            xfree(macro);
          }
          p += i;
          break;

        default:

          xvprintf("ERROR: Invalid macro.  Abort!\n", *(p + 1));
          /* need cleanup function call perhaps */
          return(NULL);
          break;

      } /* switch */
    } /* if */
    else
    {
      if ((i = UTIL_index(p, '%')) == 0)
      {
        while (*(p + i))
        {
          i++;
        }
        s_macro = xmalloc(i + 1);
        memset(s_macro, '\0', (i + 1));
        memcpy(s_macro, p, (i + 1));
      }
      else
      {
        s_macro = xmalloc(i + 1);
        memset(s_macro, '\0', (i + 1));
        memcpy(s_macro, p, i);
      }

      length += i;

      if (MACRO_addbuf(master, s_macro, (i + 1)) == FALSE)
      {
        xvprintf("Unable to allocate list node with (%s)!\n", s_macro);
        return(NULL);
      }

      p += (i - 1);
      xvprintf("Freeing s_macro temp buf (%s)\n", s_macro);
      xfree(s_macro);
    }
    p++;
    xvprintf("Remaining buffer (%s)\n", p);
  } /* while */

  xprintf("Allocated %i bytes for return buf\n", length);
  buf = xmalloc(length + 1);
  memset(buf, '\0', length);

  c_node = master->head;
  while (c_node != NULL)
  {
    kill_node = c_node;

    if (kill_node->len > 1)
    {
      xvprintf("NODE: (%s) LEN: %i\n", kill_node->s, kill_node->len);
    }
    else
    {
      xvprintf("NODE: (%c) LEN: %i\n", kill_node->s, kill_node->len);
    }

    strncat(buf, kill_node->s, kill_node->len);
    xfree(kill_node->s);
    c_node = c_node->next;
    xfree(kill_node);
  }

  xfree(cp);
  xfree(master);

  xvprintf("Returning expanded macro: (%s)\n", buf);

  return(buf);
}


/* MACRO_process
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/21/04
*
*  Desc:
*         This function takes a NULL terminated string containing only one
*  macro. It returns a new string containing whatever that macro expanded to.
*  Returns NULL if the macro can not be expanded.
*
*  exists:%{ir}.%{l1r+-}._spf.%{d}
*
*/
char *MACRO_process(peer_info_t *peer_info, char *macro, const size_t size)
{
  int     c;    /* stores a lower case version of the macro if necessary */
  size_t  i;    /* utility for string lengths */

  char    *rev_addr;    /* used by p macro */

  if (macro == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(FALSE);
  }

  xprintf("called with (%s) and len: %i\n",macro, size);

  rev_addr = NULL;
  i = 0;

  if (isupper(*macro))
  {
    c = tolower(*macro);
  }
  else
  {
    c = *macro;
  }

  switch (c)
  {
    case 'd':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->cur_dom));
      }
      else
      {
        xvprintf("'d' expands to: (%s)\n", peer_info->cur_dom);
        return(xstrndup(peer_info->cur_dom, (strlen(peer_info->cur_dom) + 1)));
      }
    case 'h':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->helo));
      }
      else
      {
        xvprintf("'h' expands to: (%s)\n", peer_info->helo);
        if (peer_info->helo != NULL)
        {
          return(xstrndup(peer_info->helo,
            (strlen(peer_info->helo) + 1)));
        }
        else
        {
          return(xstrndup(peer_info->ehlo,
            (strlen(peer_info->ehlo) + 1)));
        }
      }
    case 'i':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->r_ip));
      }
      else
      {
        xvprintf("'i' expands to: (%s)\n", peer_info->r_ip);
        return(xstrndup(peer_info->r_ip,
          (strlen(peer_info->r_ip) + 1)));
      }
    case 'l':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->local_part));
      }
      else
      {
        xvprintf("'l' expands to: (%s)\n", peer_info->local_part);
        return(xstrndup(peer_info->local_part,
          (strlen(peer_info->local_part) + 1)));
      }
    case 'o':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->cur_dom));
      }
      else
      {
        xvprintf("'o' expands to: (%s)\n", peer_info->cur_dom);
        return(xstrndup(peer_info->cur_dom,
         (strlen(peer_info->cur_dom) + 1)));
      }
    case 'p':
      xfree(peer_info->r_vhname);

      if (UTIL_validate_ptr(peer_info) == FALSE)
      {
        peer_info->r_vhname = xmalloc(8);
        snprintf(peer_info->r_vhname, 8, "unknown");
      }

      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->r_vhname));
      }
      else
      {
        xvprintf("'p' expands to: (%s)\n", peer_info->r_vhname);
        return(xstrndup(peer_info->r_vhname,
          (strlen(peer_info->r_vhname) + 1)));
      }
    case 's':
      if (peer_info->cur_eaddr != NULL || peer_info->cur_eaddr)
      {
        xfree(peer_info->cur_eaddr);
      }

      xprintf("local: (%s) cur dom (%s)\n", peer_info->local_part,
        peer_info->cur_dom);

      i = (strlen(peer_info->local_part) + strlen(peer_info->cur_dom) + 2);
      peer_info->cur_eaddr = xmalloc(i);
      memset(peer_info->cur_eaddr, '\0', i);
      snprintf(peer_info->cur_eaddr, i, "%s@%s", peer_info->local_part,
        peer_info->cur_dom);

      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->cur_eaddr));
      }
      else
      {
        xvprintf("'s' expands to: (%s)\n", peer_info->cur_eaddr);
        return(xstrndup(peer_info->cur_eaddr, (i + 1)));
      }
    case 't':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->utc_time));
      }
      else
      {
        xvprintf("'t' expands to: (%s)\n", peer_info->utc_time);
        return(xstrndup(peer_info->utc_time, UTC_TIME));
      }
    case 'v':
      if (*(macro + 1))
      {
        return(MACRO_eatmore(macro, peer_info->ip_ver));
      }
      else
      {
        xvprintf("'v' expands to: (%s)\n", peer_info->ip_ver);
        return(xstrndup(peer_info->ip_ver,
          (strlen(peer_info->ip_ver) + 1)));
      }
    case 'x':
      if (size > 1)
      {
        if (*(macro + 1) == 'R' || *(macro + 1) == 'r')
        {
          return(xstrndup(peer_info->mta_hname,
            (strlen(peer_info->mta_hname) + 1)));
        }
      }
      break;
    default:
      return(xstrndup(macro, (strlen(macro) + 1)));
  } /* switch */

  return(NULL);
}


/* MACRO_eatmore
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/22/04
*  Date:   07/02/04 James Couzens <jcouzens@codeshare.ca> (compiler warnings)
*
*  Desc:
*         This function is called whenever a macro is more than one character
*  long and so we have to 'eatmore' ;-)  macro is the unexpanded macro and
*  s is the expanded string of the original macro would have been returned as
*  had it been single.  Returns the expanded macro using dynamically allocated
*  memory upon success and in the process will FREE s after allocating more
*  memory (if necessary) and copying it over.  Returns NULL upon failure.
*
*  NOT TESTED
*/
char *MACRO_eatmore(char *macro, char *s)
{
  size_t      i;            /* how much of the buffer we are using */

  char        *cp;          /* working pointer */

  char        *buf;         /* return buffer */
  char        *rev_str;     /* temporary buffer */
  char        *d_buf;       /* temporary buffer */

  u_int8_t    n_dot;        /* number of 'delimiter' items in a string */

  u_int8_t    rev;          /* string to be reversed */
  u_int8_t    digit;        /* macro is a digit modifier */
  char        delim;        /* marco is a delimiter modifier */

  if (macro == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(NULL);
  }

  xprintf("Called with macro (%s) and string (%s)\n", macro, s);

  /* need a proper value for this not just MAX_MACRO_LEN need to know the MAXIMUM
   * expandable length of a macro. */

  cp      = macro;
  rev     = 0;
  digit   = 0;
  delim   = '.';
  buf     = NULL;
  rev_str = NULL;
  
  while (*cp)
  {
    if (isdigit(*cp))
    {
      digit = atoi(cp);
    }
    else if (UTIL_is_spf_delim(*cp) == TRUE)
    {
      delim = *cp;
    }
    else if (*cp == 'r' || *cp == 'R')
    {
      rev = 1;
    }
    cp++;
  }

  xvprintf("mac:(%s) r:(%i) dig:(%i) dlm: (%c)\n",
    macro, rev, digit, delim);

  i = 0;
  /* reverse the string */
  if (rev == 1)
  {
    /*delim = '.';*/
    rev_str = UTIL_reverse(s, delim);
    s = NULL;
  }

  if (s == NULL)
  {
    cp = rev_str;
  }
  else
  {
    cp = s;
  }

  /* exercise digit modifier on string */
  if (digit > 0)
  {
    n_dot = UTIL_count_delim(cp, delim);

    if (digit > n_dot)
    {
      digit = n_dot;
    }

    if ((d_buf = UTIL_split_strr(cp, delim, digit)) != NULL)
    {
      i = strlen(d_buf);
    }
    else
    {
      d_buf = cp;
      i = strlen(d_buf);
    }

    buf = xmalloc(i + 1);
    memset(buf, '\0', (i + 1));
    memcpy(buf, d_buf, (i + 1));

    if (d_buf != cp)
    {
      xfree(d_buf);
    }
  }
  else if (rev == 1)
  {
    buf = xstrndup(rev_str, (strlen(rev_str) + 1));
  }

  xvprintf("Returning (%s) (%i bytes)\n", buf, strlen(buf));

  if (rev == 1)
  {
    xfree(rev_str);
  }

  return(buf);
}


/* MACRO_addbuf
*
*  Author: Sean Comeau <scomeau@obscurity.org>
*  Author: James Couznes <jcouzens@codeshare.ca>
*
*  Date:    01/18/04
*  Updated: 01/24/04
*
*
*  Desc:
*         Appends nodes to a master list which is passed of type
*  strbuf_t.  The nodes are of type strbuf_node_t and appended on
*  the end and the list is reordered to reflect this.  Returns
*  TRUE upon success and FALSE
*  upon failure.
*
*/
SPF_BOOL MACRO_addbuf(strbuf_t *master, char *s, size_t size)
{
  strbuf_node_t *c_node     = NULL;  /* c_node working node */
  strbuf_node_t *new_node   = NULL;  /* newly allocated node */
  strbuf_node_t *prev_node  = NULL;  /* previous working node */

  if (s == NULL)
  {
    xeprintf("Passed a NULL string.  Abort!\n");
    return(FALSE);
  }

  xvprintf("Called with (%s) %i (%i) bytes.\n", s, size, strlen(s));

  new_node    = xmalloc(SIZEOF(strbuf_node_t));
  new_node->s = xmalloc(size + 1);

  memset(new_node->s, '\0', (size + 1));
  strncpy(new_node->s, s, size);
  new_node->len   = size;
  new_node->next  = NULL;

  xvprintf("Added (%s) to node of len: %i)\n", new_node->s,
    new_node->len);

  prev_node = NULL;
  c_node    = master->head;

  /* reorder the list with the NEW element on the end */
  while (c_node != NULL)
  {
    prev_node = c_node;
    c_node    = c_node->next;
  }

  if (prev_node != NULL)
  {
    new_node->next  = prev_node->next;
    prev_node->next = new_node;
  }
  else
  {
    master->head = new_node;
  }

  master->elements++;

  return(TRUE);
}

/* end macro.c */

