/*
 * Emulating:
 *  getipnodebyname , getipnodebyaddr , copyhostent , freehostent
 *   
 * 
 * Some of these routines are not implemented properly. At getipnodebyname(), flags
 * will be processed as if it were "0" value, whatever value it holds.
 */



#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>




/* Warning, these values may not be properly set !!! . Just for future fixes 
 * getipnodebyname does not implement the "flag" parameter. It is there for compatibility.
 */

#define AI_V4MAPPED	2048
#define AI_ADDRCONFIG	1024
#define AI_ALL		(AI_V4MAPPED | AI_ADDRCONFIG)

/*
 * free a malloced hostent
 */

void
freehostent (struct hostent *h)
{
    char **p;

    free (h->h_name);
    if (h->h_aliases != NULL) {
	for (p = h->h_aliases; *p != NULL; ++p)
	    free (*p);
	free (h->h_aliases);
    }
    if (h->h_addr_list != NULL) {
	for (p = h->h_addr_list; *p != NULL; ++p)
	    free (*p);
	free (h->h_addr_list);
    }
    free (h);
}

/*
 * Create a dynamically malloced hostent form static hostent variable
 * Only used by getipnodebyname, getipnodebyaddr...
 */
struct hostent *
copyhostent (const struct hostent *h)
{
    struct hostent *res;
    char **p;
    int i, n;

    res = malloc (sizeof (*res));
    if (res == NULL)
	return NULL;
    res->h_name      = NULL;
    res->h_aliases   = NULL;
    res->h_addrtype  = h->h_addrtype;
    res->h_length    = h->h_length;
    res->h_addr_list = NULL;
    res->h_name = strdup (h->h_name);
    if (res->h_name == NULL) {
	freehostent (res);
	return NULL;
    }
    for (n = 0, p = h->h_aliases; *p != NULL; ++p)
	++n;
    res->h_aliases = malloc ((n + 1) * sizeof(*res->h_aliases));
    if (res->h_aliases == NULL) {
	freehostent (res);
	return NULL;
    }
    for (i = 0; i < n + 1; ++i)
	res->h_aliases[i] = NULL;
    for (i = 0; i < n; ++i) {
	res->h_aliases[i] = strdup (h->h_aliases[i]);
	if (res->h_aliases[i] == NULL) {
	    freehostent (res);
	    return NULL;
	}
    }

    for (n = 0, p = h->h_addr_list; *p != NULL; ++p)
	++n;
    res->h_addr_list = malloc ((n + 1) * sizeof(*res->h_addr_list));
    if (res->h_addr_list == NULL) {
	freehostent (res);
	return NULL;
    }
    for (i = 0; i < n + 1; ++i) {
	res->h_addr_list[i] = NULL;
    }
    for (i = 0; i < n; ++i) {
	res->h_addr_list[i] = malloc (h->h_length);
	if (res->h_addr_list[i] == NULL) {
	    freehostent (res);
	    return NULL;
	}
	memcpy (res->h_addr_list[i], h->h_addr_list[i], h->h_length);
    }
    return res;
}



/*
 * lookup `name' (address family `af') in DNS,DNS6 returning a pointer
 * to a malloced struct hostent or NULL.
 */

struct hostent *
getipnodebyname (const char *name, int af, int flags, int *error_num)
{
    struct hostent *tmp;

#ifdef HAVE_GETHOSTBYNAME2
    tmp = gethostbyname2 (name, af);
#else
    if (af != AF_INET) {
	*error_num = NO_ADDRESS;
	return NULL;
    }
    tmp = gethostbyname (name);
#endif
    if (tmp == NULL) {
	switch (h_errno) {
	case HOST_NOT_FOUND :
	case TRY_AGAIN :
	case NO_RECOVERY :
	    *error_num = h_errno;
	    break;
	case NO_DATA :
	    *error_num = NO_ADDRESS;
	    break;
	default :
	    *error_num = NO_RECOVERY;
	    break;
	}
	return NULL;
    }
    tmp = (struct hostent *) copyhostent (tmp);
    if (tmp == NULL) {
	*error_num = TRY_AGAIN;
	return NULL;
    }
    return tmp;
}

/*
 * lookup `src, len' (address family `af') in DNS,DNS6 returning a pointer
 * to a malloced struct hostent or NULL.
 */

struct hostent *
getipnodebyaddr (const void *src, size_t len, int af, int *error_num)
{
    struct hostent *tmp;

    tmp = gethostbyaddr (src, len, af);
    if (tmp == NULL) {
	switch (h_errno) {
	case HOST_NOT_FOUND :
	case TRY_AGAIN :
	case NO_RECOVERY :
	    *error_num = h_errno;
	    break;
	case NO_DATA :
	    *error_num = NO_ADDRESS;
	    break;
	default :
	    *error_num = NO_RECOVERY;
	    break;
	}
	return NULL;
    }
    tmp = copyhostent (tmp);
    if (tmp == NULL) {
	*error_num = TRY_AGAIN;
	return NULL;
    }
    return tmp;
}
