# $DUH: Modules.pm,v 1.3 2002/12/16 15:35:39 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Modules - utility functions for modules

=head1 SYNOPSIS

    use PMilter::Modules;

=head1 DESCRIPTION

This is a utility package of subroutines useful for PMilter module
development.  Subroutines can be imported individually; all are exported by
default.

=head1 SUBROUTINES

=over 4

=cut

package PMilter::Modules;
use base Exporter;

use strict;
use warnings;

use Carp;
use PMilter;
use UNIVERSAL;

*VERSION = *PMilter::VERSION;

# Exported names

our @EXPORT_OK = qw(
	globrx
	list
	regex_list
	shift_errmsg
);

our @EXPORT = @EXPORT_OK;

=pod

=item globrx(GLOB[, GLOB ...])

Translates a glob-style string (using the ? and * wildcard characters) into
a regular expression bound to beginning and end of string.  Can be given
more than one glob and list context, which will return a list of all globs
(translated).

=cut

sub globrx (@) {
	my @regexes = map {
		my $s = $_;

		# escape all special chars but ? and *
		$s =~ s/([\.\(\)\[\]\{\}\^\$\|\+\\])/\\$1/g;
		$s =~ s/\?/./g;
		$s =~ s/\*/.*/g;
		$s =~ s/^/\^/;
		$s =~ s/$/\$/;

		$s
	} @_;

	return $regexes[0] unless wantarray;
	@regexes;
}

=pod

=item list(VALUE[, VALUE, ...])

Always returns a list; should only be used in list context.

Creates a list consisting of all VALUEs.  If any VALUE is an arrayref,
expands its contents instead.  Does not go deeper; only expands the first
level of arrayrefs.

=cut

sub list (@) {
	my @list = ();

	foreach my $value (@_) {
		if (UNIVERSAL::isa($value, 'ARRAY')) {
			push(@list, @$value);
		} else {
			push(@list, $value);
		}
	}

	@list;
}

=item regex_list(VALUE[, MODIFIERS])

Always returns a list; should only be used in list context.

If VALUE is a scalar, compiles it as a regular expression and returns it.  
If VALUE is an arrayref, compiles each element as a regular expression and
returns a list containing all compiled values.

If MODIFIERS is specified, it is a string of one or more characters from
"imsx", indicating what modifiers should be applied to each compiled regular
expression.  More technically, however, MODIFIERS is a string that is placed
at the start of each regex as C<(?MODIFIERS)> (see the C<(?imsx-imsx)>
operator in L<perlre>).

=cut

sub regex_list ($;$) {
	my $list = shift || return undef;
	my $modifiers = shift || '';

	$modifiers = "(?$modifiers)" if ($modifiers ne '');

	map qr/$modifiers$_/, list($list);
}

=pod

=item shift_errmsg(LIST, DEFAULT)

Shifts two items off of LIST (passed automatically by reference as if with
C<shift>), to implement the [, ERRMSG[, EXTERRMSG]] arguments of many
modules.

Creates an error string consisting of "ERRMSG - EXTERRMSG", where ERRMSG is
DEFAULT if the first shifted argument is undef, and " - EXTERRMSG" is left
off completely if the second shifted argument is undef.

=cut

sub shift_errmsg (\@$) {
	my $listref = shift;
	my $default = shift;

	my $errmsg = shift @$listref || $default;
	my $exterrmsg = shift @$listref;
	$errmsg .= " - $exterrmsg" if ($exterrmsg);

	$errmsg;
}

1;

__END__

=back

=head1 SEE ALSO

L<PMilter::Callbacks> on how to create standalone milter modules
