# $DUH: List.pm,v 1.4 2002/12/16 21:44:40 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::DNSBL::List - list/chain of sequentially queried DNSBLs

=head1 SYNOPSIS

    my $dnsbl = new PMilter::DNSBL::List($dnsbl1, $dnsbl2);

=head1 DESCRIPTION

This chains multiple DNSBLs into a sequential query list, querying each
element in sequence until one of them returns a value other than C<undef>.
Elements of the list can be any class, so long as they are blessed
references that provide a C<query> method.  Alternatively, arguments may be
simple scalars (passed to PMilter::DNSBL->new()) or arrayrefs (dereferenced,
and the whole list passed as args to PMilter::DNSBL->new()).

A PMilter::DNSBL::List object can be used just like a single DNSBL; it also
has a C<query> method that functions identically to its description in
L<PMilter::DNSBL>.

=cut

package PMilter::DNSBL::List;

use strict;
use warnings;

use Carp;
use PMilter::DNSBL;
use UNIVERSAL;

sub new {
	my $class = shift || confess 'invoked new as non-class';
	my $this = bless {}, $class;

	foreach my $x (@_) {
		my $dnsbl = $x; # make modifiable

		if (ref($dnsbl) eq 'ARRAY') {
			$dnsbl = PMilter::DNSBL->new(@$dnsbl);
		} elsif (!ref($dnsbl)) {
			$dnsbl = PMilter::DNSBL->new($dnsbl);
		}

		confess 'object does not have "query" method' unless UNIVERSAL::can($dnsbl, 'query');

		push(@{$this->{LIST}}, $dnsbl);
	}

	$this;
}

sub query {
	my $this = shift;
	my $entry = shift;
	my $rv = undef;

	foreach my $dnsbl (@{$this->{LIST}}) {
		last if ($rv = $dnsbl->query($entry));
	}

	$rv->[2] ||= $this->{default} if $rv;

	$rv;
}

sub setdefault {
	my $this = shift;
	$this->{default} = shift;

	$this;
}

1;

__END__

=head1 SEE ALSO

L<PMilter::DNSBL>
