# $DUH: Leaf.pm,v 1.2 2002/12/11 21:45:59 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Callbacks::Leaf - collection of milters that does not propagate ACCEPT

=head1 SYNOPSIS

    use PMilter::Callbacks::Leaf;

    my $cb = PMilter::Callbacks::Leaf->new(..);

=head1 DESCRIPTION

Functions identically to PMilter::Callbacks, except for its handling of
SMFIS_BREAK.

When receiving a SMFIS_BREAK return, PMilter::Callbacks::Leaf stops
executing callbacks (as per the description of SMFIS_BREAK in
L<PMilter::Callbacks>).  Then, rather than returning SMFIS_BREAK, it returns
SMFIS_ACCEPT to its caller.

With as an object tree such as:

    Callbacks1 ->
        Module1
        Leaf1 ->
            Module2
            Module3
        Module4

A SMFIS_BREAK in Module1 or Module4 will cause the entire tree traversal to
halt, and Callbacks1 would return SMFIS_BREAK.  However, if Module2 returns
SMFIS_BREAK, then Module3 would *not* run, and Leaf1 would return
SMFIS_ACCEPT; as a result, Module4 *would* run.

PMilter::Callbacks::Leaf can also be visualized as a "try-catch" exception
handling system, similar to the C++-like construct:

    try {
        module1;
        ...
        moduleN;
    } catch (SMFIS_BREAK) {
        return SMFIS_ACCEPT;
    } // SMFIS_REJECT and SMFIS_TEMPFAIL exceptions propagate

=cut

package PMilter::Callbacks::Leaf;
use base PMilter::Callbacks;

use strict;
use warnings;

use PMilter::Callbacks qw(:all);

sub call {
	my $this = shift;
	my $rc = $this->SUPER::call(@_);

	($rc == SMFIS_BREAK) ? SMFIS_ACCEPT : $rc;
}

1;
