%%  Copyright (c) 2001 Dan Gudmundsson
%%  See the file "license.terms" for information on usage and redistribution
%%  of this file, and for a DISCLAIMER OF ALL WARRANTIES.
%% 
%%     $Id: erldemo.erl,v 1.1 2004/03/30 07:57:05 bjorng Exp $
%%
%%%----------------------------------------------------------------------
%%% File    : erldemo.erl
%%% Author  : Dan Gudmundsson <dgud@erix.ericsson.se>
%%% Purpose : Show that erlang can do opengl.
%%% Created : 11 Sep 2000 by Dan Gudmundsson <dgud@erix.ericsson.se>
%%%----------------------------------------------------------------------

-module(sune).
-author('dgud@erix.ericsson.se').

-include("sdl.hrl").
-include("sdl_events.hrl").
-include("sdl_video.hrl").
-include("sdl_keyboard.hrl").
-include("gl.hrl").

-export([go/0, go/1]).
-compile(export_all).

-define(W, 640).
-define(H, 480).

go() ->
    go([window]).

go(Mode) ->
    %% Init 
    Wrapper = sdl:init(?SDL_INIT_VIDEO),
    
    sdl_events:eventState(?SDL_ALLEVENTS ,?SDL_IGNORE),
    sdl_events:eventState(?SDL_KEYDOWN ,?SDL_ENABLE),
    sdl_events:eventState(?SDL_QUIT ,?SDL_ENABLE),

    Flags = 
	case lists:member(fullscreen, Mode) of 
	    true ->
		?SDL_OPENGL  bor ?SDL_FULLSCREEN;
	    _ -> 
		?SDL_OPENGL
	end,
    sdl_video:gl_setAttribute(?SDL_GL_DOUBLEBUFFER, 1),
    R = sdl_video:setVideoMode(?W, ?H, 16, Flags),

    %% Get Ready for events
    
    initWin(),
    
    Verts = {{ 0.5,  0.5, -0.5},  %1
	     { 0.5, -0.5, -0.5},  %2
	     {-0.5, -0.5, -0.5},   
	     {-0.5,  0.5, -0.5},  %4
	     {-0.5,  0.5,  0.5},
	     { 0.5,  0.5,  0.5},  %6
	     { 0.5, -0.5,  0.5}, 
	     {-0.5, -0.5,  0.5}}, %8    

    %%         Faces    Normal     U-axis   V-axis 
    Faces = [{{1,2,3,4},{0,0,-1},{-1,0,0}, {0,1,0}},  % 
	     {{3,8,5,4},{-1,0,0},{0,0,1},  {0,1,0}},  %
	     {{1,6,7,2},{1,0,0}, {0,0,-1}, {0,1,0}},  %
	     {{6,5,8,7},{0,0,1}, {1,0,0},  {0,1,0}},  %
	     {{6,1,4,5},{0,1,0}, {-1,0,0}, {0,0,1}},  %
	     {{7,8,3,2},{0,-1,0},{1,0,0},  {0,0,1}}],
    Data = {Faces, Verts},       

    useShaders(),

    {Time, Frames} = timer:tc(?MODULE, drawBox, 
			      [Data, 0, 5.0, 
			       {0.0, 0.0, 0.0}, 
			       {0.04, 0.05, 0.06},
			       {0.2, unknown},
			       true, false
			      ]),
    Secs = Time / 1000000,
    io:format("FPS: ~p (Frames ~p)~n", [catch Frames/Secs, Frames]),
    sdl:quit(),
    unregister(sdl_port),
    ok.

is_bump_possible() ->
    Ext = gl:getString(?GL_EXTENSIONS),
    Exts = string:tokens(Ext, " "),
    GLVerHasDOT3 = 
	case gl:getString(?GL_VERSION) of
	    "1.3" ++ _ -> true;
	    "1.4" ++ _ -> true;
	    _ -> false
	end,
    HasExtensions = lists:member("GL_ARB_texture_env_dot3", Exts) and 
	lists:member("GL_ARB_multitexture", Exts),
    HasExtensions or GLVerHasDOT3.

initWin() ->
    gl:viewport(0,0,?W,?H),
    gl:matrixMode(?GL_PROJECTION),
    gl:loadIdentity(),
    glu:perspective(30.0, ?W/?H, 0.1, 30.0), 
    gl:matrixMode(?GL_MODELVIEW),
    gl:loadIdentity(),    
    gl:enable(?GL_DEPTH_TEST),
    gl:depthFunc(?GL_LEQUAL),
    gl:enable(?GL_CULL_FACE),
    
    gl:materialfv(?GL_FRONT_AND_BACK, ?GL_DIFFUSE, {0.7,0.7,0.7,1.0}),
    gl:materialfv(?GL_FRONT_AND_BACK, ?GL_AMBIENT, {0.6,0.6,0.6,1.0}),
    gl:materialfv(?GL_FRONT_AND_BACK, ?GL_SPECULAR,{0.7,0.7,0.7,1.0}),
    gl:materialf(?GL_FRONT_AND_BACK, ?GL_SHININESS, 80.2),
    
    gl:lightModelfv(?GL_LIGHT_MODEL_AMBIENT, {0.3,0.3,0.3,1.0}),
    gl:enable(?GL_LIGHT0),
    gl:lightfv(?GL_LIGHT0, ?GL_DIFFUSE, {0.7,0.7,0.7,1}),
    gl:lightfv(?GL_LIGHT0, ?GL_SPECULAR, {0.7,0.7,0.7,1}),

    gl:enable(?GL_TEXTURE_2D),
    gl:shadeModel(?GL_SMOOTH),
    gl:clearColor(0.5,0.5,0.9,1.0).

draw_quads({Faces, Verts},Mode) ->
    gl:'begin'(?GL_QUADS),
    draw_quads2(Faces, Verts,Mode).

draw_quads2([{{V1,V2,V3,V4},N,Ut,Vt}|R],Cube, color) ->
    gl:normal3fv(N),
    gl:texCoord2f(0.0, 1.0), 
    gl:vertex3fv(element(V1, Cube)),
    gl:texCoord2f(0.0, 0.0), 
    gl:vertex3fv(element(V2, Cube)),
    gl:texCoord2f(1.0, 0.0), 
    gl:vertex3fv(element(V3, Cube)),
    gl:texCoord2f(1.0, 1.0), 
    gl:vertex3fv(element(V4, Cube)),
    draw_quads2(R,Cube,color);
draw_quads2([{{V1,V2,V3,V4},N,Ut,Vt}|R],Cube, Light) ->
    X = dot(Ut,Light),
    Y = dot(Vt,Light),
    Z = dot(N,Light),
    gl:color3fv(norm_rgb(X,Y,Z)),
    gl:normal3fv(N),
    gl:multiTexCoord2f(?GL_TEXTURE0,0.0, 1.0), 
    gl:multiTexCoord2f(?GL_TEXTURE1,0.0, 1.0), 
    gl:vertex3fv(element(V1, Cube)),
    gl:multiTexCoord2f(?GL_TEXTURE0,0.0, 0.0), 
    gl:multiTexCoord2f(?GL_TEXTURE1,0.0, 0.0), 
    gl:vertex3fv(element(V2, Cube)),
    gl:multiTexCoord2f(?GL_TEXTURE0,1.0, 0.0), 
    gl:multiTexCoord2f(?GL_TEXTURE1,1.0, 0.0), 
    gl:vertex3fv(element(V3, Cube)),
    gl:multiTexCoord2f(?GL_TEXTURE0,1.0, 1.0), 
    gl:multiTexCoord2f(?GL_TEXTURE1,1.0, 1.0), 
    gl:vertex3fv(element(V4, Cube)),
    draw_quads2(R,Cube,Light);
draw_quads2([],_,_) -> 
    gl:'end'(),
    ok.

drawBox(Cube, N, R, Pos, Vel, Speed = {S, T0}, Color, BumpMap) -> 
    gl:clear(?GL_COLOR_BUFFER_BIT bor ?GL_DEPTH_BUFFER_BIT),
    {NewPos = {X, Y, Z}, NewVel} = calc_pos(Pos, Vel, S),
    gl:loadIdentity(),

    gl:lightfv(?GL_LIGHT0, ?GL_POSITION, {0,10,10,1}),
    gl:lightfv(?GL_LIGHT0, ?GL_SPOT_DIRECTION, {0,0,-1}),
    gl:lightf(?GL_LIGHT0, ?GL_SPOT_CUTOFF, 45),

    gl:translatef(X, Y, Z-3.0),
    RotV = {0.5,0.5,1.0},
    gl:rotatef(R, element(1,RotV),element(2,RotV),element(3,RotV)),

    gl:enable(?GL_LIGHTING),
    case Color of
	true ->
	    gl:enable(?GL_TEXTURE_2D),
	    gl:color3f(1,1,1),
	    gl:texEnvi(?GL_TEXTURE_ENV, ?GL_TEXTURE_ENV_MODE, ?GL_MODULATE),
	    draw_quads(Cube,color);
	false ->
	    gl:disable(?GL_TEXTURE_2D),
	    draw_quads(Cube,color)
    end,
    
    gl:disable(?GL_BLEND),
    gl:disable(?GL_LIGHTING),
        
    check_error("DrawCube"),
    T1 = gl:swapBuffers(),
    case check_event(Color,BumpMap) of
	{NewCol, NewBump} ->
	    NewS = calc_speed(Speed, T1),
	    timer:sleep(10),
	    drawBox(Cube, N + 1, R + (3 * S),NewPos, NewVel, 
		    NewS,NewCol,NewBump);
	quit ->
	    N + 1
    end.

dot({V10,V11,V12}, {V20,V21,V22}) ->
    V10*V20 + V11*V21 + V12*V22.

norm_rgb(V1, V2, V3) when is_float(V1), is_float(V2), is_float(V3) ->
    D = math:sqrt(V1*V1+V2*V2+V3*V3),
    case catch {0.5+V1/D*0.5,0.5+V2/D*0.5,0.5+V3/D*0.5} of
	{'EXIT',_} -> {0.0,0.0,0.0};
	R -> R
    end.

calc_pos({X, Y, Z}, {Vx, Vy, Vz}, S) ->
    NX = X + S*Vx,
    NY = Y + S*Vy,
    NZ = Z + S*Vz,
    Nvx = if NX > 1.0; NX < -1.0  ->  Vx * -1; 
	     true -> Vx end,    
    Nvy = if NY > 1.0; NY < -1.0  ->  Vy * -1; 
	     true -> Vy end,
    Nvz = if NZ > 0.0; NZ < -10.0 ->  Vz * -1; 
	     true -> Vz end,

    {{NX, NY, NZ}, {Nvx, Nvy, Nvz}}.

calc_speed({S, unknown}, T1) ->
    {S, T1};
calc_speed({S, T0}, T1) ->
    SPF  = 0.2 + ((T1 - T0) / 1000), %% Last frame took 
    Diff = (SPF - S) / 100,
    NS   = S + Diff,
    {NS, T1}. 


check_error(Str) ->
    case {gl:getError(), sdl:getError()} of
	{0, ""} ->
	    ok;
	{GL, SDL} ->
	    io:format("In ~s Errors Reported GL ~p SDL ~s~n", [Str, GL, SDL])
    end.

check_event(ColMap, BumpMap) ->
    case sdl_events:pollEvent() of 
	#quit{} -> 
	    quit;
	no_event -> 
	    {ColMap,BumpMap};
	Quit when record(Quit, keyboard) -> 
	    if 
		Quit#keyboard.sym == ?SDLK_ESCAPE ->
		    quit;
		Quit#keyboard.sym == ?SDLK_q ->
		    quit;
		Quit#keyboard.sym == ?SDLK_b ->
		    {ColMap,get(bumpPossible) and not BumpMap};
		Quit#keyboard.sym == ?SDLK_c ->
		    {not ColMap,BumpMap};
		true -> 
		    io:format("Got event ~p~n", [Quit]),
		    {ColMap,BumpMap}
	    end;		    
	Event -> 
	    io:format("Got event ~p~n", [Event]),
	    {ColMap,BumpMap}
    end.

useShaders() ->
    case catch compileShaders() of
	{'EXIT', Reason} ->
	    io:format("~p ~n", [Reason]),
	    ok;
	Brick ->
	    gl:useProgramObject(Brick),
	    gl:uniform3fv(getUniLoc(Brick,"BrickColor"),1,{1.0,0.3,0.2}),
	    gl:uniform3fv(getUniLoc(Brick,"MortarColor"),1,{0.85,0.86,0.84}),
	    gl:uniform3fv(getUniLoc(Brick,"BrickSize"),1,{0.3,0.15,0.3}),
	    gl:uniform3fv(getUniLoc(Brick,"BrickPct"),1,{0.9,0.85,0.9}),
	    gl:uniform3fv(getUniLoc(Brick,"LightPosition"),1,{0.0,10.0,10.0}),
	    check_error("Program attached and loaded vars")
    end.


getUniLoc(Handle,Name) ->
    case gl:getUniformLocation(Handle,Name) of
	-1 -> io:format("No variable named ~s~n", [Name]),
	      -1;
	Loc ->
	    io:format("Variable ~s => ~p ~n", [Name, Loc]),
	    Loc
    end.

compileShaders() ->
    {ok, VsBin} = file:read_file("sune.vs"),
    {ok, FsBin} = file:read_file("sune.fs"),
    Vs = compileShader(?GL_VERTEX_SHADER, VsBin),
    Fs = compileShader(?GL_FRAGMENT_SHADER, FsBin),

    Prog = gl:createProgramObject(),

    gl:attachObject(Prog, Vs),
    gl:attachObject(Prog, Fs),
    check_error("Attached"),
    gl:linkProgram(Prog),
    check_error("Linked"),
    case gl:getObjectParameteriv(Prog, ?GL_OBJECT_LINK_STATUS) of
	1 -> io:format("Linked Status ok (~p) ~n", [1]), 
	     printInfo(Prog), %% Check status even if ok
	     Prog;
	E -> 
	    io:format("Error compile status ~p ~n", [E]),
	    printInfo(Prog),
	    exit(compile_error)
    end,
    check_error("Compiled and Linked"),
    Prog.

compileShader(Type,Src) ->
    Handle = gl:createShaderObject(Type),    
    ok = gl:shaderSource(Handle, 1, [Src], [-1]),
    check_error("Shader Loaded"),
    ok = gl:compileShader(Handle),
    case gl:getObjectParameteriv(Handle, ?GL_OBJECT_COMPILE_STATUS) of
	1 -> io:format("Compile Status ok (~p) ~n", [1]), 
	     printInfo(Handle), %% Check status even if ok
	     Handle;
	E -> 
	    io:format("Error compile status ~p ~n", [E]),
	    printInfo(Handle),
	    exit(compile_error)
    end.

printInfo(ShaderObj) ->
    Len = gl:getObjectParameteriv(ShaderObj, ?GL_OBJECT_INFO_LOG_LENGTH),
    case Len > 0 of
	true ->
	    case catch gl:getInfoLog(ShaderObj, Len) of
		{_, []} ->
		    ok;
		{_, InfoStr} ->
		    io:format("Info: ~p ~n", [InfoStr]);
		Error ->
		    io:format("PrintInfo crashed with ~p ~n", [Error])
	    end;
	false ->
	    io:format("CompileInfo ~p ~n", [Len])	    
    end.

