#ifndef funxml_SERIALIZER_H_INCLUDED
#define funxml_SERIALIZER_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// data_node_serializers.h: some file parsers for the s11n framework
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////


#include <S11N_NS/debuggering_macros.h> // COUT/CERR
#include <S11N_NS/to_string.h> // to/from_string()
#include <S11N_NS/file_util.h> // get_i/ostream()
#include <S11N_NS/string_util.h> // translate_entities()
#include <S11N_NS/s11n_core.h> // classload()

#include "data_node_functor.h"
#include "data_node_io.h" // data_node_serializer<> and friends

#define MAGIC_COOKIE_FUNXML "<!DOCTYPE SerialTree>"

namespace S11N_NS {
	namespace io {
                namespace sharing {
                        /**
                           Sharing context used by funxml_serializer.
                         */
                        struct funxml_sharing_context {};

                }
                /** convenience typedef */
                typedef std::map<std::string,std::string> entity_translation_map;


                /**
                   The entity translations map used by funxml_serializer.
                 */
                entity_translation_map & funxml_serializer_translations();



                /**
                   De/serializes objects from/to a simple XML grammar,
                   with all properties and children stored as subnodes.
                */
                template <typename NodeType>
                class funxml_serializer : public tree_builder_lexer<NodeType,sharing::funxml_sharing_context>
                {
                public:
                        typedef NodeType node_type;

                        typedef funxml_serializer<node_type> this_type; // convenience typedef
                        typedef tree_builder_lexer<node_type,sharing::funxml_sharing_context> parent_type; // convenience typedef

                        funxml_serializer() : parent_type( "funxml_data_nodeFlexLexer" ), m_depth(0)
                        {
                                this->magic_cookie( MAGIC_COOKIE_FUNXML );
                        }

                        virtual ~funxml_serializer() {}

                        /**
                           Reimplemented to return this type's entity
                           translation map.
                         */
                        virtual const entity_translation_map & entity_translations() const
                        {
                                return funxml_serializer_translations();
                        }


                        /**
                           Writes src out to dest.
                        */
                        virtual bool serialize( const node_type & src, std::ostream & dest )
                        {

// TAB() is a helper macro for some serializers.
#define TAB(LEVEL,ECHO) indent = ""; for( size_t i = 0; i < depth + LEVEL; i++ ) { indent += '\t'; if(ECHO) dest << '\t'; }
                                size_t depth = this->m_depth++;
                                if ( 0 == depth )
                                {
                                        dest << this->magic_cookie() << '\n';
                                }


                                std::string nname = src.name();
                                std::string impl = src.impl_class();
                                S11N_NS::translate_entities( impl, this->entity_translations(), false ); // handle class templates

                                std::string indent;


                                dest << "<" << nname << " class=\"" << impl << "\">\n";
                                typename node_type::const_iterator cit = src.begin();
                                std::string propval;
                                std::string key;
                                TAB(1,0);
                                for ( ; src.end() != cit; ++cit )
                                {
                                        key = ( *cit ).first;
                                        propval = ( *cit ).second;
                                        S11N_NS::translate_entities( propval, this->entity_translations(), false );
                                        dest << indent;
                                        dest << "<" << key << ">";
                                        dest << propval;
                                        dest << "</" << key << ">\n";
                                }
                                TAB(1,0);
                                std::for_each( src.children().begin(),
                                               src.children().end(),
                                               node_child_simple_formatter<this_type>( *this,
                                                                                 dest,
                                                                                 indent,
                                                                                 "" )
                                               );

                                TAB(0,1);
                                dest << "</" << nname << ">\n";
                                if( 0 == depth )
                                {
                                        dest.flush();
                                        // if we don't do this then the client is possibly forced to flush() the stream :/
                                }
                                --this->m_depth;
                                return true;
#undef TAB
                        }


                        /**
                           Overridden to parse src using this object's grammar rules.
                        */
                        virtual node_type * deserialize( std::istream & src )
                        {
                                return deserialize_lex_forwarder<node_type,sharing::funxml_sharing_context>( "funxml_data_nodeFlexLexer", src );
                                //CERR << "deserialize() returning " << std::hex << ret << "\n";
                        }

                private:
                        size_t m_depth;
                };

	} // namespace io
} // namespace S11N_NS

#endif // funxml_SERIALIZER_H_INCLUDED
