#ifndef S11N_NS_DATA_SERIALIZE_H_INCLUDED
#define S11N_NS_DATA_SERIALIZE_H_INCLUDED
////////////////////////////////////////////////////////////////////////
// data_node_serialize.h:
//
// Defines the core de/serialize() functions (and close friends).
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <map>
#include <deque>

#include <cassert>
#include <typeinfo>

#include <S11N_NS/debuggering_macros.h> // COUT/CERR
#include <S11N_NS/s11n_core.h> // classload()
#include <S11N_NS/pointer_stripper.h>
#include <S11N_NS/functor.h> // several useful functors.

#define NODEERR if(0) CERR

////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.h ALLOWED!
////////////////////////////////////////////////////////////////////////////////


namespace { // anon namespace important for linking reasons, to avoid ODR violations.

        /***
            Specializations of s11n_api_marshaler may
            marshall calls to local serialization APIs.

            This is the newer form of s11n_api_adapter&lt;&gt;, and it
            will eventually replace that type.

            For the default implementation, SerializableT must support
            these two operations:

           <b>Serialize:</b>

<pre>
bool operator()( serialize_node_type & dest ) const;
</pre>

           <b>Deserialize:</b>

<pre>
bool operator()( const serialize_node_type & src );
</pre>

	   If these conditions are not met a compile-time error will occur.

           Normally specializations are created by using one of these macros:

           - S11N_NS_SERIALIZABLE_BASE()

           - S11N_NS_SERIALIZABLE_PROXY()

           - S11N_NS_SERIALIZABLE_SUBTYPE()

	   but specializations must be hand-written if the macro
	   arguments contain commas (e.g., class template names)
	   because those break the macros. Specializations may, in any
	   case, be hand-written to implement arbitrary
	   customizations.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler
        {
                typedef SerializableT serializable_type;

                /**
                   Returns src.operator()( dest ). Specializations are free to reimplement
                   this call as they see fit.
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type & src )
                {
                        NODEERR << "Unspecialized serialize-via-self.\n";
                        dest.impl_class( ::classname<serializable_type>() ); // only good for monomorphs
                        return src( dest );
                }

                /**
                   Returns dest.operator()( src ). Specializations are free to reimplement
                   this call as they see fit.
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type & dest )
                {
                        NODEERR << "Unspecialized deserialize-via-self.\n";
                        return dest( src );
                }

//                 template <typename NodeType>
//                 bool operator()( NodeType &dest, const serializable_type & src ) const
//                 {
//                         return serialize<NodeType>( dest, src );
//                 }
//                 template <typename NodeType>
//                 bool operator()( const NodeType & src, serializable_type & dest )
//                 {
//                         return deserialize<NodeType>( src, dest );
//                 }
        };

        /**
           A specialization to handle pointer types the same as
           reference/value types. It simply translates the pointers
           into references.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler<SerializableT *>
        {
                /**
                   The SerializableT templatized type, minus any
                   pointer part.
                */
                typedef SerializableT serializable_type;

                /**
                   Convenience typedef: this class' parent type.
                 */
                typedef s11n_api_marshaler<serializable_type> parent_type;

                /**
                   Returns parent_type::serialize( dest, *src );
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type * src )
                {
                        if( ! src ) return false;
                        NODEERR << "Unspecialized pointer-type serialize via-non-pointer-parent-type handler.\n";
                        return parent_type::serialize( dest, *src );
                }

                /**
                   Returns parent_type::deserialize( src, *dest );
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type * dest )
                {
                        if( ! dest ) return false;
                        NODEERR << "Unspecialized pointer-type deserialize via-non-pointer-parent-type handler.\n";
                        return parent_type::deserialize( src, *dest );
                }

//                 template <typename NodeType>
//                 bool operator()( NodeType &dest, const serializable_type * src ) const
//                 {
//                         return serialize<NodeType>( dest, src );
//                 }
//                 template <typename NodeType>
//                 bool operator()( const NodeType & src, serializable_type * dest )
//                 {
//                         return deserialize<NodeType>( src, dest );
//                 }

        };



} // anon namespace


/**********************************************************************

General Conventions:

	NodeType should conform to the conventions laid out
	by S11N_NS::data_node.


    SerializableTypes/BaseTypes:

	-BaseT must have the following in it's interface:

	- bool SerializeFunction( NodeType & dest ) const;

	- bool DeserializeFunction( const NodeType & dest );

	SerializeFunction/DeserializeFunction need not be virtual,
	though they should be unless your BaseT has a way of getting
	the stringified class NAME of it's subtypes (the Qt lib, e.g.,
	supports such a feature).
	

    Proxy functors:

	Serialization functor must have:

	bool operator()( NodeType & dest, const BaseT & src ) const;

	Derialization functor must have:

	bool operator()( const NodeType & src, BaseT & dest ) const;

	They may be the same functor type - const resolution will
	determine which s11n uses. Sometimes this causes ambiguity,
	and may require 2 functors.

	These signatures apply for all functors designed to work as
	de/serializers.



**********************************************************************/


namespace S11N_NS {



//         /**
//            Returns func( target, src ).
//         */
//         template <typename DataNodeType,typename SerializableT, typename SerializerFunctorT>
//         bool serialize( DataNodeType & target, const SerializableT & src, const SerializerFunctorT & func )
//         {
//                 NODEERR << "serialize<>(DataNodeType,SerializableT,SerializerFunctorT) forwarding to functor.\n";
//                 return func( target, src );
//         }


        /**
           Serializes src to target using a default API marshaling mechanism.
        */
        template <typename DataNodeType, typename SerializableT>
        bool serialize( DataNodeType & target, const SerializableT & src )
        {
                NODEERR << "serialize<>(DataNodeType,SerializableT): using default API-marshaling functor.\n";
                //return serialize( target, src, serialize_marshaler_selector<SerializableT>() );
                //return serialize( target, src, s11n_api_marshaler<SerializableT>() );
                return s11n_api_marshaler<SerializableT>::serialize( target, src );
        }


//         /**
//            Returns func( src, target ).
//         */
//         template <typename DataNodeType, typename DeserializableT, typename DeserializerFunctorT>
//         bool deserialize( const DataNodeType & src, DeserializableT & target, const DeserializerFunctorT & func )
//         {
//                 return func( src, target );
//         }

        /**
           Deserializes target from src using a default API marshaling
           mechanism.
        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize( const DataNodeType & src, DeserializableT & target )
        {
                NODEERR << "deserialize(DataNodeType,SerializableT): using default API marshaler.\n";
                return s11n_api_marshaler<DeserializableT>::deserialize( src, target );
        }


        /**
           Tries to deserialize a DeserializableT from src, using
           <code>classload<DeserializableT>()</code> to load
           DeserializableT. Returns 0 on error, otherwise returns a
           pointer to a new object, which the caller takes ownership
           of.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize( const DataNodeType & src )
        {
                NODEERR << "DeserializableT * deserialize(DataNodeType)\n";
                DeserializableT * ret = classload<DeserializableT>( src.impl_class() );
                if( ! ret )
                {
                        NODEERR << "deserialize<>(): impl class '"
                                << src.impl_class()<<"' classload failed.\n"
                                << "It is probably not registered with it's base classloader.\n";
                        return 0;
                }
                if( ! deserialize( src, *ret ) )
                {
                        NODEERR << "deserialize(): failed for unknown reason.\n";
                        delete( ret );
                        ret = 0;
                }
                return ret;
        }


//         /**
//            Serializes src to as a subnode of target, named
//            nodename. Except for the addition of a subnode, it is
//            identical to serialize( target, src, func ).
//            The given functor is responsible for the serialization.
//            The target node becomes the owner of any created pointer.
//            This is a convenience function: not part of the s11n kernel.
//         */
//         template <typename DataNodeType, typename SerializableT, typename SerializerFunctorT>
//         bool serialize_subnode( DataNodeType & target,
//                                 const::std::string & nodename,
//                                 const SerializableT & src,
//                                 const SerializerFunctorT & func )
//         {
//                 NODEERR << "serialize_subnode<>(DataNodeType, '"<<nodename<<"', SerializableT, SerializerFunctorT)\n";
//                 DataNodeType * sub = new DataNodeType();
//                 sub->name( nodename );
//                 if( ! serialize( *sub, src, func ) )
//                 {
//                         delete( sub );
//                         sub = 0;
//                 }
//                 else
//                 {
//                         target.children().push_back( sub );
//                 }
//                 return sub != 0;
//         }




        /**
           Serializes src to as a subnode of target, named
           nodename. Except for the addition of a subnode, it is
           identical to serialize( target, src ).

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename SerializableT >
        bool serialize_subnode( DataNodeType & target,
                                const std::string & nodename,
                                const SerializableT & src )
        {
                NODEERR << "serialize_subnode<>(DataNodeType, '"<<nodename<<"', SerializableT)\n";
                DataNodeType * sub = new DataNodeType();
                sub->name( nodename );
                if( ! serialize<DataNodeType,SerializableT>( *sub, src ) )
                {
                        delete( sub );
                        sub = 0;
                }
                else
                {
                        target.children().push_back( sub );
                }
                return sub != 0;
        }


//         /**
//            Looks for a subode of src named subnodename. If it does not
//            find one it returns false, otherwise it returns the result of
//            deserialize( child, target, func ).

//            This is a convenience function: not part of the s11n kernel.
//         */
//         template <typename DataNodeType,
//                   typename DeserializableT,
//                   typename DeserializerFunctorT>
//         bool deserialize_subnode( const DataNodeType & src,
//                                   const std::string & subnodename,
//                                   DeserializableT & target,
//                                   const DeserializerFunctorT & func )
//         {
//                 const DataNodeType * ch = find_child_by_name( src, subnodename );
//                 if( ! ch ) return false;
//                 return deserialize<
//                         DataNodeType,
//                         DeserializableT,
//                         DeserializerFunctorT
//                         >( *ch, target, func );
//         }


        /**
           If a child named subnodename is found in src then this function
           returns deserialize( child, target ) and returns it's result, otherwise
           it returns 0.

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize_subnode( const DataNodeType & src,
                                  const std::string & subnodename,
                                  DeserializableT & target )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch, target );

        }

        /**
           If a child named subnodename is found in src then this function
           returns the result of deserialize(child), otherwise
           it returns 0.

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize_subnode( const DataNodeType & src,
                                               const std::string & subnodename )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch );
        }


        /**
           Clones an arbitrary SerializableType using it's
           DataNodeType serializable implementation.

           Returns a clone of tocp, or returns 0 on error.
           The caller owns the returned pointer.

           This copy is polymorphism-safe as long as all participating
           Serializables (re)implement the appropriate de/serialize
           operations, similarly to as they would do for a copy ctor
           or classical Clone() member function.

           Tip: clone() is a convenient way to test new de/serialize
           functions, e.g., for new Serializables, because if it works
           then deserializng from streams/files will also work. This
           function takes SerializableType through the whole
           de/serialize process, including classloading.
        */
        template <typename DataNodeType, typename SerializableType>
        SerializableType * clone( const SerializableType & tocp )
        {
                DataNodeType node;
                if( ! serialize( node, tocp ) ) return 0;
                return deserialize<DataNodeType,SerializableType>( node );
        }



        /**
           "Casts" t1 to t2 using serialization. This will work
           whenever t1 and t2 are "compatible", whatever that really
           means. It can be used, e.g., to copy a list&lt;int&gt;
           to a vector&lt;double&gt;, provided both types have
           been proxied.

           Note that in the case of containers, the pointerness of the
           contained types is irrelevant: this works on both, thus
           a list&lt;int&gt; can be "cast" to a vector&lt;double*&gt;!

           As usual for a failed deserialization, if it returns false
           then t2 may be in an undefined state. There is no guaranty,
           however, that t2's deserialize operator will ever be
           called, as the serialization of t1 must first succeed
           for that to happen.
        */

        template <typename NodeType, typename Type1, typename Type2>
        bool s11n_cast( const Type1 & t1, Type2 & t2 )
        {
                NodeType n;
                return serialize( n, t1 ) && deserialize( n, t2 );
        }




} // namespace S11N_NS

#undef NODEERR

#endif // S11N_NS_DATA_SERIALIZE_H_INCLUDED
