#ifndef S11N_NS_DATA_NODE_IO_H_INCLUDED
#define S11N_NS_DATA_NODE_IO_H_INCLUDED

////////////////////////////////////////////////////////////////////////
// data_node_io.h
// some i/o interfaces & helpers for s11n
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////


#include <string>
#include <sstream>
#include <list>
#include <map>
#include <deque>
#include <iostream>
#include <memory>// auto_ptr

#include <cassert>
#include <typeinfo>


// #include <S11N_NS/to_string.h> // to/from_string()
// #include <S11N_NS/string_util.h> // translate_entities()

#include <S11N_NS/debuggering_macros.h> // COUT/CERR
#include <S11N_NS/file_util.h> // get_i/ostream(), bytes_from_file()
#include <S11N_NS/s11n_core.h> // classloader()

#include "data_node_serialize.h" // unfortunately dep

////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.h ALLOWED!
////////////////////////////////////////////////////////////////////////////////


namespace S11N_NS {

        namespace io {


                /**
                   Convenience function for grabbing the first line of a file.

                   If AsFile == true then returns the first line of the
                   file, else returns up to the first newline of src.
                */
                std::string get_magic_cookie( const std::string & src, bool AsFile = true );

                /**
                   Convenience function for grabbing the first line of a
                   stream.

                   Returns the first line of the given stream, or an
                   empty string on error.
                */
                std::string get_magic_cookie( std::istream & is );

                /**
                   data_node_serializer provides an interface for
                   saving/loading a given abstract data node type
                   to/from streams.

                   It is designed for containers which comply to the
                   S11N_NS::data_node interface and
                   conventions.


                   Conventions:

                   Must provide:

                   typedef NodeT node_type

                   Two de/serialize functions, following the
                   stream-based interface shown here (filename-based
                   variants are optional, but convenient for clients).

                */
                template <typename NodeT>
                class data_node_serializer
                {
                public:
                        /**
                           The underlying data type used to store
                           serialized data.
                        */
                        typedef NodeT node_type;

                        data_node_serializer()
                        {
                                this->magic_cookie( "WARNING: magic_cookie() not set!" );
                                // ^^^ subclasses must do this.
                                this->metadata().name( "serializer_metadata" );
                        };
                        virtual ~data_node_serializer(){};


                        /**
                           A convenience typedef, mainly for subclasses.
                        */
                        typedef std::map<std::string,std::string> translation_map;

                        /**
                           Returns a map intended for use with
                           S11N_NS::translate_entities().
                           
                           The default implementation returns an empty map.
                           
                           Subclasses should override this to return a translation
                           map, if they need one. The default map is empty.

                           Be aware that this may very well be called
                           post-main(), so subclasses should take that into
                           account and provide post-main()-safe maps! (Tip:
                           see S11N_NS::phoenix.)
                        */
                        virtual const translation_map & entity_translations() const
                        {
                                typedef phoenix<translation_map,data_node_serializer<node_type> > TMap;
                                return TMap::instance();
                        }



                        /**
                           Must be implemented to format node_type to the given ostream.

                           It should return true on success, false on error.

                           The default implementation always returns false.
                        */
                        virtual bool serialize( const node_type & src, std::ostream & dest )
                        {
                                return false;
                        }

                       /**
                           Overloaded to save dest to the given filename.

                           The output file is compressed if
                           S11N_NS::compression_policy() has
                           been set to enable it.

                           Returns true on success, false on error.
                        */
                        bool serialize( const node_type & src, const std::string & destfile )
                        {
                                std::ostream * os = S11N_NS::get_ostream( destfile );
                                if( ! os ) return false;
                                bool b = this->serialize( src, *os );
                                delete( os );
                                return b;
                        }

                        /**
                           Must be implemented to parse a node_type from the given istream.

                           It should return true on success, false on error.

                           The default implementation always returns false and does nothing.
                        */

                        virtual node_type * deserialize( std::istream & )
                        {
                                return false;
                        }


                        /**
                           Overloaded to load dest from the given filename.

                           It supports zlib/bz2lib decompression for
                           files (not source strings) if your s11n lib
                           supports them.

                           This is virtual ONLY to avoid a bogus(?) 
                           error from gcc when accessing it via
                           subclass instances.
                        */
                        virtual node_type * deserialize( const std::string & src )
                        {
                                typedef std::auto_ptr<std::istream> AP;
                                AP is = AP( S11N_NS::get_istream( src ) );
                                if( ! is.get() ) return 0;
                                return this->deserialize( *is );
                        }


                        /**
                           Gets this object's magic cookie.

                           Cookies are registered with
                           <code>class_loader<noder_serializable<NodeType>></code>
                           types to match files to file input parsers.
                        */
                        std::string magic_cookie() const
                        {
                                return this->m_cookie;
                        }

                protected:
                        /**
                           Sets the magic cookie for this type.
                        */
                        void magic_cookie( const std::string & c )
                        {
                                this->m_cookie = c;
                        }

                        /**
                           metadata is an experimental feature allowing
                           serializers to store arbitrary information
                           in their data strings.
                         */
                        node_type & metadata()
                        { return this->m_meta; }
                        /**
                           A const overload of metadata().
                         */
                        const node_type & metadata() const
                        { return this->m_meta;}
                private:
                        std::string m_cookie;
                        node_type m_meta;
                }; // data_node_serializer<>

                /**

                Tries to load a NodeType object from the given
                node. It uses the cookie from the input stream and
                classload<SerializerBaseType>() to find a matching
                Serializer.

                0 is returned on error, else a new pointer, which the
                caller owns.

                Achtung: the first line of input from the input stream
                is consumed by this function (to find the cookie), and
                the cookie is not passed on to the handler! The only
                reliable way around [that i know of] this is to buffer
                the whole input as a string, and i don't wanna do that
                (it's bad for massive data files).

                ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeType, typename SerializerBaseType>
                NodeType * load_node_classload_serializer( std::istream & is )
                {
                        typedef SerializerBaseType NS;
                        std::string cookie = get_magic_cookie( is );
                        // CERR << "cookie="<<cookie<<std::endl;
                        if( cookie.empty() )
                        {
                                CERR << "Odd: got a null cookie from the istream.\n";
                                return 0; // happens post-main() on valid streams sometimes!?!?!
                        }
                        typedef std::auto_ptr<NS> AP;
                        AP ser = AP( S11N_NS::classload<NS>( cookie ) );
                        if( ! (ser.get()) )
                        {
                                CERR << "Did not find serializer for cookie ["<<cookie<<"]."<<std::endl;
                                return NULL;
                        }
                        // CERR << "Dispatching to node loader for cookie ["<<cookie<<"]\n";
                        return ser->deserialize( is );
                }

                /**
                   Returns a node pointer, parsed from the given stream, using
                   <code>S11N_NS::io::data_node_serializer<NodeType></code>
                   as the base type for looking up a stream handler.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeType>
                NodeType * load_node( std::istream & is )
                {
                        return load_node_classload_serializer<NodeType,data_node_serializer<NodeType> >( is );
                }

                /**
                   Overloaded form of load_node( istream ), provided for
                   convenience.

                   If AsFile is true, input is treated as a file,
                   otherwise it is treated as a string containing input
                   to parse.

                   ACHTUNG: Only usable for loading ROOT nodes.

                   Maintenance note: AsFile==false may be extremely
                   inefficient, as src may get copied one additional
                   time.
                */
                template <typename NodeType>
                NodeType * load_node( const std::string & src, bool AsFile = true )
                {
                        typedef std::auto_ptr<std::istream> AP;
                        AP is = AP( S11N_NS::get_istream( src, AsFile ) );
                        if( ! is.get() ) return 0;
                        return load_node<NodeType>( *is );
                }

                /**
                   Tries to load a SerializableT from the given stream.
                   On success returns a new object, else 0.

                   The caller owns the returned pointer.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeT,typename SerializableT>
                SerializableT * load_serializable( std::istream & src )
                {
                        typedef std::auto_ptr<NodeT> AP;
                        AP node = AP( load_node<NodeT>( src ) );
                        if( ! node.get() )
                        {
                                CERR << "load_serializable<>(istream) Could not load a root node from the input.\n";
                                return 0;
                        }
                        return S11N_NS::deserialize<NodeT,SerializableT>( *node );
                }

                /**
                   An overloaded form which takes an input string. If
                   AsFile is true the string is treated as a file
                   name, otherwise it is processed as an input stream.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeT,typename SerializableT>
                SerializableT * load_serializable( const std::string & src, bool AsFile = true )
                {
                        typedef std::auto_ptr<std::istream> AP;
                        AP is = AP( get_istream( src, AsFile ) );
                        if( ! is.get() )
                        {
                                CERR << "load_serializable<>(string) Could not load a root node from the input.\n";
                                return 0;
                        }
                        return load_serializable<NodeT,SerializableT>( *is );
                }


                /**
                   Saves src to the given ostream using the given
                   Serializer type.

                   ONLY use this for saving root nodes!
                */
                template <typename NodeType, typename SerializerT>
                bool save_node( const NodeType & src, std::ostream & dest )
                {
                        return SerializerT().serialize( src, dest );
                }


                /**
                   Saves src, a Serializable type, to the given
                   ostream using a SerializerT serializer.

                   SerializerT must be compatible with
                   S11N_NS::io::data_node_serializer<>
                   conventions and must provide a <code>typedef XXX
                   node_type</code>, where XXX is a data type
                   conforming to S11N_NS::data_node
                   conventions.

                   Returns true on success, false on error.

                   ONLY use this for saving root nodes!
                */
                template <typename SerializerT,typename SerializableT>
                bool save_serializable( const SerializableT & src,
                                        std::ostream & dest )
                {
                        typedef typename SerializerT::node_type node_type;
                        node_type node( "serializable" );
                        if( ! S11N_NS::serialize<node_type,SerializableT>( node, src ) ) return false;
                        return SerializerT().serialize( node, dest );
                }

                /**
                   An overloaded form which takes a filename.

                   ONLY use this for saving root nodes!
                */
                template <typename SerializerT,typename SerializableT>
                bool save_serializable( const SerializableT & src,
                                        const std::string & filename )
                {
                        typedef std::auto_ptr<std::ostream> AP;
                        AP os = AP( S11N_NS::get_ostream( filename ) );
                        if( ! os.get() ) return false;
                        return save_serializable( src, *os );
                }


        } // namespace io

} // namespace S11N_NS

#endif // S11N_NS_DATA_NODE_IO_H_INCLUDED
