/*
  GNU polyxmass - the massist's program.
  -------------------------------------- 
  Copyright (C) 2000,2001,2002 Filippo Rusconi

  http://www.polyxmass.org

  This file is part of the "GNU polyxmass" project.
   
  The "GNU polyxmass" project is an official GNU project package (see
  www.gnu.org) released ---in its entirety--- under the GNU General
  Public License and was started at the Centre National de la
  Recherche Scientifique (FRANCE), that granted me the formal
  authorization to publish it under this Free Software License.

  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU  General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
   
  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.
   
  You should have received a copy of the GNU  General Public
  License along with this software; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-polymer-plugins.h"
#include "pxmchem-polymer.h"
#include "pxmchem-monomer.h"
#include "pxmchem-oligomer.h"
#include "libpolyxmass-plugin.h"


/* This one was declared extern in pxmchem-polymer-plugins.h
 */
PxmPolymer *
(*pxmchem_polseqdata_xml_node_render_plugin) (xmlDocPtr xml_doc,
					      xmlNodePtr xml_node,
					      GPtrArray *mnm_refGPA,
					      gint codelen,
					      gpointer data);



gpointer
pxmchem_polseqdata_xml_node_choose_renderer (gchar *version)
{

  if (version == NULL)
    return pxmchem_polseqdata_xml_node_renderer_v_none;
  
  if (0 == strcmp (version, "0.1"))
    return pxmchem_polseqdata_xml_node_renderer_v_0_1;
  

  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: plugin for version: '%s' has no registered xml renderer\n"),
	 __FILE__, __LINE__, version);
  
  return NULL;
}



   
PxmPolymer *
pxmchem_polseqdata_xml_node_renderer_v_none (xmlDocPtr xml_doc,
					    xmlNodePtr xml_node,
					    GPtrArray *mnm_refGPA,
					    gint codelen,
					    gpointer user_data)
{
  /*
    The xml node we are in is structured this way:
    
    <polseqdata>
    <polseqinfo>
    <type>protein</type>
    <name>hamster P-glycoprotein</name>
      
    And the xml_node parameter points to the 
    
    <polseqdata version="0.1"> element tag:
    ^
    |
    +----- here we are right now.
     
    Which means that xml_node->name == "polseqdata" and that
    we'll have to go one step down to the first child of the 
    current node in order to get to the <polseqinfo> element.
    
  */
  PxmPolymer *polymer = NULL;
  PxmProp *prop = NULL;
  
  xmlNodePtr xml_child_node = NULL;

  gchar *help = NULL;
  
  gint count = -1;
    
  /*
    The array of reference monomers cannot be NULL or empty.
  */
  g_assert (mnm_refGPA != NULL && mnm_refGPA->pdata != NULL);

  /*
    Make sure we have parameters pointing bona fide to the right
    xml element.
  */
  g_assert (xml_node != NULL);
  g_assert (xml_doc != NULL);

  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseqdata"));
  
  polymer = pxmchem_polymer_new ();


  /*
    Now go to the first child of current node: <polseqinfo>.
  */
  xml_node = xml_node->children;

  /*
    From a rigorous XML parsing point of view, the blanks found in
    the XML document are considered to be nodes, and we have to detect
    these and take proper action: go next sibling (next blank) as long
    as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /*
    Check that we have effectively a <polseqinfo> element here.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseqinfo"));


  xml_child_node = xml_node;

  /*
    The plminfo structure is allocated in pxmchem_polymer_new (), so 
    we first have to free it  before calling the function below that
    allocates also a plminfo instance.
  */
  g_assert (polymer->plminfo != NULL);
  pxmchem_plminfo_free (polymer->plminfo);
  
  polymer->plminfo = 
    pxmchem_polymer_render_xml_node_polseqinfo (xml_doc,
						xml_child_node,
						NULL);
  
  /*
    Now go to the next child of <polseqinfo>, which is <polseq>
  */
  xml_node = xml_node->next;

  /*
    From a rigorous XML parsing point of view, the blanks found in
    the XML document are considered to be nodes, and we have to detect
    these and take proper action: go next sibling (next blank) as long
    as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /*
    Check that we have effectively a <polseq> element here.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseq"));
  
  xml_child_node = xml_node;

  count = pxmchem_polymer_render_xml_node_polseq (xml_doc,
						  xml_child_node,
						  polymer->monomerGPA,
						  mnm_refGPA,
						  codelen,
						  NULL);
  if (count == -1)
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	  _("%s@%d: failed to render the polseq node\n"),
	   __FILE__, __LINE__);



  /*
    The <prop> elements that follow the <polseq> element are
    optional, and there may be more than one. So, the best is
    to go while ().
  */
  /*
    Now go to the next child of <polseq>, which may be <prop>
  */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      g_assert (0 == strcmp ((gchar *) xml_node->name, "prop"));
  
      /*
	Just get the prop's name so that we can get to the
	proper xml renderer plugin function, without modifying
	the xml_node, since the pluging function expects a node 
	pointer corresponding to <prop> and not <name>.
      */
      xml_child_node = xml_node->children;
      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
  
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "name"));

      help = 
	(gchar *) xmlNodeListGetString (xml_doc, xml_child_node->xmlChildrenNode, 1);
      g_assert (help != NULL);
      
      libpolyxmass_prop_renderer_xml_prop_plugin = 
	libpolyxmass_prop_choose_render_xml_prop_plugin (help);
      
      if (libpolyxmass_prop_renderer_xml_prop_plugin == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: plugin prop: '%s' has no registered xml renderer\n"),
		 __FILE__, __LINE__, help);
	  
	  /*
	    Go to the next <prop>, if any.
	  */
	  xml_node = xml_node->next;
	  /*
	    From a rigorous XML parsing point of view, the blanks
	    found in the XML document are considered to be nodes, and
	    we have to detect these and take proper action: go next
	    sibling (next blank) as long as blanks are encountered.
	  */
	  while (TRUE == xmlIsBlankNode (xml_node))
	    xml_node = xml_node->next;
	}
      
      g_free (help);
      
      prop = 
	libpolyxmass_prop_renderer_xml_prop_plugin (xml_doc, xml_node, NULL);
      g_assert (prop != NULL);

      /*
	Append the newly allocated prop instance to the array of the 
	polymer.
      */
      g_ptr_array_add (polymer->propGPA, prop);

      /*
	Go to the next <prop>, if any.
      */
      xml_node = xml_node->next;

      /*
	From a rigorous XML parsing point of view, the blanks found in
	the XML document are considered to be nodes, and we have to detect
	these and take proper action: go next sibling (next blank) as long
	as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  return polymer;
}



   
PxmPolymer *
pxmchem_polseqdata_xml_node_renderer_v_0_1 (xmlDocPtr xml_doc,
					    xmlNodePtr xml_node,
					    GPtrArray *mnm_refGPA,
					    gint codelen,
					    gpointer user_data)
{
  /*
    The xml node we are in is structured this way:
    
    <polseqdata version="0.1">
    <polseqinfo>
    <type>protein</type>
    <name>hamster P-glycoprotein</name>
      
    And the xml_node parameter points to the 
    
    <polseqdata version="0.1"> element tag:
    ^
    |
    +----- here we are right now.
     
    Which means that xml_node->name == "polseqdata" and that
    we'll have to go one step down to the first child of the 
    current node in order to get to the <polseqinfo> element.
    
  */
  PxmPolymer *polymer = NULL;
  PxmProp *prop = NULL;
  
  xmlNodePtr xml_child_node = NULL;

  gchar *help = NULL;
  
  gint count = -1;
    
  /*
    The array of reference monomers cannot be NULL or empty.
  */
  g_assert (mnm_refGPA != NULL && mnm_refGPA->pdata != NULL);

  /*
    Make sure we have parameters pointing bona fide to the right
    xml element.
  */
  g_assert (xml_node != NULL);
  g_assert (xml_doc != NULL);

  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseqdata"));
  
  polymer = pxmchem_polymer_new ();

  polymer->version = (gchar *) xmlGetProp (xml_node, 
					   (guchar *) "version");
  g_assert (polymer->version != NULL);
  

  /*
    Now go to the first child of current node: <polseqinfo>.
  */
  xml_node = xml_node->children;

  /*
    From a rigorous XML parsing point of view, the blanks found in
    the XML document are considered to be nodes, and we have to detect
    these and take proper action: go next sibling (next blank) as long
    as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /*
    Check that we have effectively a <polseqinfo> element here.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseqinfo"));


  xml_child_node = xml_node;

  /*
    The plminfo structure is allocated in pxmchem_polymer_new (), so 
    we first have to free it  before calling the function below that
    allocates also a plminfo instance.
  */
  g_assert (polymer->plminfo != NULL);
  pxmchem_plminfo_free (polymer->plminfo);
  
  polymer->plminfo = 
    pxmchem_polymer_render_xml_node_polseqinfo (xml_doc,
						xml_child_node,
						NULL);
  
  /*
    Now go to the next child of <polseqinfo>, which is <polseq>
  */
  xml_node = xml_node->next;

  /*
    From a rigorous XML parsing point of view, the blanks found in
    the XML document are considered to be nodes, and we have to detect
    these and take proper action: go next sibling (next blank) as long
    as blanks are encountered.
  */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /*
    Check that we have effectively a <polseq> element here.
  */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "polseq"));
  
  xml_child_node = xml_node;

  count = pxmchem_polymer_render_xml_node_polseq (xml_doc,
						  xml_child_node,
						  polymer->monomerGPA,
						  mnm_refGPA,
						  codelen,
						  NULL);
  if (count == -1)
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	  _("%s@%d: failed to render the polseq node\n"),
	   __FILE__, __LINE__);



  /*
    The <prop> elements that follow the <polseq> element are
    optional, and there may be more than one. So, the best is
    to go while ().
  */
  /*
    Now go to the next child of <polseq>, which may be <prop>
  */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      g_assert (0 == strcmp ((gchar *) xml_node->name, "prop"));
  
      /*
	Just get the prop's name so that we can get to the
	proper xml renderer plugin function, without modifying
	the xml_node, since the pluging function expects a node 
	pointer corresponding to <prop> and not <name>.
      */
      xml_child_node = xml_node->children;
      while (TRUE == xmlIsBlankNode (xml_child_node))
	xml_child_node = xml_child_node->next;
  
      g_assert (0 == strcmp ((gchar *) xml_child_node->name, "name"));

      help = 
	(gchar *) xmlNodeListGetString (xml_doc, xml_child_node->xmlChildrenNode, 1);
      g_assert (help != NULL);
      
      libpolyxmass_prop_renderer_xml_prop_plugin = 
	libpolyxmass_prop_choose_render_xml_prop_plugin (help);
      
      if (libpolyxmass_prop_renderer_xml_prop_plugin == NULL)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		_("%s@%d: plugin prop: '%s' has no registered xml renderer\n"),
		 __FILE__, __LINE__, help);
	  
	  /*
	    Go to the next <prop>, if any.
	  */
	  xml_node = xml_node->next;
	  /*
	    From a rigorous XML parsing point of view, the blanks
	    found in the XML document are considered to be nodes, and
	    we have to detect these and take proper action: go next
	    sibling (next blank) as long as blanks are encountered.
	  */
	  while (TRUE == xmlIsBlankNode (xml_node))
	    xml_node = xml_node->next;
	}
      
      g_free (help);
      
      prop = 
	libpolyxmass_prop_renderer_xml_prop_plugin (xml_doc, xml_node, NULL);
      g_assert (prop != NULL);

      /*
	Append the newly allocated prop instance to the array of the 
	polymer.
      */
      g_ptr_array_add (polymer->propGPA, prop);

      /*
	Go to the next <prop>, if any.
      */
      xml_node = xml_node->next;

      /*
	From a rigorous XML parsing point of view, the blanks found in
	the XML document are considered to be nodes, and we have to detect
	these and take proper action: go next sibling (next blank) as long
	as blanks are encountered.
      */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  return polymer;
}

