/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-formula.h"
#include "pxmchem-atom.h"



gboolean
pxmchem_formula_parse (gchar *formula, GPtrArray *atom_refGPA,
		       gint times, GPtrArray *GPA)
{
  /* We are asked to parse a formula in order to check that it 
   * is syntactically valid. A formula is something like
   * "H2O2" for water, for example, or event the "H2O" form.
   *
   * If the GPA is non-NULL, then all the PxmAtomcount instances
   * that are created during parsing are added to this array, 
   * which most probably means that a composition/mass calculation 
   * is going on. 
   *
   * The 'times' parameter governs the number of times the count results
   * need to be calculated into the GPA array. For example 
   * one may ask that a given formula be taken into account ten times.
   * The 'times' value may have a negative sign, in which case the
   * operation is for DEC, otherwise it is for INC.
   */
  gchar parsed_symb [MAX_ATOM_SYMBOL_LEN + 1];
  gchar prohibited_chars [] = { "\n\t\a,\\'" "!()[]{}" };
  gchar cur_char = '\x0';
  gchar *bad_char = NULL;

  GString *gs_atomcount = NULL;
  PxmAtomcount atomcount;

  gint atom_idx = 0;
  gint iter_formula = 0;
  gint iter_symbol = 0;
  gint iter_count = 0;

  gboolean was_upper = FALSE;
  
  gboolean was_digit = FALSE;


  g_assert (formula != NULL);
  g_assert (atom_refGPA != NULL);
  
  g_return_val_if_fail (strlen (formula) >= 0, FALSE);
  
  memset (parsed_symb, '\x0', MAX_ATOM_SYMBOL_LEN + 1);

  /* printf ("pxmchem_formula_parse for formula '%s'\n", formula); */
  

  /* Check some basic error conditions.
   */

  /* The formula should start with an uppercase letter, without digit
   */
  if (0 == isalpha (formula [0]) || 0 == isupper (formula [0]))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: the formula is incorrect: '%s'\n"),
	     __FILE__, __LINE__, formula);

      return FALSE;
    }

  /* The formula should not contain spaces or newline or whatever 
   * character contained in the prohibited_chars string.
   */
  bad_char = strpbrk (formula, prohibited_chars);

  if (bad_char != NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: the formula is incorrect: '%s'\n"),
	     __FILE__, __LINE__, formula);

      return FALSE;
    }

  /* Initialize the atomcount members (count number and atom string).
   */
  atomcount.count = 0;
  memset (atomcount.atom, '\x0', MAX_ATOM_SYMBOL_LEN + 1);
		
  /* Allocate/initialize the GString which will hold the growing
   * atom symbol, by appending atom symbol characters at each
   * iteration in the formula.
   */
  gs_atomcount = g_string_new ("");
  g_assert (gs_atomcount != NULL);


  /* ****************** REAL PARSING STUFF ******************** 
   */

  while (formula [iter_formula] != '\x0')
    {
      cur_char = formula [iter_formula];

      if (isalpha (cur_char))
	{
	  /* Character is alpha.
	   */
	  if (isupper (cur_char))
	    {
	      /* Character is uppercase.
	       */

	      /* In a formula, a character may be uppercase only
	       * when :
	       *
	       * 1. it is the first char in the formula. In this case
	       * nothing really special is to be done here, unless
	       * put the character in the parsed_symbol holder for
	       * next run.
	       *
	       * 2. it is the first char of a new atom symbol after any
	       * of the three following cases, then that means that
	       * a symbol had been already parsed.
	       *
	       *   a. a digit has been encountered, like when
	       *      iterating 'O' in "H2O2". 
	       *      This means that (was_digit = TRUE)
	       *      a symbol/count pair has just been parsed, and
	       *      this pair should be processed before going on.
	       *
	       *   b. a lowercase character 'a' when no digit is given
	       *      after a previously parsed atom symbol "Ca", like
	       *      the 'O' after "Ca" in the formula "C5H19CaO3".
	       *      This means that (was_lower == TRUE)
	       *      a symbol has just been parsed, with def count = 1, 
	       *      and this pair should be processed before going on.
	       *
	       *   c. an uppercase character 'N' when no digit is given
	       *      after a previously parsed atom symbol 'N', like
	       *      the 'O' after 'N' in the formula "C5H19NO3".
	       *      This means that (was_upper == TRUE)
	       *      a symbol has just been parsed, with def count = 1, 
	       *      and this pair should be processed before going on.
	       */
	      if (iter_formula == 0)
		{
		  /* Character is the very first character of the
		   * formula. We do nothing and let the
		   * code below start a new atom symbol with this 
		   * uppercase character.
		   */
		}
	      
	      else if (was_digit)
		{
		  /* Check that the atom that was parsed right before
		   * this iteration is known, i.e. its
		   * symbol is present in the GPtrArray of atoms 
		   */
		  if (FALSE == pxmchem_atom_known_by_symbol (parsed_symb,
							     &atom_idx,
							     atom_refGPA))
		    {
		      g_string_free (gs_atomcount, TRUE);

		      if (atom_idx == -1)
			{
			  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
				 _("%s@%d: failed to check if atom"
				 " is known: '%s'\n"),
				 __FILE__, __LINE__, parsed_symb);

			  return FALSE;
			}
		      else
			{
			  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
				_("%s@%d: the atom is unknown: '%s'\n"),
				 __FILE__, __LINE__, parsed_symb);

			  return FALSE;
			}
		    }

		  /* Set the atomcount atom member.
		   */
		  strcpy (atomcount.atom, parsed_symb);

		  /* Set the atomcount count member. 
		   */
		  if (FALSE == libpolyxmass_globals_strtoi (gs_atomcount->str,
							    &atomcount.count,
							    10))
		    {
		      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			     _("%s@%d: failed to convert from '%s' to gint\n"),
			     __FILE__, __LINE__, gs_atomcount->str);

		      g_string_free (gs_atomcount, TRUE);

		      return FALSE;
		    }

		  /* Apply the "times" factor to the atomcount count
		   * member.
		   */
		  atomcount.count = atomcount.count * times;
		  
		  /* Reset to initial values the variables needed to 
		   * parse the formula.
		   */
		  /* The atom symbol string.
		   */
		  iter_symbol = 0;	
		  memset (parsed_symb, '\x0', MAX_ATOM_SYMBOL_LEN + 1);

		  /* The atom count string.
		   */
		  iter_count = 0;
		  g_string_free (gs_atomcount, TRUE);
		  gs_atomcount = g_string_new ("");


		  /* If user wanted to append the newly created pair
		   * to the atomcount array, do it.
		   * 
		   * The function below will iterate in the array
		   * and check if there is already a atomcount instance
		   * with the same symbol as ours. Depending on the
		   * inc_or_dec parameter, action will be taken to 
		   * increment, subtract, the count of symbol. Note that
		   * if the action is increment and no atomcount
		   * is found in the array, the one passed as parameter
		   * will be duplicated in order to be added to the
		   * GPA.
		   */
		  if (GPA != NULL)
		    pxmchem_formula_set_atomcount_in_GPA (GPA, 
							  &atomcount);
		}		
	      /* end was_digit */

	      else if (!was_digit) /* == was_alpha */
		{
		  /* Check that the atom that was parsed right before
		   * this iteration is known, i.e. its
		   * symbol is present in the GPtrArray of atoms 
		   */
		  if (FALSE == pxmchem_atom_known_by_symbol (parsed_symb,
							     &atom_idx,
							     atom_refGPA))
		    {
		      g_string_free (gs_atomcount, TRUE);

		      if (atom_idx == -1)
			{
			  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
				 _("%s@%d: failed to check if atom"
				 " is known: '%s'\n"),
				 __FILE__, __LINE__, parsed_symb);

			  return FALSE;
			}
		      else
			{
			  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
				 _("%s@%d: the atom is unknown: '%s'\n"),
				 __FILE__, __LINE__, parsed_symb);

			  return FALSE;
			}
		    }

		  /* Set the atomcount atom member.
		   */
		  strcpy (atomcount.atom, parsed_symb);

		  /* Set the atomcount count member to 1, as we 
		   * set to 1 by default the count of atoms of which
		   * the symbol is not followed by a number in a formula.
		   * "H2O" equals to "H2O1".
		   */
		  atomcount.count = 1;
		  
		  /* Apply the "times" factor to the atomcount count
		   * member.
		   */
		  atomcount.count = atomcount.count * times;
		  
		  /* Reset to initial values the variables needed to 
		   * parse the formula.
		   */
		  /* The atom symbol string.
		   */
		  iter_symbol = 0;	
		  memset (parsed_symb, '\x0', MAX_ATOM_SYMBOL_LEN + 1);

		  /* If user wanted to append the newly created pair
		   * to the atomcount array, do it.
		   * 
		   * The function below will iterate in the array
		   * and check if there is already a atomcount instance
		   * with the same symbol as ours. Depending on the
		   * inc_or_dec parameter, action will be taken to 
		   * increment, subtract, the count of symbol. Note that
		   * if the action is increment and no atomcount
		   * is found in the array, the one passed as parameter
		   * will be duplicated in order to be added to the
		   * GPA.
		   */
		  if (GPA != NULL)
		    pxmchem_formula_set_atomcount_in_GPA (GPA, 
							  &atomcount);
		}
	      
	      /* Now that we have terminated a previously 
	       * "waiting-to-complete" atom (IF ANY) , 
	       * we can start a new one with the current __uppercase__
	       * character that sits in the cur_char variable.
	       */
	      memset (parsed_symb + iter_symbol, cur_char, 1);
	      was_digit = FALSE;
	      was_upper = TRUE;
	      
	      iter_symbol++;
	      iter_formula++;

	      continue;
	    }			
	  /* end of if (isupper (cur_char))
	   */

	  else
	    {
	      /* Character is lowercase.
	       */

	      /* For example, it cannot be anything other than
	       * the second lowercase char of a two-char atom symbol,
	       * exactly like 'a' is the second lowercase char of "Ca".
	       */
	      if (!was_upper) /* == was_lower */
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			 _("%s@%d: two lowercase chars in a row in this"
			 " formula: '%s'\n"),
			 __FILE__, __LINE__, formula);
		  
		  g_string_free (gs_atomcount, TRUE);

		  return FALSE;
		}

	      /* Set this character into the second slot of 
	       * parsed_symb.
	       */
	      memset (parsed_symb + iter_symbol, cur_char, 1);
	      was_upper = FALSE;
	      
	      /* Reset the iterator for symbol chars.
	       */
	      iter_symbol--;	
	      iter_formula++;

	      continue;

	    }			
	  /* end of is character is lowercase */
	}	
      /* end of character is alphabetic */

      else
	{
	  /* Character is digit.
	   */

	  /* Character is a digit, which either is the first or another
	   * one in the representation of the number of atoms, ie might
	   * be '1' or '7' in "Ca171" - thus simply set it 
	   */
	  gs_atomcount = g_string_append_c (gs_atomcount, cur_char);
	  was_digit = TRUE;

	  /* For next atom, after digits have finished parsing in case
	   * there are more than only one.
	   */
	  iter_count++;
	  iter_formula++;
	}			
      /* end of character is digit */
      
    }				
  /* end of while */


  /* The while loop accepted only non-null characters,
   * but when the last null char arrives, it's too late to construct
   * the atom that was being constructed in the loop, so we should
   * deal with the interrupted material now.
   */

  /* Beware that if we are here it is only because the formula
   * string came to and end, which does not mean that it came to and end
   * correctly. For example imagine we have a formula "C2Z", we
   * will arrive at this point with an 'Z' in the parsed_symb, only
   * to discover that 'Z' does not correspond to any valid atom symbol.
   *
   * Also if we had parsed the formula "C2H", we certainly would have
   * a valid hydrogen ('H') but without a gs_atomcount->str representing
   * the atom count, which means that we have to set the count to 1
   * by default.
   */

  /* Check that the atom is known, i.e. its
   * symbol is present in the GPtrArray of atoms .
   */
  if (FALSE == pxmchem_atom_known_by_symbol (parsed_symb, &atom_idx,
					     atom_refGPA))
    {
      if (atom_idx == -1)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: failed to check if atom is known: '%s'\n"),
	 __FILE__, __LINE__, parsed_symb);
      
	  g_string_free (gs_atomcount, TRUE);

	  return FALSE;
	}
      else
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: the atom is unknown: '%s'\n"),
	 __FILE__, __LINE__, parsed_symb);

	  g_string_free (gs_atomcount, TRUE);

	  return FALSE;
	}
    }

  /* Set the atomcount atom member.
   */
  strcpy (atomcount.atom, parsed_symb);
  
  /* Set the atomcount count member. 
   */
  if (strlen (gs_atomcount->str) <= 0)
    gs_atomcount = g_string_append_c (gs_atomcount, '1');
  else
    strcpy (atomcount.atom, parsed_symb);
  
  if (FALSE == libpolyxmass_globals_strtoi (gs_atomcount->str,
				  &atomcount.count,
				  10))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to convert from '%s' to gint\n"),
	     __FILE__, __LINE__, gs_atomcount->str);
      
      g_string_free (gs_atomcount, TRUE);
      
      return FALSE;
    }
  
  /* Now that we have a correct atomcount.count 
   * member we can apply the "times" factor to it.
   */
  atomcount.count = atomcount.count * times;
  
  g_string_free (gs_atomcount, TRUE);

  /* If user wanted to append the newly created pairs
   * to the atomcount array, do it.
   * 
   * The function below will iterate in the array
   * and check if there is already a atomcount instance
   * with the same symbol as ours. Depending on the
   * inc_or_dec parameter, action will be taken to 
   * increment, subtract, the count of symbol.
   */
  if (GPA != NULL)
    pxmchem_formula_set_atomcount_in_GPA (GPA, &atomcount);

  
  return TRUE;
}


gboolean
pxmchem_formula_set_atomcount_in_GPA (GPtrArray *GPA,
				      PxmAtomcount* atomcount)
{
  /* The atomcount will
   * -according to the inc_or_dec parameter-  be integrated
   * or removed from the GPA. Integrated means that 
   * if a PxmAtomcount instance is found in the array with the same 
   * symbol, the count of the atomcount is summed to the  count
   * of the PxmAtomcount found in the GPtrArray.
   * Removed means, on the contrary that the count of the PxmAtomcount
   * found in the GPtrArray is subtracted by the count in the 
   * atomcount parameter. 
   * 
   * If no PxmAtomcount is found and inc_or_dec asks that atomcount be
   * integrated, then it is first duplicated, and then appended. 
   * Otherwise it is an error, as we cannot have negative counts.
   */
  gint iter = 0;

  PxmAtomcount * iter_atomcount = NULL;
  PxmAtomcount * new_atomcount = NULL;

  g_assert (GPA != NULL);
  g_assert (atomcount != NULL);

  if (atomcount->count == 0)
    return TRUE;
  
  /* Iterate in the array in search for a PxmAtomcount instance having
   * the same symbol member as the one of the atomcount passed as
   * parameter.
   */
  for (iter = 0; iter < GPA->len; iter++)
    {
      iter_atomcount = g_ptr_array_index (GPA, iter);
      g_assert (iter_atomcount != NULL);

      /* If we find the item we are looking for, just
       * increment/decrement (according to the sign of the
       * atomcount->count parameter) the count member of the found
       * PxmAtomcount and return.
       */
      if (0 == strcmp (iter_atomcount->atom, atomcount->atom))
	{
	  /* If atomcount->count is < 0 the operation is a lighting 
	   * operation, otherwise it is a heavying operation.
	   */
	  iter_atomcount->count += atomcount->count;

	  /* Just inform the user that the count if negative for current
	   * atom.
	   */
	  if (iter_atomcount->count < 0)
	    {
	      /*
		g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
		_("%s@%d: the atomcount for atom '%s' has a negative"
		" count: '%d'\n"),
		__FILE__, __LINE__,
		iter_atomcount->atom,
		iter_atomcount->count);
	      */
	    }
	  
	  return TRUE;
	  
	}
      
    }

  /* If we are here, that means that we did not find our item in the
   * array. First we have to duplicate the PxmAtomcount passed as
   * parameter.
   */
  new_atomcount = pxmchem_atomcount_dup (atomcount);
  g_ptr_array_add (GPA, new_atomcount);
  
  /* Just inform the user that the count if negative for current
   * atom.
   */
  if (new_atomcount->count < 0)
    {
      /*
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: the atomcount for atom '%s' has a negative"
	" count: '%d'\n"),
	__FILE__, __LINE__,
	new_atomcount->atom,
	new_atomcount->count);
      */
    }
  
  return TRUE;
}


gboolean 
pxmchem_formula_check (gchar *formula, GPtrArray *atom_refGPA)
{
  /* Last arg is NULL (atomcount GPtrArray) so the function will only
   * parse the formula, and not take any INC or DEC action.
   */
  return pxmchem_formula_parse (formula, atom_refGPA, +1, NULL);
}


gboolean
pxmchem_actform_split_parts (gchar *actform,
			     GPtrArray *atom_refGPA,
			     gchar **plus,
			     gchar **minus)
{
  /* The actform param is of the form "-C1H2S1+O1" or "+O1-C1H2S1",
   * which is fundamentally a chemical reaction formalism. From this
   * pair of actform substrings (-C1H2S1 and +O1) we have to calculate a
   * plus formula and a minus formula (with no signs after that the
   * parsing has been accomplished). Further, the actform that we get
   * as parameter may also be a formula without any sign, in which
   * case the the actform string is simply treated as a plus formula.
   *
   * Note that the syntax imposes that at least one '+' or '-' be
   * present in the string.
   *
   * Equally important is that the actform parts are validated
   * for syntactical correctness before the function returns. Thus,
   * if any of the parts is syntactically incorrect, its corresponding
   * pointer is set to NULL, and FALSE is returned.
   *
   * Note that an actform can consist of only one '+' or '-'
   * part. In this case, the opposite part is returned as a NULL pointer.
   */
  gint length = 0;
  gint iter = 0;

  gsize size = 0;
  
  /* By default, if no '+' sign is found and no '-' neither, 
   * '+' is considered.
   */
  gboolean was_plus = TRUE;
  gboolean wait_formula = FALSE;
  gboolean was_formula = FALSE;
  gboolean error = FALSE;

  GString *gs_formula = NULL;

  GString *gs_plus = NULL;
  GString *gs_minus = NULL;

  g_assert (actform != NULL);
  g_assert (atom_refGPA != NULL);


  /* Mininum length of formula is 1 char, as in "H". 
   */
  size = strlen (actform);
  g_assert (size < G_MAXINT);
  length = (gint) size;

  if (length <= 0) 
    return FALSE;


  /* Allocate needed GString instances.
   */
  gs_formula = g_string_new ("");
  gs_plus = g_string_new ("");
  gs_minus = g_string_new ("");

  
  /* printf ("pxmchem_actform_split_parts for actform '%s'\n", actform); */
  

  /* Start with some easy checks.
   */
  if (NULL == strchr (actform, '+') && NULL == strchr (actform, '-'))
    {
      /* There are no signs in actform. That means that it is by 
       * default considered to be a plus formula, as if it were
       * prefixed with '+'.
       */
      gs_formula = g_string_append (gs_formula, actform);
      was_plus = TRUE;
      was_formula = TRUE;

      /* Set the iter variable to the last '\x0' character index,
       * so that we do not enter the while loop below.
       */
      size = strlen (actform);
      g_assert (size < G_MAXINT);
      iter = (gint) size;

    }
  
  while (actform [iter] != '\x0')
    {
      if (actform [iter] == '+' && wait_formula == TRUE)
	{
	  error = TRUE;
	  break;
	}

      else if (actform [iter] == '+' && was_formula == TRUE)
	{
	  /* Starting a new formula, so take into account 
	   * the one that just ended.
	   */
	  if (was_plus == TRUE)
	    gs_plus = g_string_append (gs_plus, gs_formula->str);
	  else
	    gs_minus = g_string_append (gs_minus, gs_formula->str);

	  /* and now set the gbooleans to proper values for next round
	   */
	  wait_formula = TRUE;
	  was_plus = TRUE;
	  
	  g_string_free (gs_formula, TRUE);
	  gs_formula = g_string_new ("");
	  
	  iter++;
	  continue;
	}

      else if (actform [iter] == '+')
	{
	  /* Starting with '+' a totally new string.
	   */
	  was_plus = TRUE;
	  wait_formula = TRUE;
	  iter++;
	  continue;
	}

      else if (actform [iter] == '-' && wait_formula == TRUE)
	{
	  error = TRUE;
	  break;
	}

      else if (actform [iter] == '-' && was_formula == TRUE)
	{
	  /* Starting a new formula, so take into account 
	   * the one that just ended.
	   */
	  if (was_plus == TRUE)
	    gs_plus = g_string_append (gs_plus, gs_formula->str);
	  else
	    gs_minus = g_string_append (gs_minus, gs_formula->str);

	  /* Set the gboolean variables to proper values for next round.
	   */
	  wait_formula = TRUE;
	  was_plus = FALSE;
	  
	  g_string_free (gs_formula, TRUE);
	  gs_formula = g_string_new ("");
	  
	  iter++;
	  continue;
	}

      else if (actform [iter] == '-')
	{
	  /* Starting with '-' a totally new string.
	   */
	  was_plus = FALSE;
	  wait_formula = TRUE;
	  iter++;
	  continue;
	}

      else
	{
	  gs_formula = 
	    g_string_append_c (gs_formula, actform [iter]);
	  was_formula = TRUE;
	  
	  /* We had at least one char, so let's say in first approximation
	   * that at next run we could close the formula.
	   */
	  wait_formula = FALSE;
	  iter++;
	  continue;
	}
    }

  /* We are here either because:
   * 1. we arrived to the end of the parsed string, in which case 
   * we still have to manage the last portion of the string.
   *
   * 2. we were thrown out by break statement with error set to TRUE.
   */
  if (was_formula == TRUE && strlen (gs_formula->str) > 0)
    {
      if (was_plus == TRUE)
	gs_plus = g_string_append (gs_plus, gs_formula->str);
      else
	gs_minus = g_string_append (gs_minus, gs_formula->str);
    }

  g_string_free (gs_formula, TRUE);

  if (error == TRUE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	 _("%s@%d: the actform is incorrect: '%s'\n"),
	 __FILE__, __LINE__, actform);

      g_string_free (gs_plus, TRUE);
      g_string_free (gs_minus, TRUE);

      return FALSE;
    }

  /* We now should check the validity of the formulae that we may have
   * in the gs_plus and/or gs_minus GString objects.  Remember that 
   * gs_minus may be empty (if no '-' sign is found) and that gs_plus 
   * may be empty (if '-' is the first character in the actform and
   * that no other '+' character is found).
   */

  /* The plus formula.
   */
  if (strlen (gs_plus->str) > 0)
    {
      if (FALSE == pxmchem_formula_check (gs_plus->str, atom_refGPA))
	{
	  error = TRUE;
	  g_string_free (gs_plus, TRUE);
	  
	  if (plus != NULL)
	    *plus = NULL;
	}
      else
	{
	  /* printf ("The plus component: '%s'\n", gs_plus->str); */
	  
	  if (plus != NULL)
	    {
	      (*plus) = gs_plus->str;
	      g_string_free (gs_plus, FALSE);
	    }
	  else
	    g_string_free (gs_plus, TRUE);
	}
    }
  else
    g_string_free (gs_plus, TRUE);
  
  /* The minus formula.
   */
  if (strlen (gs_minus->str) > 0)
    {
      if (FALSE == pxmchem_formula_check (gs_minus->str, atom_refGPA))
	{
	  error = TRUE;
	  g_string_free (gs_minus, TRUE);

	  if (minus != NULL)
	    *minus = NULL;
	}
      else
	{
	  /* printf ("The minus component: '%s'\n", gs_minus->str); */

	  if (minus != NULL)
	    {
	      (*minus) = gs_minus->str;
	      g_string_free (gs_minus, FALSE);
	    }
	  else
	    g_string_free (gs_minus, TRUE);
	}
    }
  else
    g_string_free (gs_minus, TRUE);
  
  /* printf ("Terminated the parsing--------------------------\n\n"); */
  
  return !error;
}


gboolean
pxmchem_actform_check (gchar *actform, GPtrArray *atom_refGPA)
{

  g_assert (actform != NULL);
  
  return pxmchem_actform_split_parts (actform, atom_refGPA, NULL, NULL);
}



gchar * 
pxmchem_formula_make_string_from_atomcount_GPA (GPtrArray *GPA)
{
  /* We get a pointer to a GPtrArray in which PxmAtomcount instances
   * reflect the composition of a compound. We just iterate in this
   * array and for each PxmAtomcount instance we elongate a string
   * that will, in the end, be a formula string.
   */
  gint iter = 0;

  PxmAtomcount *atomcount = NULL;

  GString *gs = g_string_new ("");
  gchar *help = NULL;

  g_assert (GPA != NULL);

  /* Iterate through the items in GPtrArray and get the atom and count
   * of the each iterated PxmAtomcount instance.
   */
  for (iter = 0; iter < GPA->len; iter++)
    {
      atomcount = g_ptr_array_index (GPA, iter);
      g_assert (atomcount != NULL);

      g_string_append_printf (gs, "%s%d",
			      atomcount->atom, atomcount->count);
    }

  help = gs->str;
  
  g_string_free (gs, FALSE);

  return help;
}


/* MASS CALCULATION FUNCTIONS
 */
gboolean
pxmchem_formula_account_mass (gchar *formula, GPtrArray *atom_refGPA,
			      gint times, PxmMasspair *masspair)
{
  /* We get a formula, and we have to calculate a mass for it. Thus we
   * first have to parse it and make an array of atomcount objects.
   * The calculation will be done by iterating the atomcount array.
   *
   * Note that the inc or dec is determined by the sign of the times
   * parameter: if negative, the formula is dec, if positive it is inc.
   */
  GPtrArray *GPA = NULL;
  PxmAtomcount *atomcount = NULL;
  
  gchar *symbol = NULL;
  
  PxmMasspair atom_mp;
  
  gint count = 0;
  gint iter = 0;
  
  g_assert (formula != NULL);
  g_assert (masspair != NULL);
  g_assert (atom_refGPA != NULL);


  atom_mp.mono = 0;
  atom_mp.avg = 0;
  
  /* Allocate the array of atomcount objects.
   */
  GPA = g_ptr_array_new ();
  
  /* Parse the formula and pass as param the array into which all the 
   * allocate atomcount object are to be stored during the formula 
   * parsing. The times factor is taken into account both for 
   * value and for sign (if < 0 DEC, if > 0 INC).
   */
  if (FALSE == pxmchem_formula_parse (formula, atom_refGPA, times, GPA))
    {
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	       _("%s@%d: failed to parse formula: '%s'\n"),
	       __FILE__, __LINE__, formula);

	g_ptr_array_free (GPA, TRUE);
	
	return FALSE;
    }
  
  /* We now have an array of atomcount instances that we need to
   * translate into masses.
   */
  for (iter = 0; iter < GPA->len; iter++)
    {
      atomcount = g_ptr_array_index (GPA, iter);
      g_assert (atomcount != NULL);
      
      symbol = atomcount->atom;
      count = atomcount->count;
      
      if (FALSE == pxmchem_atom_fill_masspair (symbol, &atom_mp, atom_refGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to get masses for atom: '%s'\n"),
		 __FILE__, __LINE__, symbol);
	  
	  pxmchem_atomcount_GPA_free (GPA);
	  
	  return FALSE;
	}
      
      /* At this point we do not care of the times factor nor sign
       * because they were already taken into account during the
       * formula parsing. Thus, we just add the atom_mp.mono * count,
       * and if the count is negative this will equal to a "lighting"
       * operation, otherwise it is "heavying" operation.
       */
      masspair->mono += atom_mp.mono * count;
      masspair->avg += atom_mp.avg * count;
    }
  
  /* We now have to free all the atomcount elements that we created
   * in order to parse the formula !
   */
  pxmchem_atomcount_GPA_free (GPA);

  return TRUE;
}



gboolean
pxmchem_actform_account_mass (gchar *actform, GPtrArray *atom_refGPA,
			      gint times, PxmMasspair *masspair)
{
  gchar *minus = NULL;
  gchar *plus = NULL;
  
  g_assert (actform != NULL);
  g_assert (masspair != NULL);
  g_assert (atom_refGPA != NULL);
  
  
  if (FALSE == pxmchem_actform_split_parts (actform, atom_refGPA, 
					    &plus, &minus))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to split parts of actform: '%s'\n"),
	     __FILE__, __LINE__, actform);
      
      if (plus != NULL)
	g_free (plus);
      
      if (minus != NULL)
	g_free (minus);
      
      return FALSE;
    }


#if 0
  /***************************************************************/
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	 _("%s@%d: actform: '%s' was split into"
	   " plus: '%s' and minus: '%s' - times is: '%d'\n"),
	 __FILE__, __LINE__, actform, plus, minus, times);
  
  
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	 _("%s@%d: initial masspair:"
	   " mono=%f, avg=%f\n"),
	 __FILE__, __LINE__, 
	 masspair->mono, masspair->avg);
  /***************************************************************/
#endif


  /* We now have one or two formulae in plus and/or minus. Just use
   * them to account the mass for each. 
   */
  if (plus != NULL)
    {
      /* Since this is the 'plus' formula, and times may have a sign
       * we do not modify the sign of times. Thus, if times is negative,
       * the overall effect is going to be negative.
       */
      if (FALSE == pxmchem_formula_account_mass (plus, atom_refGPA,
						 times, masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for formula plus: '%s'\n"),
		 __FILE__, __LINE__, plus);
	  
	  g_free (plus);
	  
	  if (minus != NULL)
	    g_free (minus);
	  
	  return FALSE;
	}
      
      g_free (plus);

#if 0
  /***************************************************************/
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	     _("%s@%d: after accounting for 'plus' with times=%d:"
	       " mono=%f, avg=%f\n"),
	     __FILE__, __LINE__, 
	     times, masspair->mono, masspair->avg);
  /***************************************************************/
#endif
    }
  
  
  if (minus != NULL)
    {
      /* Since this is the 'minus' formula, and times may have a sign
       * we multiply by -1 times. Thus, if times is negative,
       * the overall effect is going to be positive.
       */
      if (FALSE == pxmchem_formula_account_mass (minus, atom_refGPA,
						 -1 * times, masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for formula minus: '%s'\n"),
		 __FILE__, __LINE__, minus);
	  
	  g_free (minus);
	  
	  return FALSE;
	}
      
      g_free (minus);
      
#if 0
  /***************************************************************/
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	     _("%s@%d: after accounting 'minus' with times=%d:"
	       " mono=%f, avg=%f\n"),
	     __FILE__, __LINE__, 
	     -1 * times, masspair->mono, masspair->avg);
  /***************************************************************/
#endif
    }
  
  return TRUE;
}


gboolean
pxmchem_actform_account_elemcompos (gchar *actform, GPtrArray *atom_refGPA,
				    gint times, GPtrArray *acGPA)
{
  gchar *minus = NULL;
  gchar *plus = NULL;
  
  g_assert (actform != NULL);
  g_assert (atom_refGPA != NULL);
  g_assert (acGPA != NULL);
  
  if (FALSE == pxmchem_actform_split_parts (actform, atom_refGPA, 
					    &plus, &minus))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to split parts of actform: '%s'\n"),
	     __FILE__, __LINE__, actform);
      
      if (plus != NULL)
	g_free (plus);
      
      if (minus != NULL)
	g_free (minus);
      
      return FALSE;
    }


#if 0
  /***************************************************************/
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	 _("%s@%d: actform: '%s' was split into"
	   " plus: '%s' and minus: '%s' - times is: '%d'\n"),
	 __FILE__, __LINE__, actform, plus, minus, times);
  
  
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
	 _("%s@%d: initial masspair:"
	   " mono=%f, avg=%f\n"),
	 __FILE__, __LINE__, 
	 masspair->mono, masspair->avg);
  /***************************************************************/
#endif


  /* We now have one or two formulae in plus and/or minus. Just use
   * them to account the mass for each. 
   */
  if (plus != NULL)
    {
      /* Since this is the 'plus' formula, and times may have a sign
       * we do not modify the sign of times. Thus, if times is negative,
       * the overall effect is going to be negative.
       */
      if (FALSE == pxmchem_formula_parse (plus, atom_refGPA, 
					  times, acGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to  account for formula plus: '%s'\n"),
		 __FILE__, __LINE__, plus);
	  
	  g_free (plus);
	  
	  if (minus != NULL)
	    g_free (minus);
	  
	  return FALSE;
	}
      
      g_free (plus);
    }
  
  
  if (minus != NULL)
    {
      /* Since this is the 'minus' formula, and times may have a sign
       * we multiply by -1 times. Thus, if times is negative,
       * the overall effect is going to be positive.
       */
      if (FALSE == pxmchem_formula_parse (minus, atom_refGPA, 
					  -1 * times, acGPA))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for formula minus: '%s'\n"),
		 __FILE__, __LINE__, minus);
	  
	  g_free (minus);
	  
	  return FALSE;
	}
      
      g_free (minus);
    }
  
  return TRUE;
}
