#define PHIDGETS_INTERNAL

#include <phidgets/quadservo.h>

#include <debug.h>
#include <assert.h>

#include <limits.h>

static unsigned short convert_to_12bit_pulse(PhidgetQuadServo* const qsc,
    PhidgetQuadServoSelector const servo, unsigned int const position)
{
  ASSERT(qsc);

  PhidgetServoMotor const* const srv = &qsc->servos[servo];
  unsigned short ret = srv->min_pulse  + (srv->max_pulse - srv->min_pulse)
    * ((float)position)/UINT_MAX;
  return ret & 0xfff;
}

PhidgetQuadServo* phidget_new_PhidgetQuadServo()
{
  TRACE("creating a new PhidgetQuadServo instance...");

  PhidgetQuadServo* ret = (PhidgetQuadServo*)malloc(sizeof(PhidgetQuadServo));
  if (!ret) {
    ERROR("could not allocate memory for PhidgetQuadServo instance.");
    return 0;
  }

  ret->phidget = phidget_new_Phidget();

  phidget_reset_PhidgetQuadServo(ret);
  return ret;
}

void phidget_delete_PhidgetQuadServo(PhidgetQuadServo** const qsc)
{
  if (!qsc || !*qsc) {
    ERROR("cannot delete NULL PhidgetQuadServo.");
    return;
  }

  free(*qsc);
  *qsc = 0;
}

void phidget_reset_PhidgetQuadServo(PhidgetQuadServo* const qsc)
{
  if (!qsc) {
    ERROR("cannot reset NULL PhidgetQuadServo.");
    return;
  }

  phidget_reset_Phidget(qsc->phidget);

  unsigned char i = 0;
  for (; i < 4; ++i)
    phidget_reset_PhidgetServoMotor(&qsc->servos[i]);
}

phidget_return phidget_quadservo_open(PhidgetQuadServo* const qsc,
    unsigned int serial, unsigned short retries)
{
  if (!phidget_is_initialised()) {
    ERROR("cannot open PhidgetQuadServo when Phidgets library has not been initialised.")
    return PHIDGET_RET_NOT_INITIALISED;
  }

  if (!qsc) {
    ERROR("cannot open NULL PhidgetQuadServo.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }
    
  if (phidget_quadservo_is_opened(qsc)) {
    ERROR("cannot open already opened PhidgetQuadServo.");
    return PHIDGET_RET_DEVICE_ALREADY_OPENED;
  } 

  TRACE("opening PhidgetQuadServo with serial number %d...", serial);

  HIDInterfaceMatcher matcher;
  matcher.vendor_id = PHIDGETS_USB_VENDORID;
  matcher.product_id = PHIDGETS_USB_PRODUCTID_QUADSERVO;

  phidget_return ret = phidget_open(qsc->phidget, 0, &matcher, serial, retries);
  if (ret != PHIDGET_RET_SUCCESS) return ret;

  NOTICE("successfully opened PhidgetQuadServo %s...", qsc->phidget->id);
  return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_quadservo_close(PhidgetQuadServo* const qsc)
{
  if (!qsc) {
    ERROR("cannot close NULL PhidgetQuadServo.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }
  
  if (phidget_quadservo_is_opened(qsc)) {

    TRACE("closing PhidgetQuadServo %s...", qsc->phidget->id);

    phidget_return ret = phidget_close(qsc->phidget);
    if (ret != PHIDGET_RET_SUCCESS) return ret;
  }
  else WARNING("attempt to close unopened PhidgetQuadServo.");
  
  NOTICE("successfully closed PhidgetQuadServo %s.", qsc->phidget->id);
  return PHIDGET_RET_SUCCESS;
}

bool phidget_quadservo_is_opened(PhidgetQuadServo const* const qsc)
{
  if (!qsc) WARNING("attempt to query open status of NULL PhidgetQuadServo.");
  return qsc && phidget_is_opened(qsc->phidget);
}

phidget_return phidget_quadservo_set_servo_parameters(PhidgetQuadServo* const qsc,
    PhidgetQuadServoSelector const servo, unsigned short const min,
    unsigned short const max, float const factor)
{
  if (!qsc) {
    ERROR("cannot set servo parameters of NULL PhidgetQuadServo.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }
    
  TRACE("setting parameters of servo motor %d of PhidgetQuadServo %s...",
      servo, qsc->phidget->id);

  return phidget_servomotor_set_parameters(&qsc->servos[servo], min, max, factor);
}

phidget_return phidget_quadservo_set_single_position(PhidgetQuadServo* const qsc,
    PhidgetQuadServoSelector const servo, unsigned int const position)
{
  if (!phidget_quadservo_is_opened(qsc)) {
    ERROR("cannot set servo position of unopened PhidgetQuadServo.");
    return PHIDGET_RET_DEVICE_NOT_OPENED;
  } 

  unsigned int p0 = (servo != 0) ? qsc->servos[0].position : position;
  unsigned int p1 = (servo != 1) ? qsc->servos[1].position : position;
  unsigned int p2 = (servo != 2) ? qsc->servos[2].position : position;
  unsigned int p3 = (servo != 3) ? qsc->servos[3].position : position;
  return phidget_quadservo_set_all_positions(qsc, p0, p1, p2, p3);
}

phidget_return phidget_quadservo_set_all_positions(PhidgetQuadServo* const qsc,
    unsigned int const p0, unsigned int const p1, unsigned int const p2,
    unsigned int const p3)
{
  if (!phidget_quadservo_is_opened(qsc)) {
    ERROR("cannot set servo position of unopened PhidgetQuadServo.");
    return PHIDGET_RET_DEVICE_NOT_OPENED;
  } 

  TRACE("sending positions 0x[%08x %08x %08x %08x] to PhidgetQuadServo %s.",
        p0, p1, p2, p3, qsc->phidget->id);

  unsigned short pulse0 = convert_to_12bit_pulse(qsc, 0, p0);
  unsigned short pulse1 = convert_to_12bit_pulse(qsc, 1, p1);
  unsigned short pulse2 = convert_to_12bit_pulse(qsc, 2, p2);
  unsigned short pulse3 = convert_to_12bit_pulse(qsc, 3, p3);

  TRACE("sending pulses 0x[%03hx %03hx %03hx %03hx] to PhidgetQuadServo %s.",
      pulse0, pulse1, pulse2, pulse3, qsc->phidget->id);

  unsigned char const PACKETLEN = 6;
  char packet[PACKETLEN];

  packet[0] = pulse0 & 0xff;
  packet[1] = ((pulse0 & 0xf00) >> 8) | ((pulse1 & 0xf00) >> 4);
  packet[2] = pulse1 & 0xff;
  packet[3] = pulse2 & 0xff;
  packet[4] = ((pulse2 & 0xf00) >> 8) | ((pulse3 & 0xf00) >> 4);
  packet[5] = pulse3 & 0xff;

  TRACE("sending packet 0x[%02hhx %02hhx %02hhx %02hhx %02hhx %02hhx] to "
      "PhidgetQuadServo %s.", packet[0], packet[1], packet[2], packet[3],
      packet[4], packet[5], qsc->phidget->id);

  /* TODO: export these constants */
  unsigned char const PATHLEN = 3;
  int const PATH[] = { PHIDGETS_HID_PATH_1, PHIDGETS_HID_PATH_2, 0xffa10004 };

  hid_return ret = hid_set_output_report(qsc->phidget->hid_iface,
      PATH, PATHLEN, packet, PACKETLEN);
  if (ret != HID_RET_SUCCESS) {
    qsc->phidget->hid_error = ret;
    WARNING("failed to send packet to PhidgetQuadServo %s.", qsc->phidget->id);
    return PHIDGET_RET_HID_ERROR;
  }

  qsc->servos[0].position = p0;
  qsc->servos[1].position = p1;
  qsc->servos[2].position = p2;
  qsc->servos[3].position = p3;
  
  TRACE("successfully sent a packet to PhidgetQuadServo %s.", qsc->phidget->id);
  return PHIDGET_RET_SUCCESS;
}

/* COPYRIGHT --
 *
 * This file is part of libphidgets, a user-space library for phidgets.
 * libphidgets is (c) 2003-2004 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
