
// File: test_timsxicextractor.cpp
// Created by: Olivier Langella
// Created on: 3/2/2021
//
/*******************************************************************************
 * Copyright (c) 2021 Olivier Langella
 *<olivier.langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// make test ARGS="-V -I 1,1"

// ./tests/catch2-only-tests [TimsXicExtractor] -s


#include <catch2/catch.hpp>

#include <QDebug>

#include <pappsomspp/vendors/tims/timsframebase.h>
#include <pappsomspp/mzrange.h>
#include <pappsomspp/msfile/msfileaccessor.h>
#include <pappsomspp/msrun/private/timsmsrunreaderms2.h>
#include <pappsomspp/msrun/private/timsmsrunreader.h>
#include <pappsomspp/pappsoexception.h>
#include <pappsomspp/processing/filters/filtersuitestring.h>
#include "config.h"
#include "common.h"
#include <time.h>


using namespace pappso;

TEST_CASE("Extracting XIC from timsdata.", "[TimsXicExtractor]")
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));
#if USEPAPPSOTREE == 1


  INFO("reading Tims data file start");
  /*
      pappso::MsFileAccessor accessor(
        "/gorgone/pappso/jouy/raw/2020_Tims_TOF/20201102_HeLa/"
        "11-2-2020_1_QC_HeLa10ng_219.d/analysis.tdf",
        "a1");
  */
  /*
      pappso::MsFileAccessor accessor(
        "/gorgone/pappso/fichiers_fabricants/Bruker/tims_doc/tdf-sdk/"
        "example_data/"
        "200ngHeLaPASEF_2min_compressed.d/analysis.tdf",
        "a1");
    */
//   pappso::MsFileAccessor accessor(
//     "/gorgone/pappso/jouy/raw/2021_Tims_TOF/20210126_HeLa/"
//     "1-26-2021_1_QC_HeLa10ng_825.d/analysis.tdf",
//     "a1");
//   

  pappso::MsFileAccessor accessor(
    "/data/test_tdf_quality/2-3-2021_1_HeLa10ng_871.d/analysis.tdf",
    "a1");


  accessor.setPreferedFileReaderType(pappso::MzFormat::brukerTims,
                                     pappso::FileReaderType::tims_ms2);


  pappso::MsRunReaderSPtr p_msreader =
    accessor.msRunReaderSp(accessor.getMsRunIds().front());

  REQUIRE(p_msreader != nullptr);

  REQUIRE(accessor.getFileReaderType() == pappso::FileReaderType::tims_ms2);

  pappso::TimsMsRunReaderMs2 *tims2_reader =
    dynamic_cast<pappso::TimsMsRunReaderMs2 *>(p_msreader.get());
  REQUIRE(tims2_reader != nullptr);


  pappso::TimsDataSp tims_data = tims2_reader->getTimsDataSPtr();

  std::vector<std::size_t> precursor_list = {2159};
  std::vector<pappso_double> mz_list      = {466.738, 467.239};

  std::vector<pappso::TimsXicStructure> xic_list;
  std::vector<pappso::TimsXicStructure> xic_struct_list;

  clock_t start = clock();
  for(auto precursor_id : precursor_list)
    {
      xic_list.push_back(tims_data.get()->getTimsXicStructureFromPrecursorId(
        precursor_id, PrecisionFactory::getPpmInstance(30.0)));
    }

  for(pappso::pappso_double mz : mz_list)
    {
      pappso::TimsXicStructure new_xic_struct =
        pappso::TimsXicStructure(xic_list[0]);
      new_xic_struct.mzRange =
        pappso::MzRange(mz, PrecisionFactory::getPpmInstance(30.0));
      new_xic_struct.xicSptr = std::make_shared<Xic>();
      xic_struct_list.push_back(new_xic_struct);
    }

  tims_data.get()->extractXicListByTimsXicStructureList(
    xic_struct_list, pappso::XicExtractMethod::sum, 100000);
  
  qInfo() << QString("Time taken: %1\n").arg((double)(clock() - start)/CLOCKS_PER_SEC);
  for(auto &&xic_struct : xic_struct_list)
    {
      REQUIRE(xic_struct.xicSptr.get() != nullptr);
      QFile fileods(QString("/data/test_tdf_quality/tims_xic_%1_%2.ods")
                      .arg(xic_struct.precursorId)
                      .arg(xic_struct.mzRange.getMz()));
      OdsDocWriter writer(&fileods);
      writer.writeLine();
      writer.writeCell("precursorId");
      writer.writeCell(xic_struct.precursorId);
      writer.writeLine();
      writer.writeCell("rtTarget");
      writer.writeCell(xic_struct.rtTarget);
      writer.writeLine();
      XicOdsWriter xic_writer(writer);
      xic_writer.write(*xic_struct.xicSptr.get());
      writer.close();
      fileods.close();
    }

#elif USEPAPPSOTREE == 1

  cout << endl << "..:: NO test TIMS TDF parsing ::.." << endl;

#endif
}
