#include "osl/move_generator/allMoves.h"
#include "osl/move_generator/capture_.h"
#include "osl/move_action/store.h"
#include "osl/record/record.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/move_classifier/classifier.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <boost/progress.hpp>
#include <iostream>
#include <fstream>

typedef osl::NumEffectState test_state_t;
using namespace osl;

class generateCaptureMovesTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( generateCaptureMovesTest );
  CPPUNIT_TEST( testMove );
  CPPUNIT_TEST( testFile );
  CPPUNIT_TEST_SUITE_END();
private:
  SimpleState state;
public:
  generateCaptureMovesTest();
  template<typename State>
  void testMove(SimpleState& state);
  void testMove();
  void testFile();
};

CPPUNIT_TEST_SUITE_REGISTRATION(generateCaptureMovesTest);

using namespace osl;
using namespace osl::move_generator;
using namespace osl::move_action;

generateCaptureMovesTest::generateCaptureMovesTest()
{
  state=CsaString(
    "P1+NY+TO * +HI *  * -OU-KE-KY\n"
    "P2 *  *  *  *  * -GI-KI *  *\n"
    "P3 *  * +GI * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU *  *  *  *  *\n"
    "P5 *  * -KE+FU+FU *  * +FU *\n"
    "P6+KE *  *  *  * -FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n"
    ).getInitialState();
}

template<typename State>
void generateCaptureMovesTest::testMove(SimpleState& state)
{
  State eState(state);

  MoveVector moves;
  {
    Store store(moves);
    GenerateCapture::generate(BLACK,eState,Square(6,4),store);
  }
  moves.unique();
  CPPUNIT_ASSERT(moves.isMember(Move(Square(6,1),Square(6,4),PROOK,PAWN,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
  // the next move is not generated bacause the rook should promote
  CPPUNIT_ASSERT(moves.isMember(Move(Square(7,3),Square(6,4),PSILVER,PAWN,true,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(7,3),Square(6,4),SILVER,PAWN,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(6,5),Square(6,4),PAWN,PAWN,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(5,3),Square(6,4),PBISHOP,PAWN,false,BLACK)));
  CPPUNIT_ASSERT(moves.size()==5);

  for(size_t i=0;i<moves.size();i++)
    CPPUNIT_ASSERT(eState.isValidMove(moves[i]));
  moves.clear();
}

void generateCaptureMovesTest::testMove()
{
  testMove<test_state_t>(state);
}

template<typename State>
static void testFile(const std::string& fileName)
{
  Record rec=CsaFile(fileName).getRecord();
  SimpleState state=rec.getInitialState();
  State eState(state);
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++)
  {
    MoveVector allMoves;
    {
      Store store(allMoves);
      AllMoves<Store>::
	generate(eState.turn(),eState,store);
    }
    // 王手がかかっているときはcaptureを呼ばない
    if(eState.hasEffectAt(alt(eState.turn()),eState.kingSquare(eState.turn())))
      continue;
    for(int y=1;y<=9;y++)
      for(int x=9;x>0;x--)
      {
	Square pos=Square(x,y);
	Piece p=eState.pieceAt(pos);
	if(! p.isEmpty() && p.owner()==alt(eState.turn()))
	{
	  MoveVector captureMoves;
	  {
	    Store storeCapture(captureMoves);
	    GenerateCapture::generate(eState.turn(),eState,pos,storeCapture);
	  }
	  captureMoves.unique();
	  for(size_t j=0;j<captureMoves.size();j++)
	  {
	    CPPUNIT_ASSERT( (eState.isValidMove(captureMoves[j],true)  &&
			     captureMoves[j].to()==pos ) ||
			    (std::cerr << eState << captureMoves[j] << std::endl,0)
	      );
	    CPPUNIT_ASSERT( !captureMoves[j].ignoreUnpromote());
	    CPPUNIT_ASSERT( Classifier::isSafeMove(eState,captureMoves[j]));
	  }
	  for(size_t j=0;j<allMoves.size();j++)
	  {
	    if(allMoves[j].to()==pos)
	    {
	      if(!Classifier::isSafeMove(eState,allMoves[j])) continue;
	      CPPUNIT_ASSERT(captureMoves.isMember(allMoves[j])||
			     (std::cerr << eState << pos << std::endl << allMoves[j] << std::endl << captureMoves << std::endl,0));
	    }
	  }
	}
      }
    Move move=moves[i];
    eState.makeMove(move);
  }
}

void generateCaptureMovesTest::testFile()
{
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=1000;
  if (OslConfig::inUnitTestShort()) 
    count=10;
  std::string filename;
  boost::scoped_ptr<boost::progress_display> progress;
  if (OslConfig::inUnitTestLong())
    progress.reset(new boost::progress_display(count, std::cerr));
  while((ifs >> filename) && filename != "" && ++i<count) {
    if (progress)
      ++(*progress);
    ::testFile<test_state_t>(OslConfig::testCsaFile(filename));
  }
}


// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
