#include "osl/repetitionCounter.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/move_classifier/check_.h"
#include "osl/move_classifier/moveAdaptor.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"
#include <iostream>
#include <fstream>
#include <iomanip>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class RepetitionCounterTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(RepetitionCounterTest);
  CPPUNIT_TEST(testCsaFiles);
  CPPUNIT_TEST(testEmpty);
  CPPUNIT_TEST(testDraw);
  CPPUNIT_TEST(testLose);
  CPPUNIT_TEST(testLose2);
  CPPUNIT_TEST(testClone);
  CPPUNIT_TEST_SUITE_END();
 public:
  void testCsaFiles();
  void testEmpty();
  void testDraw();
  void testLose();
  void testLose2();
  void testClone();
  void testFile(const std::string& filename);
};

using namespace osl;
using namespace osl::record;
extern bool isShortTest;

CPPUNIT_TEST_SUITE_REGISTRATION(RepetitionCounterTest);

void RepetitionCounterTest::testClone()
{
  using namespace osl::move_classifier;
  const char *stateString = 
    "P1-KY *  *  *  *  *  * +NY * \n"
    "P2 * -OU-KI-KI *  *  *  * +RY\n"
    "P3 * -GI-KE+KI *  *  *  * +HI\n"
    "P4 *  * -FU-KY-FU *  * -FU * \n"
    "P5-FU-FU * -KE * -FU *  *  * \n"
    "P6 *  * +FU-FU+FU * -FU *  * \n"
    "P7+FU+FU *  *  *  *  *  *  * \n"
    "P8+KY+GI+GI-UM *  *  *  *  * \n"
    "P9+OU+KE *  *  *  *  * +KE * \n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00GI\n"
    "P+00KI\n"
    "P-00KA\n"
    "+\n";
  NumEffectState state((CsaString(stateString).getInitialState()));
  const Move m = Move(Position(7,1),GOLD,BLACK);

  RepetitionCounter counter;
  counter.push(state, m);

  ApplyMoveOfTurn::doMove(state, m);

  const Move m2 = Move(Position(7,9),SILVER,WHITE);
  RepetitionCounter copy(counter);
  CPPUNIT_ASSERT(copy.isConsistent());
  copy.push(state,m2);
  CPPUNIT_ASSERT(copy.isConsistent());

  copy.pop();
  CPPUNIT_ASSERT(copy.isConsistent());
}

void RepetitionCounterTest::testEmpty()
{
  using namespace osl::move_classifier;
  const char *stateString = 
    "P1-KY *  *  *  *  *  * +NY * \n"
    "P2 * -OU-KI-KI *  *  *  * +RY\n"
    "P3 * -GI-KE+KI *  *  *  * +HI\n"
    "P4 *  * -FU-KY-FU *  * -FU * \n"
    "P5-FU-FU * -KE * -FU *  *  * \n"
    "P6 *  * +FU-FU+FU * -FU *  * \n"
    "P7+FU+FU *  *  *  *  *  *  * \n"
    "P8+KY+GI+GI-UM *  *  *  *  * \n"
    "P9+OU+KE *  *  *  *  * +KE * \n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00FU\n"
    "P-00GI\n"
    "P+00KI\n"
    "P-00KA\n"
    "+\n";
  NumEffectState state((CsaString(stateString).getInitialState()));
  const Move m = Move(Position(7,1),GOLD,BLACK);
  const HashKey new_key = HashKey(state).newHashWithMove(m);

  RepetitionCounter counter;
  CPPUNIT_ASSERT_EQUAL(-1, counter.getFirstMove(new_key));
  CPPUNIT_ASSERT_EQUAL(-1, counter.getLastMove(new_key));
  CPPUNIT_ASSERT_EQUAL(0u, counter.countRepetition(new_key));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), counter.isSennichite(state, m));
  CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(BLACK));
  CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(WHITE));

  counter.push(state, m);

  CPPUNIT_ASSERT_EQUAL(0, counter.getFirstMove(new_key));
  CPPUNIT_ASSERT_EQUAL(0, counter.getLastMove(new_key));
  CPPUNIT_ASSERT_EQUAL(1u, counter.countRepetition(new_key));
  CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), counter.isSennichite(state, m));
  CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(BLACK));
  CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(WHITE));
}

void RepetitionCounterTest::testDraw()
{
  const char *test_record_str = 
    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
    "P2 * -HI *  *  *  *  * -KA * \n"
    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
    "P8 * +KA *  *  *  *  * +HI * \n"
    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
    "+\n"
    "+3948GI\n"
    "-7162GI\n"
    "+4839GI\n"
    "-6271GI\n"
    "+3948GI\n"
    "-7162GI\n"
    "+4839GI\n"
    "-6271GI\n"
    "+3948GI\n"
    "-7162GI\n"
    "+4839GI\n"
    "-6271GI\n";
  CsaString test_record(test_record_str);
  
  NumEffectState state(test_record.getInitialState());
  const vector<Move>& moves = test_record.getRecord().getMoves();

  HashKey key(state);
  RepetitionCounter counter(state);
  for (int i=0; i<(int)moves.size(); ++i)
  {
    key = HashKey(state);
    CPPUNIT_ASSERT_EQUAL(i % 4, counter.getFirstMove(key));
    CPPUNIT_ASSERT_EQUAL(i, counter.getLastMove(key));
    CPPUNIT_ASSERT_EQUAL((i / 4u)+1, counter.countRepetition(key));
    CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(BLACK));
    CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(WHITE));
    const Move m = moves[i];
    if (i <(int)moves.size() -1)
      CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(),
			   counter.isSennichite(state, m));
    else
      CPPUNIT_ASSERT_EQUAL(Sennichite::DRAW(), counter.isSennichite(state, m));
    counter.push(state, m);
    ApplyMoveOfTurn::doMove(state, m);
  }
}

void RepetitionCounterTest::testLose()
{
  const char *test_record_str = 
    "P1 *  *  *  *  * +UM *  * -KY\n"
    "P2 *  *  *  *  *  *  * -KE-OU\n"
    "P3 *  *  *  *  *  * +FU * -FU\n"
    "P4 *  *  *  *  *  *  * +FU * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9+OU *  *  *  *  *  *  *  * \n"
    "P-00AL\n"
    "+\n"
    "+4123UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n"
    "+3223UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n"
    "+3223UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n"
    "+3223UM\n";
  CsaString test_record(test_record_str);
  
  NumEffectState state(test_record.getInitialState());
  const vector<Move>& moves = test_record.getRecord().getMoves();

  HashKey key = HashKey(state);
  RepetitionCounter counter(state);
  for (int i=0; i<(int)moves.size(); ++i)
  {
    key = HashKey(state);
    CPPUNIT_ASSERT_EQUAL(i, counter.getLastMove(key));
    CPPUNIT_ASSERT_EQUAL((i+1)/2, counter.checkCount(BLACK));
    CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(WHITE));
    const Move m = moves[i];
    if (i <(int)moves.size() -1)
      CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), counter.isSennichite(state, m));
    else
      CPPUNIT_ASSERT_EQUAL(Sennichite::BLACK_LOSE(),
			   counter.isSennichite(state, m));
    counter.push(state, m);
    ApplyMoveOfTurn::doMove(state, m);
  }
}

void RepetitionCounterTest::testLose2()
{
  const char *test_record_str = 
    "P1 *  *  *  *  *  *  *  * -KY\n"
    "P2 *  *  *  *  *  * +UM-KE-OU\n"
    "P3 *  *  *  *  *  * +FU * -FU\n"
    "P4 *  *  *  *  *  *  * +FU * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9+OU *  *  *  *  *  *  *  * \n"
    "P-00AL\n"
    "+\n"
    "+3223UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n"
    "+3223UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n"
    "+3223UM\n"
    "-1221OU\n"
    "+2332UM\n"
    "-2112OU\n";
  CsaString test_record(test_record_str);
  
  NumEffectState state(test_record.getInitialState());
  const vector<Move>& moves = test_record.getRecord().getMoves();

  HashKey key = HashKey(state);
  RepetitionCounter counter(state);
  for (int i=0; i<(int)moves.size(); ++i)
  {
    key = HashKey(state);
    CPPUNIT_ASSERT_EQUAL(i, counter.getLastMove(key));
    CPPUNIT_ASSERT_EQUAL((i+1)/2, counter.checkCount(BLACK));
    CPPUNIT_ASSERT_EQUAL(0, counter.checkCount(WHITE));
    const Move m = moves[i];
    if (i <(int)moves.size() -1)
      CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), counter.isSennichite(state, m));
    else
      CPPUNIT_ASSERT_EQUAL(Sennichite::BLACK_LOSE(),
			   counter.isSennichite(state, m));
    counter.push(state, m);
    ApplyMoveOfTurn::doMove(state, m);
  }
}

void RepetitionCounterTest::testFile(const std::string& filename)
{
  typedef RepetitionCounter::list_t list_t;

  Record rec=CsaFile(filename).getRecord();
  NumEffectState state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  vector<SimpleState> states;
  states.push_back(state);
  RepetitionCounter counter(state);
  for (size_t i=0;i<moves.size();i++)
  {
    const Move move=moves[i];
    const HashKey key = HashKey(state).newHashWithMove(move);
    const Sennichite sennichite = counter.isSennichite(state, move);
    if (counter.countRepetition(key) < 3)
    {
      CPPUNIT_ASSERT_EQUAL(Sennichite::NORMAL(), sennichite);
    }

    counter.push(state, move);
    CPPUNIT_ASSERT(counter.isConsistent());
    const RepetitionCounter copy(counter);
    const HashKey new_key = HashKey(state).newHashWithMove(move);
    counter.pop();
    counter.push(state, move);
    CPPUNIT_ASSERT(RepetitionCounter::maybeEqual(copy,counter));
    ApplyMoveOfTurn::doMove(state, move);
    states.push_back(state);
    const int repeated = counter.countRepetition(key);
    if (repeated > 1)
    {
      const list_t& l = counter.getRepetitions(key);
      assert(l.size() > 1);
      list_t::const_iterator p=l.begin();
      const SimpleState& s0 = states.at(*p);
      ++p;
      for (; p!= l.end(); ++p)
      {
	CPPUNIT_ASSERT(s0 == states.at(*p));
      }
      if (repeated >= 4)
      {
	if (sennichite.isDraw())
	{
	  if (! isShortTest)
	    std::cerr << "DRAW\n";
	}
	else
	{
	  std::cerr << "王手がらみの4回目\n";
	  counter.printMatches(key);
	}
	CPPUNIT_ASSERT(! sennichite.isNormal());
      }
    }
  }
}

void RepetitionCounterTest::testCsaFiles()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=1000;
  if (isShortTest)
      count=10;
  std::string filename;
  while((ifs >> filename) && (++i<count)) {
    if(filename == "") 
	break;
    if (! isShortTest)
      std::cerr << filename << " ";
    testFile(OslConfig::testCsaFile(filename));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
