#include "osl/effect/liberty8.h"
#include "osl/state/simpleState.h"
#include "osl/state/numEffectState.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <fstream>
#include <iostream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class liberty8Test : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE( liberty8Test );
  CPPUNIT_TEST( testShortMaskTable );
  CPPUNIT_TEST( testLongEffectTable );
  CPPUNIT_TEST( testBasic );
  CPPUNIT_TEST( testCsaFiles );
  CPPUNIT_TEST_SUITE_END();
 public:
  void testShortMaskTable();
  void testLongEffectTable();
  void testBasic();
  void testCsaFiles();
};

using namespace osl;
using namespace osl::record;
using namespace osl::effect;
extern bool isShortTest;

CPPUNIT_TEST_SUITE_REGISTRATION(liberty8Test);

void liberty8Test::testShortMaskTable(){
  {
    NearMask nearMask=
      Liberty8_Table.getShortMask<BLACK>(PAWN,Position(5,7),
					Position(5,5));
    CPPUNIT_ASSERT(NearMask::makeDirect(0xff & ~(DirectionTraits<D>::mask))
		   == nearMask);
  }
  {
    NearMask nearMask=
      Liberty8_Table.getShortMask<BLACK>(GOLD,Position(5,7),
					Position(5,5));
    CPPUNIT_ASSERT(NearMask::makeDirect(0xff 
					& ~(DirectionTraits<D>::mask|
					    DirectionTraits<DL>::mask|
					    DirectionTraits<DR>::mask))
		   == nearMask);

  }
  {
    NearMask nearMask=
      Liberty8_Table.getShortMask<WHITE>(PROOK,Position(4,3),
					Position(5,5));
    CPPUNIT_ASSERT(NearMask::makeDirect(0xff & ~(DirectionTraits<D>::mask))
		   == nearMask);
    // DLは伸ばせるので除外する
  }
}

void liberty8Test::testLongEffectTable(){
  {
    LongEffect8 longEffect=
      Liberty8_Table.getLongEffect<BLACK>(ROOK,Position(4,8),
					 Position(5,5));
    CPPUNIT_ASSERT_EQUAL(Offset(-1,1),
			 longEffect.getOffset());
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<DR>::mask,
			 longEffect.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<R>::mask,
			 longEffect.getMask(1));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<UR>::mask,
			 longEffect.getMask(2));
  }
  {
    LongEffect8 longEffect=
      Liberty8_Table.getLongEffect<WHITE>(PBISHOP,Position(1,1),
					 Position(5,5));
    CPPUNIT_ASSERT_EQUAL(Offset(1,1),
			 longEffect.getOffset());
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<DL>::mask,
			 longEffect.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<UR>::mask,
			 longEffect.getMask(1));
    CPPUNIT_ASSERT_EQUAL(0u,
			 longEffect.getMask(2));
  }
  {
    LongEffect8 longEffect=
      Liberty8_Table.getLongEffect<WHITE>(LANCE,Position(1,6),
					 Position(1,8));
    // Dはdirect
    CPPUNIT_ASSERT_EQUAL(Offset(0,1),
			 longEffect.getOffset());
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<D>::mask,
			 longEffect.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<U>::mask,
			 longEffect.getMask(1));
    CPPUNIT_ASSERT_EQUAL(0u,
			 longEffect.getMask(2));
  }
  {
    LongEffect8 longEffect=
      Liberty8_Table.getLongEffect<BLACK>(PROOK,Position(3,1),
					 Position(6,2));
    // Dはdirect
    CPPUNIT_ASSERT_EQUAL(Offset(-1,-1),
			 longEffect.getOffset());
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<UR>::mask,
			 longEffect.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<U>::mask,
			 longEffect.getMask(1));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<UL>::mask,
			 longEffect.getMask(2));
  }
  {
    LongEffect8 longEffect=
      Liberty8_Table.getLongEffect<BLACK>(ROOK,Position(4,4),
					 Position(5,5));
    CPPUNIT_ASSERT_EQUAL(Offset(-1,0),
			 longEffect.getOffset());
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<R>::mask,
			 longEffect.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<DR>::mask,
			 longEffect.getMask(1));
    CPPUNIT_ASSERT_EQUAL(0u,
			 longEffect.getMask(2));
    LongEffect8 longEffect2=
      Liberty8_Table.getLongEffect2<BLACK>(Position(4,4),
					  Position(5,5));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<U>::mask,
			 longEffect2.getMask(0));
    CPPUNIT_ASSERT_EQUAL(0xff & DirectionTraits<UL>::mask,
			 longEffect2.getMask(1));
  }
}

void liberty8Test::testBasic(){
  {
    NumEffectState state(CsaString(
"P1-KY-KE *  *  *  *  *  *  * \n"
"P2 *  *  *  *  *  * -KI-OU * \n"
"P3 *  * -FU *  * -KI *  *  * \n"
"P4-FU-HI *  *  *  *  * +GI-KY\n"
"P5 * -FU+FU * -FU-FU-FU-FU+FU\n"
"P6+FU *  *  *  *  * +KE * -KY\n"
"P7 * +FU * +FU * +FU+FU+FU * \n"
"P8+KY *  *  *  * +KI+KI * +OU\n"
"P9 * +KE-UM * +KA *  * +KE * \n"
"P-00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00GI\n"
"P-00GI\n"
"P-00GI\n"
"P-00HI\n"
"+\n"
).getInitialState());
    Liberty8<WHITE> liberty8White(state,state.getKingPosition<WHITE>());
    CPPUNIT_ASSERT(NearMask::makeDirect(DirectionTraits<UR>::mask
					| DirectionTraits<U>::mask
					| DirectionTraits<UL>::mask
					| DirectionTraits<R>::mask)
		   == liberty8White.getMask());
    Liberty8<BLACK> liberty8Black(state,state.getKingPosition<BLACK>());
    CPPUNIT_ASSERT(NearMask::makeDirect(DirectionTraits<R>::mask)
		   == liberty8Black.getMask());
  }
  {
    NumEffectState state(CsaString(
"P1-KY-KE+GI+KA *  * +RY * -KY\n"
"P2 *  * -OU * -KI * +NK *  * \n"
"P3-FU * -GI-FU-FU-FU *  * -FU\n"
"P4 *  * -FU *  *  *  *  *  * \n"
"P5 *  *  *  * +KA *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 *  * -NK * +OU *  *  *  * \n"
"P9+KY+KE * -HI * +KI+GI * +KY\n"
"P+00FU\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00GI\n"
"P-00KI\n"
"P-00KI\n"
"-\n"
).getInitialState());
    Liberty8<WHITE> liberty8White(state,state.getKingPosition<WHITE>());
    CPPUNIT_ASSERT(liberty8White.getMask()==
		   NearMask::makeDirect(DirectionTraits<U>::mask));
  }
  {
    NumEffectState state(CsaString(
"P1-KY-KE *  *  *  * +RY * -KY\n"
"P2 *  *  * -OU-KI * +NK *  * \n"
"P3-FU * -GI-FU-FU-FU *  * -FU\n"
"P4 *  * -FU *  *  *  *  *  * \n"
"P5 *  *  *  * +KA *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 *  * -NK * +OU *  *  *  * \n"
"P9+KY+KE * -HI * +KI+GI * +KY\n"
"P+00FU\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00GI\n"
"P-00GI\n"
"P-00KI\n"
"P-00KI\n"
"P+00KA\n"
"+\n"
).getInitialState());
    Liberty8<WHITE> liberty8White(state,state.getKingPosition<WHITE>());
    CPPUNIT_ASSERT(liberty8White.getMask()==
		   NearMask::makeDirect(DirectionTraits<L>::mask));
  }
  {
    // 竜が2方向に長い利きを持つ珍しい例
    NumEffectState state(CsaString(
"P1+RY *  *  *  *  * -KA-KE-KY\n"
"P2 *  *  *  * +TO * -KI * -OU\n"
"P3 *  * -KE *  * -KI-GI-GI * \n"
"P4-FU *  *  * -FU-FU-FU-FU-FU\n"
"P5 * -FU+FU *  *  *  * +FU * \n"
"P6+FU+FU * +GI * +FU+FU * +FU\n"
"P7 *  *  *  *  * +KI+KE+GI+OU\n"
"P8+KY-TO *  *  *  *  * -RY * \n"
"P9 * +KE *  *  *  *  * -KI+KY\n"
"P+00FU\n"
"P-00FU\n"
"P+00KY\n"
"P+00KA\n"
"+\n"
).getInitialState());
    Liberty8<BLACK> liberty8White(state,state.getKingPosition<BLACK>());
    CPPUNIT_ASSERT(liberty8White.getMask()==
		   NearMask::makeDirect(DirectionTraits<DR>::mask));
  }
}

template <Player P>
static void checkKingLiberty8(NumEffectState const& state, const NumEffectState& state2)
{
  Position kingPosition=state.template getKingPosition<P>();
  Liberty8<P> liberty(state,kingPosition);
  Liberty8<P> liberty2(state2,kingPosition);
  CPPUNIT_ASSERT_EQUAL(liberty.getMask(), liberty2.getMask());
  for(int i=0;i<8;i++){
    Direction dir=static_cast<Direction>(i);
    Position to=kingPosition-Board_Table.getOffset<P>(dir);
    if (state.getPieceAt(to).template canMoveOn<P>()
	&& (!state.template 
	    hasEffectByWithRemove<PlayerTraits<P>::opponent>(to,kingPosition)))
    {
      CPPUNIT_ASSERT(liberty.getMask().isSet(i));
      CPPUNIT_ASSERT(liberty2.getMask().isSet(i));
    }
    else
    {
      CPPUNIT_ASSERT(! (liberty.getMask().isSet(i)));
      CPPUNIT_ASSERT(! (liberty.getMask().isSet(i)));
    }
  }
}


static void testFile(const std::string& fileName)
{
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    NumEffectState state1(state);
    NumEffectState state2(state);
    checkKingLiberty8<BLACK>(state1, state2);
    checkKingLiberty8<WHITE>(state1, state2);
    Move move=moves[i];
    ApplyMoveOfTurn::doMove(state, move);
  }
}

void liberty8Test::testCsaFiles(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (isShortTest)
      count=10;
  std::string fileName;
  while((ifs >> fileName) && (++i<count)) {
    if(fileName == "") 
	break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testFile(OslConfig::testCsaFile(fileName));
  }
}
/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
