#include "../config.h"
#include <sys/types.h>
#include <stdlib.h>
#include "libnjb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "unicode.h"
#include "protocol3.h"
#include "byteorder.h"

extern int __sub_depth;
extern int njb_unicode_flag;

datafile_t *datafile_new (void)
{
	__dsub= "datafile_new";
	datafile_t *df;

	__enter;

	df= (datafile_t *) malloc(sizeof(datafile_t));
	if ( df == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	memset(df, 0, sizeof(datafile_t));

	__leave;
	return df;
}

void datafile_destroy (datafile_t *df)
{
	__dsub= "datafile_destroy";

	__enter;
	if ( df->filename != NULL ) free(df->filename);
	free(df);
	__leave;
}

datafile_t *datafile_unpack (unsigned char *data, size_t nbytes)
{
	__dsub= "datafile_unpack";
	u_int16_t lname;
	unsigned char *dp= (unsigned char *) data;
	datafile_t *df;

	__enter;

	df= datafile_new();
	if ( df == NULL ) {
		__leave;
		return NULL;
	}

	/* Add metadata in correct byte order */
	
	df->msdw = njb1_bytes_to_32bit(&data[0]);
	df->lsdw = njb1_bytes_to_32bit(&data[4]);
	lname = njb1_bytes_to_16bit(&data[8]);

	if ( (lname + 10) > nbytes ) {
		datafile_destroy(df);
		NJB_ERROR(EO_BADDATA);
		__leave;
		return NULL;
	}

	/* Handle unicode conversion of filename, or
	 * default to ISO 8859-1 */
	df->filename= (char *) malloc(lname+1);
	memcpy(df->filename, &dp[10], lname);
	df->filename[lname] = '\0';
	if ( df->filename == NULL ) {
	  datafile_destroy(df);
	  NJB_ERROR(EO_NOMEM);
	  __leave;
	  return NULL;
	}

	if (njb_unicode_flag == NJB_UC_UTF8) {
	  char *utf8str = NULL;
	  
	  utf8str= strtoutf8(df->filename);
	  if (utf8str == NULL) {
	    datafile_destroy(df);
	    NJB_ERROR(EO_NOMEM);
	    __leave;
	    return NULL;
	  }
	  free(df->filename);
	  df->filename=utf8str;
	}

	__leave;
	return df;
}

void datafile_dump (datafile_t *df, FILE *fp)
{
	__dsub= "datafile_dump";
	u_int64_t size= datafile_size(df);

	__enter;

	fprintf(fp, "File ID :  %u\n", df->dfid);
	fprintf(fp, "Filename : %s\n", df->filename);
	fprintf(fp, "Size :     %Lu bytes\n", size);

	__leave;
}

u_int64_t datafile_size (datafile_t *df)
{
	__dsub= "datafile_size";
	u_int64_t val;

	__enter;

	val = make64(df->msdw, df->lsdw);

	__leave;
	return val;
}

void datafile_set_size (datafile_t *df, u_int64_t size)
{
	__dsub= "datafile_set_size";

	__enter;
	split64(size, &df->msdw, &df->lsdw);
	__leave;
}

void datafile_set_time (datafile_t *df, time_t ts)
{
	__dsub= "datafile_set_time";

	__enter;
	df->timestamp = (u_int32_t) ts;
	__leave;
}

unsigned char *datafile_pack (datafile_t *df, u_int32_t *size)
{
	__dsub= "datafile_pack";
	unsigned char *ptag;
	char *filename = NULL;
	u_int16_t len;

	__enter;


	/* Convert filename to ISO 8859-1 as is used on NJB 1 */
	if (njb_unicode_flag == NJB_UC_UTF8) {
	  filename = utf8tostr(df->filename);
	} else {
	  filename = strdup(df->filename);
	}
	if (filename == NULL) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	len= (u_int16_t) strlen(filename) + 1;
	*size= len + 10;

	ptag= (unsigned char *) malloc(*size);
	if ( ptag == NULL ) {
		free(filename);
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	/* Pack tag with correct byte order */
	from_32bit_to_njb1_bytes(df->msdw, &ptag[0]);
	from_32bit_to_njb1_bytes(df->lsdw, &ptag[4]);
	from_16bit_to_njb1_bytes(len, &ptag[8]);
	memcpy(&ptag[10], filename, len);
	
	free(filename);

	__leave;
	return ptag;
}

int datafile_set_name (datafile_t *df, const char *filename)
{
	__dsub= "datafile_set_name";

	__enter;

	df->filename= strdup(filename);
	if ( df->filename == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return -1;
	}

	__leave;
	return 0;
}


static void add_bin_unistr(unsigned char *data, u_int32_t *datap, u_int32_t tagtype, unsigned char *unistr)
{
  u_int32_t binlen;

  binlen = ucs2strlen(unistr) * 2 + 2;
  from_16bit_to_njb3_bytes(binlen+2, &data[*datap]);
  *datap += 2;
  from_16bit_to_njb3_bytes(tagtype, &data[*datap]);
  *datap += 2;
  memcpy(data+(*datap), unistr, binlen);
  *datap += binlen;
}

/* Same thing but for NJB2, NJB3, ZEN and ZEN USB 2.0 */
unsigned char *datafile_pack3 (njb_t *njb, datafile_t *df, u_int32_t *size)
{
	__dsub= "datafile_pack3";
	unsigned char *ptag;
	unsigned char *filename = NULL;
	u_int32_t p = 0;
	u_int16_t len;

	__enter;

	/* Create a filename */
	filename = strtoucs2(df->filename);
	if (filename == NULL) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	len= (u_int16_t) ucs2strlen(filename)*2;
	if ( njb->device_type == NJB_DEVICE_NJB3 
	     || njb->device_type == NJB_DEVICE_NJBZEN ) {
	  *size= len + 38; /* Misc metadata magic size 38 */
	} else {
	  *size= len + 40; /* Misc metadata magic size 40 */
	}

	ptag= (unsigned char *) malloc(*size);
	if ( ptag == NULL ) {
		free(filename);
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	/* Add filename tag */
	add_bin_unistr(ptag, &p, NJB3_FNAME_FRAME_ID, filename);
	free(filename);
	/* Add a root directory indicator ... */
	from_16bit_to_njb3_bytes(0x0006, &ptag[p]);
	p += 2;
	from_16bit_to_njb3_bytes(NJB3_DIR_FRAME_ID, &ptag[p]);
	p += 2;
	from_16bit_to_njb3_bytes(0x005C, &ptag[p]); /* Backslash */
	p += 2;
	from_16bit_to_njb3_bytes(0x0000, &ptag[p]);
	p += 2;
	/* Add filesize in 32 bits */
	from_16bit_to_njb3_bytes(0x0006, &ptag[p]);
	p += 2;
	from_16bit_to_njb3_bytes(NJB3_FILESIZE_FRAME_ID, &ptag[p]);
	p += 2;
	/* NJB3 doesn't seem to support
	 * filesizes greater than 32 bits (?) 
	 * so df->msdw is thrown away! */
	from_32bit_to_njb3_bytes(df->lsdw, &ptag[p]);
	p += 4;
	/* Add timestamp. */
	from_16bit_to_njb3_bytes(0x0006, &ptag[p]);
	p += 2;
	from_16bit_to_njb3_bytes(NJB3_FILETIME_FRAME_ID, &ptag[p]);
	p += 2;
	from_32bit_to_njb3_bytes(df->timestamp, &ptag[p]);
	p += 4;
	if ( njb->device_type == NJB_DEVICE_NJB3 
	     || njb->device_type == NJB_DEVICE_NJBZEN 
	     ) {
	  /* This is the locked status used in older firmware. The file is not locked. */
	  from_16bit_to_njb3_bytes(0x0004, &ptag[p]);
	  p += 2;
	  from_16bit_to_njb3_bytes(NJB3_LOCKED_FRAME_ID, &ptag[p]);
	  p += 2;
	  from_16bit_to_njb3_bytes(0x0000, &ptag[p]); /* = not locked */
	  p += 2;
	  from_16bit_to_njb3_bytes(0x0000, &ptag[p]); /* Terminator */
	  p += 2;
	} else {
	  /* Unknown ending - used in NJB2 and NJB Zen USB 2.0
	   * Probably locking files was deemed unnecessary,
	   * so that was removed. Instead this strange thing was introduced. */
	  from_16bit_to_njb3_bytes(0x0006, &ptag[p]);
	  p += 2;
	  from_16bit_to_njb3_bytes(NJB3_UNKNOWN8_FRAME_ID, &ptag[p]); /* Could be readable/writeable etc... */
	  p += 2;
	  from_16bit_to_njb3_bytes(0x2000, &ptag[p]); /* Frame contents */
	  p += 2;
	  from_16bit_to_njb3_bytes(0x0000, &ptag[p]); /* Frame contents */
	  p += 2;
	  from_16bit_to_njb3_bytes(0x0000, &ptag[p]); /* Terminator */
	  p += 2;
	}

	__leave;
	return ptag;
}
