/* SSLHMac.java -- SSLv3's MAC algorithm.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.util.Arrays;
import java.util.Map;

import gnu.crypto.hash.HashFactory;
import gnu.crypto.hash.IMessageDigest;
import gnu.crypto.mac.IMac;

/**
 * The MAC function in SSLv3. This mac is defined as:
 *
 * <pre>
 * hash(MAC_write_secret, pad_2 +
 *      hash(MAC_write_secret + pad_1 + data));</pre>
 *
 * <p><tt>hash</tt> is e.g. MD5 or SHA-1, <tt>pad_1</tt> is the value
 * 0x36 48 times for MD5 and 40 times for SHA-1, and <tt>pad_2</tt> is
 * the value 0x5c repeated similarly.
 */
class SSLHMac implements IMac, Cloneable
{

  // Fields.
  // -------------------------------------------------------------------------

  static final byte PAD1 = 0x36;
  static final byte PAD2 = 0x5c;

  protected IMessageDigest md;
  protected byte[] key;
  protected final byte[] pad1, pad2;

  // Constructors.
  // -------------------------------------------------------------------------

  SSLHMac(String mdName)
  {
    super();
    this.md = HashFactory.getInstance(mdName);
    if (mdName.equalsIgnoreCase("MD5"))
      {
        pad1 = new byte[48];
        pad2 = new byte[48];
      }
    else
      {
        pad1 = new byte[40];
        pad2 = new byte[40];
      }
    Arrays.fill(pad1, PAD1);
    Arrays.fill(pad2, PAD2);
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public Object clone()
  {
    try
      {
        return super.clone();
      }
    catch (CloneNotSupportedException cnse)
      {
        throw new Error();
      }
  }

  public String name()
  {
    return "SSLHMac-" + md.name();
  }

  public int macSize()
  {
    return md.hashSize();
  }

  public void init(Map attributes)
  {
    key = (byte[]) attributes.get(MAC_KEY_MATERIAL);
    if (key == null)
      throw new NullPointerException();
    reset();
  }

  public void reset()
  {
    md.reset();
    md.update(key, 0, key.length);
    md.update(pad1, 0, pad1.length);
  }

  public byte[] digest()
  {
    byte[] h1 = md.digest();
    md.update(key, 0, key.length);
    md.update(pad2, 0, pad2.length);
    md.update(h1, 0, h1.length);
    byte[] result = md.digest();
    reset();
    return result;
  }

  public void update(byte b)
  {
    md.update(b);
  }

  public void update(byte[] buf, int off, int len)
  {
    md.update(buf, off, len);
  }

  public boolean selfTest()
  {
    return true; // XXX
  }
}
