
/* Command line oriented batch and dialog tool which creates, loads,
   manipulates and burns ISO 9660 filesystem images.

   Copyright 2007-2008 Thomas Schmitt, <scdbackup@gmx.net>

   Provided under GPL version 2.

   This file contains inner declarations of xorriso.
   The public interface is in xorriso.h
*/


/* For now, #ifdef Xorriso_is_xorriso_selF  has no meaning.
   But it is already now to be set only by the xorriso.c module.
*/

#ifndef Xorriso_private_includeD
#define Xorriso_private_includeD yes

#define Xorriso_program_versioN "0.2.2"

/** The source code release timestamp */
#include "xorriso_timestamp.h"
#ifndef Xorriso_timestamP
#define Xorriso_timestamP "-none-given-"
#endif

/** The binary build timestamp is to be set externally by the compiler
    or by a macro definition in xorriso_buildstamp.h.
*/
#include "xorriso_buildstamp.h"
#ifndef Xorriso_build_timestamP
#define Xorriso_build_timestamP "-none-given-"
#endif


/* Because regex_t is mentioned in struct XorrisO */
#ifdef Xorriso_with_regeX
#include <regex.h>
#endif /* Xorriso_with_regeX */


#define Smem_malloC malloc
#define Smem_freE free
#define SfileadrL 4096

/* <<< ??? */
typedef int (*Cleanup_app_handler_T)();

struct LinkiteM;          /* Trace of hops during symbolic link resolution */
struct ExclusionS;        /* List of -not_* conditions */
struct PermiteM;          /* Stack of temporarily altered access permissions */


/* maximum number of history lines to be reported with -status:long_history */
#define Xorriso_status_history_maX 100

/* <<< ??? */
/* try to catch signals and ignore them during abort handling */
#define Xorriso_abort_handler_defaulT (1|(2<<4))


/** The list of startup file names */
#define Xorriso_rc_nuM 4


struct XorrisO { /* the global context of xorriso */

 int libs_are_started;

 /* source */
 char progname[SfileadrL];
 char initial_wdx[SfileadrL];
 int no_rc;

 /* Command line argument emulations:
      0=xorriso mode
      1=mkisofs mode
      2=cdrecord mode
 */
 int argument_emulation;

 /** List of startupfiles */
 char rc_filenames[Xorriso_rc_nuM][SfileadrL];
 int rc_filename_count;

 char wdi[SfileadrL];
 char wdx[SfileadrL];
 int did_something_useful;

 int add_plainly;
 off_t split_size;

 /* >>> put libisofs aspects here <<< */

 int do_joliet;
 int do_follow_pattern;
 int do_follow_param;
 int do_follow_links;
 int follow_link_limit;
 int do_follow_mount;
 int do_global_uid;
 uid_t global_uid;
 int do_global_gid;
 gid_t global_gid;
 int do_global_mode;
 mode_t global_dir_mode;
 mode_t global_file_mode;

 int do_overwrite; /* 0=off, 1=on, 2=nondir */
 int do_reassure;  /* 0=off, 1=on, 2=tree */

 char volid[33];
 int volid_default;
 char loaded_volid[33];

 char publisher[129];

 char session_logfile[SfileadrL];
 int session_lba;
 int session_blocks;

 /* >>> put libburn/isoburn aspects here */

 int toc_emulation_flag; /* bit0= bit3 for isoburn_drive_aquire()
                                  scan -ROM profiles for ISO sessions
                         */

 int image_start_mode;  /* From what address to load the ISO image
                           bit0-15= addressing mode
                            0= automatic lba as deduced from media
                            1= value is session number
                            2= value is track number
                            3= value is lba
                           bit16= with mode 3 : value is possibly 16 too high.
                                  Let isoburn_set_msc1() adjust it.
                           bit30= interference with normal msc1 processing
                                  is enabled. Without this bit,
                                  isoburn_set_msc1() will not be called.
                           bit31= image loading has happened,
                                  setting is kept for rollback only.
                                  Always apply as 0=auto.
                        */
 char image_start_value[81]; /* value according image_start_mode */

 char indev[SfileadrL];
 void *in_drive_handle;  /* interpreted only by xorrisoburn.c */
 void *in_volset_handle; /* interpreted only by xorrisoburn.c */

 int volset_change_pending; /* whether -commit would make sense */
 int no_volset_present;     /* set to 1 on first failure */

 char outdev[SfileadrL];
 void *out_drive_handle; /* interpreted only by xorrisoburn.c */
 int dev_fd_1; /* The fd which substitutes for /dev/fd/1 and is
                     connected to externaly perveived stdout.
                  */

 int grow_blindly_msc2; /* if >= 0 this causes growing from drive to drive.
                           The value is used as block address offset for
                           image generation. Like in: mkisofs -C msc1,msc2
                         */

 int ban_stdio_write;
 int do_dummy;
 int do_close;
 int speed;     /* in libburn units : 1000 bytes/second , 0 = Max, -1 = Min */
 int fs;        /* fifo size in 2048 byte chunks : at most 1 GB */
 int padding;   /* number of bytes to add after ISO 9660 image */
 int do_stream_recording;

 int keep_boot_image;
 int patch_isolinux_image;


 /* XORRISO options */
 int allow_graft_points;

 int allow_restore; /* 0= disallowed, 1=allowed, 2=device files allowed */
 int do_concat_split;  /* 1= restore complete split file directories as
                             regular files
                        */
 int do_auto_chmod;    /* 1= eventually temporarily open access permissions
                             of self-owned directories during restore
                        */

 int dialog;


 /* Pattern matching facility. It still carries legacy from scdbackup/askme.c
    but is fully functional for xorriso.
 */
 int search_mode;
 /* 0= start text
    1= fgrep , 
    2= regular expression
    3= (eventually structured) shell parser expression
    4= shell parser expression for leaf name
 */

 int structured_search;
 /* 0= flat text search
    1= '/' is a significant separator that cannot be matched by wildcards
    ( 2= like 1 : but report only occurence in tree, no payload, no location )
    ( 3= like 2 : but report first content level of matching directories )
    4= actually not structured but unique find mode (with search_mode 4)
 */

 int do_iso_rr_pattern; /* 0=off, 1=on, 2=ls */
 int do_disk_pattern;   /* 0=off, 1=on, 2=ls */

 int temp_mem_limit;

 struct ExclusionS *disk_exclusions;
 int disk_excl_mode; /* bit0= on (else off)
                        bit1= parameter too (else rekursion only)
                        bit2= whole subtree banned (else only exact path)
                        bit3= when comparing ignore excluded files rather
                              than to treat them as truely missing on disk
                     */

 int use_stdin; /* use raw stdin even if readline support is compiled */
 int result_page_length;
 int result_page_width;
 char mark_text[SfileadrL]; /* ( stdout+stderr, M: ) */
 int packet_output;
 char logfile[4][SfileadrL];
 int status_history_max; /* for -status long_history */

 char report_about_text[20];
 int report_about_severity;
 int library_msg_direct_print;
 char abort_on_text[20];
 int abort_on_severity; /* A severity rank number as threshold */
 int problem_status; /* Severity rank number. 0= no abort condition present */
 char problem_status_text[20];

 char errfile_log[SfileadrL]; /* for -errfile_log */
 int errfile_mode; /* bit0= marked */
 FILE *errfile_fp;

 int img_read_error_mode; /* 0=best_effort , 1=failure , 2=fatal */

 char return_with_text[20];
 int return_with_severity;
 int return_with_value;
 int eternal_problem_status;
 char eternal_problem_status_text[20];

 /* temporary search facilities */
#ifdef Xorriso_with_regeX
 regex_t *re;
 regmatch_t match[1]; 
#endif /* Xorriso_with_regeX */
 char **re_constants;
 int re_count;
 int re_fill;
 char reg_expr[2*SfileadrL];

 /* run state */
 int run_state; /* 0=preparing , 1=writing image */
 int is_dialog;
 int bar_is_fresh;
 char pending_option[SfileadrL]; /* eventual option entered at page prompt */
 int request_to_abort; /* abort a single operation like -ls, not the program */
 int request_not_to_ask; /* suppress reassure and pager */
 double idle_time;
 int re_failed_at; /* mismatch position with structured_search */
 int prepended_wd;
 double insert_count;
 double insert_bytes;
 double error_count; /* double will not roll over */

 /* pacifiers */
 int pacifier_style; /* 0= xorriso, 1=mkisofs 2=cdrecord */
 double pacifier_interval;
 double start_time;
 double last_update_time;
 /* optional global counters for brain reduced callback functions */
 off_t pacifier_count;
 off_t pacifier_total;
 off_t pacifier_byte_count; /* auxiliary counter for data bytes */

 void *pacifier_fifo;

 int find_compare_result; /* 1=everything matches , 0=mismatch , -1=error */ 

 struct PermiteM *perm_stack; /* Temporarily altered dir access permissions */

 /* result (stdout, R: ) */
 char result_line[5*SfileadrL];
 int result_line_counter;
 int result_page_counter;
 int result_open_line_len;


 /* info (stderr, I:) */
 char info_text[10*SfileadrL];

};


int Xorriso_prepare_regex(struct XorrisO *xorriso, char *adr, int flag);

int Xorriso_result(struct XorrisO *xorriso, int flag);

int Xorriso_info(struct XorrisO *xorriso, int flag);

int Xorriso_request_confirmation(struct XorrisO *xorriso, int flag);

int Xorriso_prescan_args(struct XorrisO *xorriso, int argc, char **argv,
                         int flag);

int Xorriso_execute_option(struct XorrisO *xorriso, char *line, int flag);

/* @return 0=match , else no match
*/
int Xorriso_regexec(struct XorrisO *xorriso, char *to_match, int *failed_at,
                    int flag);

int Xorriso_prepare_expansion_pattern(struct XorrisO *xorriso, char *pattern,
                                      int flag);

int Xorriso__mode_to_perms(mode_t st_mode, char perms[10], int flag);

int Xorriso_much_too_long(struct XorrisO *xorriso, int len, int flag);

int Xorriso_check_temp_mem_limit(struct XorrisO *xorriso, off_t mem, int flag);

int Xorriso_eval_nonmatch(struct XorrisO *xorriso, char *pattern,
                          int *nonconst_mismatches, off_t *mem, int flag);

/* @param flag bit0= a match count !=1 is a SORRY event
*/
int Xorriso_check_matchcount(struct XorrisO *xorriso,
                int count, int nonconst_mismatches, int num_patterns,
                char **patterns, int flag);

int Xorriso_no_pattern_memory(struct XorrisO *xorriso, off_t mem, int flag);

int Xorriso_alloc_pattern_mem(struct XorrisO *xorriso, off_t mem,
                              int count, char ***filev, int flag);

/* @param flag bit0= count results rather than storing them
   @return <=0 error , 1 is root (end processing) ,
                       2 is not root (go on processing)
*/
int Xorriso_check_for_root_pattern(struct XorrisO *xorriso,
              int *filec, char **filev, int count_limit, off_t *mem, int flag);

/* @param flag bit0= prepend wd only if name does not begin by '/'
               bit2= prepend wd (automatically done if wd[0]!=0)
*/
int Xorriso_make_abs_adr(struct XorrisO *xorriso, char *wd, char *name,
                             char adr[], int flag);

/* @param flag bit0= count result rather than storing it
               bit1= unexpected change of number is a FATAL event
*/
int Xorriso_register_matched_adr(struct XorrisO *xorriso,
                               char *adr, int count_limit,
                               int *filec, char **filev, off_t *mem, int flag);

int Xorriso_format_ls_l(struct XorrisO *xorriso, struct stat *stbuf, int flag);

/* @param flag bit0= simple readlink(): no normalization, no multi-hop
*/
int Xorriso_resolve_link(struct XorrisO *xorriso,
                     char *link_path, char result_path[SfileadrL], int flag);

/* @param flag bit0= for Xorriso_msgs_submit: use pager
*/
int Xorriso_hop_link(struct XorrisO *xorriso, char *link_path,
                   struct LinkiteM **link_stack, struct stat *stbuf, int flag);

/* reg_expr should be twice as large as bourne_expr ( + 2 to be exact) */
/* return: 2= bourne_expr is surely a constant */
int Xorriso__bourne_to_reg(char bourne_expr[], char reg_expr[], int flag);

int Xorriso_no_malloc_memory(struct XorrisO *xorriso, char **to_free,
                             int flag);

int Xorriso_pacifier_reset(struct XorrisO *xorriso, int flag);

/* This call is to be issued by long running workers in short intervals.
   It will check whether enough time has elapsed since the last pacifier
   message and eventually issue an update message.
   @param what_done  A sparse description of the action, preferrably in past 
                     tense. E.g. "done" , "files added".
   @param count The number of objects processed so far.
                Is ignored if <=0.
   @param todo  The number of objects to be done in total.
                Is ignored if <=0.
   @param current_object A string telling the object currently processed.
                         Ignored if "".
   @param flag  bit0= report unconditionally, no time check
*/
int Xorriso_pacifier_callback(struct XorrisO *xorriso, char *what_done,
                              off_t count, off_t todo, char *current_object,
                              int flag);

/* @param boss_iter Opaque handle to be forwarded to actions in ISO image
                    Set to NULL if calling this function from outside ISO world
   @param flag bit0= update rather than compare
*/
int Xorriso_find_compare(struct XorrisO *xorriso, void *boss_iter,
                         char *iso_path, char *iso_prefix, char *disk_prefix, 
                         int flag);

/* @param boss_iter Opaque handle to be forwarded to actions in ISO image
                    Set to NULL if calling this function from outside ISO world
*/
int Xorriso_update_interpreter(struct XorrisO *xorriso, void *boss_iter,
                               int compare_result, char *disk_path,
                               char *iso_rr_path, int flag);

int Xorriso_path_is_excluded(struct XorrisO *xorriso, char *path, int flag);

/* @param flag bit0= long format
               bit1= do not print count of nodes
               bit2= du format
               bit3= print directories as themselves (ls -d)
*/
int Xorriso_lsx_filev(struct XorrisO *xorriso, char *wd,
                      int filec, char **filev, off_t boss_mem, int flag);

/*
   @param flag >>> bit0= remove whole sub tree: rm -r
               bit1= remove empty directory: rmdir
               bit2= recursion: do not reassure in mode 2 "tree"
               bit3= this is for overwriting and not for plain removal
               bit4= count deleted files in xorriso->pacifier_count
               bit5= with bit0 only remove directory content, not the directory
   @return   <=0 = error
               1 = removed leaf file object
               2 = removed directory or tree
               3 = did not remove on user revocation
*/
int Xorriso_rmx(struct XorrisO *xorriso, off_t boss_mem, char *path, int flag);


int Xorriso_make_tmp_path(struct XorrisO *xorriso, char *orig_path,
                          char *tmp_path, int *fd, int flag);

/* @param flag bit0= path is a directory
               bit2= recursion: do not reassure in mode 2 "tree"
               bit3= this is for overwriting and not for plain removal
*/
int Xorriso_reassure_restore(struct XorrisO *xorriso, char *path, int flag);


int Xorriso_auto_chmod(struct XorrisO *xorriso, char *disk_path, int flag);

int Xorriso_protect_stdout(struct XorrisO *xorriso, int flag);


int Sfile_str(char target[SfileadrL], char *source, int flag);

double Sfile_microtime(int flag);

int Sfile_add_to_path(char path[SfileadrL], char *addon, int flag);

int Sfile_scale(double value, char *result, int siz, double thresh, int flag);

int Sfile_destroy_argv(int *argc, char ***argv, int flag);

/*
 bit0= do not ignore trailing slash
 bit1= do not ignore empty components (other than the empty root name)
*/
int Sfile_count_components(char *path, int flag);

/*
 @param flag
 bit0= return -1 if file is missing
 bit1= return a hardlink with siblings as type 5
 bit2= evaluate eventual link target rather than the link object itself
 bit3= return a socket or a char device as types 7 or 8 rather than 0
 @return
  0=unknown
  1=regular
  2=directory
  3=symbolic link
  4=named pipe
  5=multiple hardlink (with bit1)
  6=block device
  7=socket (with bit3)
  8=character device (with bit3)
*/
int Sfile_type(char *filename, int flag);



char *Text_shellsafe(char *in_text, char *out_text, int flag);

int Sort_argv(int argc, char **argv, int flag);

/* @param flag bit0= single letters */
char *Ftypetxt(mode_t st_mode, int flag);

/* @param flag bit0=with year and seconds
               bit1=timestamp format YYYY.MM.DD.hhmmss
*/
char *Ftimetxt(time_t t, char timetext[40], int flag);


struct DirseQ;

int Dirseq_new(struct DirseQ **o, char *adr, int flag);

int Dirseq_destroy(struct DirseQ **o, int flag);

int Dirseq_next_adr(struct DirseQ *o, char reply[SfileadrL], int flag);


int Linkitem_reset_stack(struct LinkiteM **o, struct LinkiteM *to, int flag);


struct FindjoB;

/* @return 0=no match , 1=match , <0 = error
*/
int Findjob_test(struct FindjoB *job, char *name, 
                 struct stat *boss_stbuf, struct stat *stbuf,
                 int depth, int flag);

/* @return <0 error, >=0 see xorriso.c struct FindjoB.action
*/
int Findjob_get_action(struct FindjoB *o, int flag);

/* @return <0 error, >=0 see xorriso.c struct FindjoB.action
*/ 
int Findjob_get_action_parms(struct FindjoB *o, char **target,
                             uid_t *user, gid_t *group,
                             mode_t *mode_and, mode_t *mode_or,
                             int *type, time_t *date, struct FindjoB **subjob,
                             int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_target(struct FindjoB *o, int action, char *target,
                              int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_chgrp(struct FindjoB *o, gid_t group, int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_chmod(struct FindjoB *o,
                             mode_t mode_and, mode_t mode_or, int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_ad(struct FindjoB *o, int type, time_t date, int flag);

int Findjob_set_start_path(struct FindjoB *o, char *start_path, int flag);

int Findjob_get_start_path(struct FindjoB *o, char **start_path, int flag);


struct SplitparT;

int Splitparts_new(struct SplitparT **o, int count, int flag);

int Splitparts_destroy(struct SplitparT **o, int count, int flag);

int Splitparts_set(struct SplitparT *o, int idx,
                   char *name, int partno, int total_parts,
                   off_t offset, off_t bytes, off_t total_bytes, int flag);

int Splitparts_get(struct SplitparT *o, int idx, char **name, int *partno,
                   int *total_parts, off_t *offset, off_t *bytes,
                   off_t *total_bytes, int flag);

int Splitpart__parse(char *name, int *partno, int *total_parts,
                    off_t *offset, off_t *bytes, off_t *total_bytes, int flag);

int Splitpart__compose(char *adr, int partno, int total_parts,
                       off_t offset, off_t bytes, off_t total_bytes, int flag);

int Splitparts_sort(struct SplitparT *o, int count, int flag);

struct LstrinG {
  char *text;
  struct LstrinG *prev,*next;
};
 
int Lstring_destroy(struct LstrinG **lstring, int flag);

int Lstring_destroy_all(struct LstrinG **lstring, int flag);

/*
 @param flag Bitfield for control purposes
        bit0= insert before link rather than after it 
        bit1= do not copy data (e.g. because *data is invalid)
*/
int Lstring_new_binary(struct LstrinG **lstring, char *data, int data_len,
                       struct LstrinG *link, int flag);

/* 
 @param flag Bitfield for control purposes
             bit0= insert before link rather than after it
*/
int Lstring_new(struct LstrinG **lstring, char *text, struct LstrinG *link,
                int flag);

int Lstring_append_binary(struct LstrinG **entry, char *data, int data_len,
                           int flag);


int Permstack_push(struct PermiteM **o, char *disk_path, struct stat *stbuf,
                   int flag);

int Permstack_pop(struct PermiteM **o, struct PermiteM *stopper,
                  struct XorrisO *xorriso, int flag);


#endif /* Xorriso_private_includeD */

