/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/object.h
//! @brief Reference counted base class interface.
//!
//! Provides a reference counted base class for dynamic objects handled the Inti smart pointer.

#ifndef INTI_OBJECT_H
#define INTI_OBJECT_H

#ifndef INTI_BASE_H
#include <inti/base.h>
#endif

#ifndef INTI_UTF_STRING_H
#include <inti/utf-string.h>
#endif

namespace Inti {

namespace G {
class Boxed;
}

//! @class ReferencedObject object.h inti/object.h
//! @brief Reference counted base class.
//!
//! ReferencedObject is a reference counting base class. Derived from ReferencedBase, it adds
//! an integer reference counter so that dynamic objects can have their memory allocation
//! handled by the Inti smart pointer: Pointer<>. This keeps the memory management in Inti
//! consistent across all classes. If you derive a class from ReferencedObject and allocate it
//! on the heap, you free the memory and destroy the object by calling unref(), not delete.
//!
//! Note if you declare a class that derives from ReferencedObject on the stack you don't
//! need to call unref();

class ReferencedObject : virtual public ReferencedBase
{
	friend class G::Boxed;

	ReferencedObject(const ReferencedObject&);
	ReferencedObject& operator=(const ReferencedObject&);

	int ref_count;

protected:
	ReferencedObject();
	//!< Constructor.

	virtual ~ReferencedObject() = 0;
	//!< Destructor.

public:
	virtual void ref();
	//!< Increase an object's reference count by one.

	virtual void unref();
	//!< Decrease an object's reference count by one.
	//!< When the reference count becomes zero delete is called. Remember, with ReferencedObject
	//!< you must call unref() on dynmaically allocated objects, not delete.
};

} // namespace Inti

#endif // INTI_OBJECT_H

