/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/displaymanager.h
//! @brief A GdkDisplayManager C++ wrapper interface.
//!
//! Provides DisplayManager, an object that Maintains a list of all open Gdk::Displays.

#ifndef INTI_GDK_DISPLAY_MANAGER_H
#define INTI_GDK_DISPLAY_MANAGER_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef INTI_GDK_TYPES
#include <inti/gdk/types.h>
#endif

#ifndef _CPP_VECTOR
#include <vector>
#endif

namespace Inti {

namespace Gdk {

class Display;
class DisplayManagerClass;

//! @class DisplayManager displaymanager.h inti/gdk/displaymanager.h
//! @brief A GdkDisplayManager C++ wrapper class.
//!
//! The purpose of the DisplayManager singleton object is to offer notification
//! when displays appear or disappear or the default display changes.

class DisplayManager : public G::Object
{
	friend class G::Object;
	friend class DisplayManagerClass;

	DisplayManager(const DisplayManager&);
	DisplayManager& operator=(const DisplayManager&);

protected:
//! @name Constructors
//! @{

	explicit DisplayManager(GdkDisplayManager *display_manager, bool reference = true);
	//!< Construct a new DisplayManager from an existing GdkDisplayManager.
	//!< @param display_manager A pointer to a GdkDisplayManager.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>display_manager</EM> can be a newly created GdkDisplayManager or an existing
	//!< GdkDisplayManager. (see G::Object::Object).

//! @}
//! @name Signal Handlers
//! @{

	virtual void on_display_opened(Display& display);
	//!< Called when a display is opened.
	//!< @param display The Display being opened.

//! @}
//  Properties

	typedef G::Property<Display*, G::Object*> DefaultDisplayPropertyType;
	typedef G::PropertyProxy<G::Object, DefaultDisplayPropertyType> DefaultDisplayPropertyProxy;
	static const DefaultDisplayPropertyType default_display_property;

//  Signals

	typedef G::Signal1<void, GdkDisplay*> DisplayOpenedSignalType;
	typedef G::SignalProxy<TypeInstance, DisplayOpenedSignalType> DisplayOpenedSignalProxy;
	static const DisplayOpenedSignalType display_opened_signal;

public:
//! @name Constructors
//! @{

	virtual ~DisplayManager();
	//!< Destructor.

//! @}

	static DisplayManager* get();
	//!< Returns the global DisplayManager singleton object.

//! @name Accessors
//! @{

	GdkDisplayManager* gdk_display_manager() const { return (GdkDisplayManager*)instance; }
	//!< Get a pointer to the GdkDisplayManager structure.

	GdkDisplayManagerClass* gdk_display_manager_class() const;
	//!< Get a pointer to the GdkDisplayManagerClass structure.

	operator GdkDisplayManager* () const;
	//!< Conversion operator; safely converts a DisplayManager to a GdkDisplayManager pointer.

	Display* get_default_display() const;
	//!< Gets the default Display.
	//!< @return A Display, or null if there is no default display.

	bool list_displays(std::vector<Display*>& displays) const;
	//!< List all currently open displays.
	//!< @param displays A reference to a vector of Display* to hold the list of opened displays.
	//!< @return <EM>true</EM> if the vector is not empty.

//! @}
//! @name Methods
//! @{

	void set_default_display(const Display& display);
	//!< Sets display as the default display.
	//!< @param display A Display.

//! @}
//! @name Property Proxies
//! @{

	const DefaultDisplayPropertyProxy prop_default_display()
	{
		return DefaultDisplayPropertyProxy(this, &default_display_property);
	}
	//!< The default Display (Display* : Read / Write).

//! @}
//! @name Signal Proxies
//! @{

	const DisplayOpenedSignalProxy sig_display_opened()
	{
		return DisplayOpenedSignalProxy(this, &display_opened_signal);
	}
	//!< Connect to the display_opened_signal; emitted when a Display is opened.

//! @}
};

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_DISPLAY_MANAGER_H

