/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  table.cc - AtkTable C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "table.h"
#include "private/table_p.h"
#include "object.h"
#include "../glib/object.h"

using namespace Inti;

/*  Atk::Table
 */

Atk::Table::~Table()
{
}

AtkTableIface*
Atk::Table::atk_table_iface() const
{
	return peek<AtkTableIface>(ATK_TYPE_TABLE);
}

Atk::Table::operator AtkTable* () const
{
	return this ? atk_table() : 0;
}

Atk::Object*
Atk::Table::ref_at(int row, int column) const
{
	AtkObject *object = atk_table_ref_at(atk_table(), row, column);
	return object ? G::Object::wrap<Object>(object) : 0;
}

int
Atk::Table::get_index_at(int row, int column) const
{
	return atk_table_get_index_at(atk_table(), row, column);
}

int
Atk::Table::get_column_at_index(int index) const
{
	return atk_table_get_column_at_index(atk_table(), index);
}

int
Atk::Table::get_row_at_index(int index) const
{
	return atk_table_get_row_at_index(atk_table(), index);
}

int
Atk::Table::get_n_columns() const
{
	return atk_table_get_n_columns(atk_table());
}

int
Atk::Table::get_n_rows() const
{
	return atk_table_get_n_rows(atk_table());
}

int
Atk::Table::get_column_extent_at(int row, int column) const
{
	return atk_table_get_column_extent_at(atk_table(), row, column);
}

int
Atk::Table::get_row_extent_at(int row, int column) const
{
	return atk_table_get_row_extent_at(atk_table(), row, column);
}

Atk::Object*
Atk::Table::get_caption() const
{
	AtkObject *object = atk_table_get_caption(atk_table());
	return object ? G::Object::wrap<Object>(object) : 0;
}

String
Atk::Table::get_column_description(int column) const
{
	return atk_table_get_column_description(atk_table(), column);
}

Atk::Object*
Atk::Table::get_column_header(int column) const
{
	AtkObject *object = atk_table_get_column_header(atk_table(), column);
	return object ? G::Object::wrap<Object>(object) : 0;
}

String
Atk::Table::get_row_description(int row) const
{
	return atk_table_get_row_description(atk_table(), row);
}

Atk::Object*
Atk::Table::get_row_header(int row) const
{
	AtkObject *object = atk_table_get_row_header(atk_table(), row);
	return object ? G::Object::wrap<Object>(object) : 0;
}

Atk::Object*
Atk::Table::get_summary() const
{
	AtkObject *object = atk_table_get_summary(atk_table());
	return object ? G::Object::wrap<Object>(object) : 0;
}

bool
Atk::Table::get_selected_columns(std::vector<int>& selected) const
{
	g_return_val_if_fail(selected.empty(), false);
	int *tmp_selected = 0;
	int n_columns = atk_table_get_selected_columns(atk_table(), &tmp_selected);

	int i = 0;
	while (i < n_columns)
	{
		selected.push_back(tmp_selected[i]);
		++i;
	}

	g_free(tmp_selected);
	return !selected.empty();
}

bool
Atk::Table::get_selected_rows(std::vector<int>& selected) const
{
	g_return_val_if_fail(selected.empty(), false);
	int *tmp_selected = 0;
	int n_rows = atk_table_get_selected_rows(atk_table(), &tmp_selected);

	int i = 0;
	while (i < n_rows)
	{
		selected.push_back(tmp_selected[i]);
		++i;
	}

	g_free(tmp_selected);
	return !selected.empty();
}

bool
Atk::Table::is_column_selected(int column) const
{
	return atk_table_is_column_selected(atk_table(), column);
}

bool
Atk::Table::is_row_selected(int row) const
{
	return atk_table_is_row_selected(atk_table(), row);
}

bool
Atk::Table::is_selected(int row, int column) const
{
	return atk_table_is_selected(atk_table(), row, column);
}

void
Atk::Table::set_caption(const Object *caption)
{
	atk_table_set_caption(atk_table(), *caption);
}

void
Atk::Table::set_column_description(int column, const String& description)
{
	atk_table_set_column_description(atk_table(), column, description.c_str());
}

void
Atk::Table::set_column_header(int column, const Object *header)
{
	atk_table_set_column_header(atk_table(), column, *header);
}

void
Atk::Table::set_row_description(int row, const String& description)
{
	atk_table_set_row_description(atk_table(), row, description.c_str());
}

void
Atk::Table::set_row_header(int row, const Object *header)
{
	atk_table_set_row_header(atk_table(), row, *header);
}

void
Atk::Table::set_summary(const Object *accessible)
{
	atk_table_set_summary(atk_table(), *accessible);
}

bool
Atk::Table::add_row_selection(int row)
{
	return atk_table_add_row_selection(atk_table(), row);
}

bool
Atk::Table::remove_row_selection(int row)
{
	return atk_table_remove_row_selection(atk_table(), row);
}

bool
Atk::Table::add_column_selection(int column)
{
	return atk_table_add_column_selection(atk_table(), column);
}

bool
Atk::Table::remove_column_selection(int column)
{
	return atk_table_remove_column_selection(atk_table(), column);
}

/*  Atk::TableIface
 */

void
Atk::TableIface::init(AtkTableIface *g_iface)
{
	g_iface->ref_at = &ref_at_proxy;
	g_iface->get_index_at = &get_index_at_proxy;
	g_iface->get_column_at_index = &get_column_at_index_proxy;
	g_iface->get_row_at_index = &get_row_at_index_proxy;
	g_iface->get_n_columns = &get_n_columns_proxy;
	g_iface->get_n_rows = &get_n_rows_proxy;
	g_iface->get_column_extent_at = &get_column_extent_at_proxy;
	g_iface->get_row_extent_at = &get_row_extent_at_proxy;
	g_iface->get_caption = &get_caption_proxy;
	g_iface->get_column_description = &get_column_description_proxy;
	g_iface->get_column_header = &get_column_header_proxy;
	g_iface->get_row_description = &get_row_description_proxy;
	g_iface->get_row_header = &get_row_header_proxy;
	g_iface->get_summary = &get_summary_proxy;
	g_iface->set_caption = &set_caption_proxy;
	g_iface->set_column_description = &set_column_description_proxy;
	g_iface->set_column_header = &set_column_header_proxy;
	g_iface->set_row_description = &set_row_description_proxy;
	g_iface->set_row_header = &set_row_header_proxy;
	g_iface->set_summary = &set_summary_proxy;
	g_iface->get_selected_columns = &get_selected_columns_proxy;
	g_iface->get_selected_rows = &get_selected_rows_proxy;
	g_iface->is_column_selected = &is_column_selected_proxy;
	g_iface->is_row_selected = &is_row_selected_proxy;
	g_iface->is_selected = &is_selected_proxy;
	g_iface->add_row_selection = &add_row_selection_proxy;
	g_iface->remove_row_selection = &remove_row_selection_proxy;
	g_iface->add_column_selection = &add_column_selection_proxy;
	g_iface->remove_column_selection = &remove_column_selection_proxy;
	g_iface->row_inserted = &row_inserted_proxy;
	g_iface->column_inserted = &column_inserted_proxy;
	g_iface->row_deleted = &row_deleted_proxy;
	g_iface->column_deleted = &column_deleted_proxy;
	g_iface->row_reordered = &row_reordered_proxy;
	g_iface->column_reordered = &column_reordered_proxy;
	g_iface->model_changed = &model_changed_proxy;
}

AtkObject*
Atk::TableIface::ref_at_proxy(AtkTable *table, gint row, gint column)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_ref_at(row, column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->ref_at)
			result = g_iface->ref_at(table, row, column);
	}
	return result;
}

gint
Atk::TableIface::get_index_at_proxy(AtkTable *table, gint row, gint column)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_index_at(row, column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_index_at)
			result = g_iface->get_index_at(table, row, column);
	}
	return result;
}

gint
Atk::TableIface::get_column_at_index_proxy(AtkTable *table, gint index)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_column_at_index(index);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_column_at_index)
			result = g_iface->get_column_at_index(table, index);
	}
	return result;
}

gint
Atk::TableIface::get_row_at_index_proxy(AtkTable *table, gint index)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_row_at_index(index);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_row_at_index)
			result = g_iface->get_row_at_index(table, index);
	}
	return result;
}

gint
Atk::TableIface::get_n_columns_proxy(AtkTable *table)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_n_columns();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_n_columns)
			result = g_iface->get_n_columns(table);
	}
	return result;
}

gint
Atk::TableIface::get_n_rows_proxy(AtkTable *table)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_n_rows();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_n_rows)
			result = g_iface->get_n_rows(table);
	}
	return result;
}

gint
Atk::TableIface::get_column_extent_at_proxy(AtkTable *table, gint row, gint column)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_column_extent_at(row, column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_column_extent_at)
			result = g_iface->get_column_extent_at(table, row, column);
	}
	return result;
}

gint
Atk::TableIface::get_row_extent_at_proxy(AtkTable *table, gint row, gint column)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_row_extent_at(row, column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_row_extent_at)
			result = g_iface->get_row_extent_at(table, row, column);
	}
	return result;
}

AtkObject*
Atk::TableIface::get_caption_proxy(AtkTable *table)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_caption();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_caption)
			result = g_iface->get_caption(table);
	}
	return result;
}

const gchar*
Atk::TableIface::get_column_description_proxy(AtkTable *table, gint column)
{
	const gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_column_description(column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_column_description)
			result = g_iface->get_column_description(table, column);
	}
	return result;
}

AtkObject*
Atk::TableIface::get_column_header_proxy(AtkTable *table, gint column)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_column_header(column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_column_header)
			result = g_iface->get_column_header(table, column);
	}
	return result;
}

const gchar*
Atk::TableIface::get_row_description_proxy(AtkTable *table, gint row)
{
	const gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_row_description(row);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_row_description)
			result = g_iface->get_row_description(table, row);
	}
	return result;
}

AtkObject*
Atk::TableIface::get_row_header_proxy(AtkTable *table, gint row)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_row_header(row);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_row_header)
			result = g_iface->get_row_header(table, row);
	}
	return result;
}

AtkObject*
Atk::TableIface::get_summary_proxy(AtkTable *table)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_summary();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_summary)
			result = g_iface->get_summary(table);
	}
	return result;
}

void
Atk::TableIface::set_caption_proxy(AtkTable *table, AtkObject *caption)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_caption(caption);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_caption)
			g_iface->set_caption(table, caption);
	}
}

void
Atk::TableIface::set_column_description_proxy(AtkTable *table, gint column, const gchar *description)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_column_description(column, description);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_column_description)
			g_iface->set_column_description(table, column, description);
	}
}

void
Atk::TableIface::set_column_header_proxy(AtkTable *table, gint column, AtkObject *header)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_column_header(column, header);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_column_header)
			g_iface->set_column_header(table, column, header);
	}
}

void
Atk::TableIface::set_row_description_proxy(AtkTable *table, gint row, const gchar *description)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_row_description(row, description);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_row_description)
			g_iface->set_row_description(table, row, description);
	}
}

void
Atk::TableIface::set_row_header_proxy(AtkTable *table, gint row, AtkObject *header)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_row_header(row, header);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_row_header)
			g_iface->set_row_header(table, row, header);
	}
}

void
Atk::TableIface::set_summary_proxy(AtkTable *table, AtkObject *accessible)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->do_set_summary(accessible);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->set_summary)
			g_iface->set_summary(table, accessible);
	}
}

gint
Atk::TableIface::get_selected_columns_proxy(AtkTable *table, gint **selected)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_selected_columns(selected);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_selected_columns)
			result = g_iface->get_selected_columns(table, selected);
	}
	return result;
}

gint
Atk::TableIface::get_selected_rows_proxy(AtkTable *table, gint **selected)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_get_selected_rows(selected);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->get_selected_rows)
			result = g_iface->get_selected_rows(table, selected);
	}
	return result;
}

gboolean
Atk::TableIface::is_column_selected_proxy(AtkTable *table, gint column)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_is_column_selected(column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->is_column_selected)
			result = g_iface->is_column_selected(table, column);
	}
	return result;
}

gboolean
Atk::TableIface::is_row_selected_proxy(AtkTable *table, gint row)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_is_row_selected(row);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->is_row_selected)
			result = g_iface->is_row_selected(table, row);
	}
	return result;
}

gboolean
Atk::TableIface::is_selected_proxy(AtkTable *table, gint row, gint column)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_is_selected(row, column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->is_selected)
			result = g_iface->is_selected(table, row, column);
	}
	return result;
}

gboolean
Atk::TableIface::add_row_selection_proxy(AtkTable *table, gint row)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_add_row_selection(row);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->add_row_selection)
			result = g_iface->add_row_selection(table, row);
	}
	return result;
}

gboolean
Atk::TableIface::remove_row_selection_proxy(AtkTable *table, gint row)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_remove_row_selection(row);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->remove_row_selection)
			result = g_iface->remove_row_selection(table, row);
	}
	return result;
}

gboolean
Atk::TableIface::add_column_selection_proxy(AtkTable *table, gint column)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_add_column_selection(column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->add_column_selection)
			result = g_iface->add_column_selection(table, column);
	}
	return result;
}

gboolean
Atk::TableIface::remove_column_selection_proxy(AtkTable *table, gint column)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		result = tmp_table->do_remove_column_selection(column);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->remove_column_selection)
			result = g_iface->remove_column_selection(table, column);
	}
	return result;
}

void
Atk::TableIface::row_inserted_proxy(AtkTable *table, gint row, gint num_inserted)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_row_inserted(row, num_inserted);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->row_inserted)
			g_iface->row_inserted(table, row, num_inserted);
	}
}

void
Atk::TableIface::column_inserted_proxy(AtkTable *table, gint column, gint num_inserted)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_column_inserted(column, num_inserted);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->column_inserted)
			g_iface->column_inserted(table, column, num_inserted);
	}
}

void
Atk::TableIface::row_deleted_proxy(AtkTable *table, gint row, gint num_deleted)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_row_deleted(row, num_deleted);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->row_deleted)
			g_iface->row_deleted(table, row, num_deleted);
	}
}

void
Atk::TableIface::column_deleted_proxy(AtkTable *table, gint column, gint num_deleted)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_column_deleted(column, num_deleted);
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->column_deleted)
			g_iface->column_deleted(table, column, num_deleted);
	}
}

void
Atk::TableIface::row_reordered_proxy(AtkTable *table)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_row_reordered();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->row_reordered)
			g_iface->row_reordered(table);
	}
}

void
Atk::TableIface::column_reordered_proxy(AtkTable *table)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_column_reordered();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->column_reordered)
			g_iface->column_reordered(table);
	}
}

void
Atk::TableIface::model_changed_proxy(AtkTable *table)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(table));
	Atk::Table *tmp_table = dynamic_cast<Atk::Table*>(object);
	if (tmp_table)
		tmp_table->on_model_changed();
	else
	{
		AtkTableIface *g_iface = static_cast<AtkTableIface*>(g_type_interface_peek_parent(ATK_TABLE_GET_IFACE(table)));
		if (g_iface->model_changed)
			g_iface->model_changed(table);
	}
}

/*  Overridable methods
 */

AtkObject*
Atk::Table::do_ref_at(int row, int column)
{
	AtkObject *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->ref_at)
		result = g_iface->ref_at(atk_table(), row, column);
	return result;
}

int
Atk::Table::do_get_index_at(int row, int column)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_index_at)
		result = g_iface->get_index_at(atk_table(), row, column);
	return result;
}

int
Atk::Table::do_get_column_at_index(int index)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_column_at_index)
		result = g_iface->get_column_at_index(atk_table(), index);
	return result;
}

int
Atk::Table::do_get_row_at_index(int index)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_row_at_index)
		result = g_iface->get_row_at_index(atk_table(), index);
	return result;
}

int
Atk::Table::do_get_n_columns()
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_n_columns)
		result = g_iface->get_n_columns(atk_table());
	return result;
}

int
Atk::Table::do_get_n_rows()
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_n_rows)
		result = g_iface->get_n_rows(atk_table());
	return result;
}

int
Atk::Table::do_get_column_extent_at(int row, int column)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_column_extent_at)
		result = g_iface->get_column_extent_at(atk_table(), row, column);
	return result;
}

int
Atk::Table::do_get_row_extent_at(int row, int column)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_row_extent_at)
		result = g_iface->get_row_extent_at(atk_table(), row, column);
	return result;
}

AtkObject*
Atk::Table::do_get_caption()
{
	AtkObject *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_caption)
		result = g_iface->get_caption(atk_table());
	return result;
}

const char*
Atk::Table::do_get_column_description(int column)
{
	const char *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_column_description)
		result = g_iface->get_column_description(atk_table(), column);
	return result;
}

AtkObject*
Atk::Table::do_get_column_header(int column)
{
	AtkObject *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_column_header)
		result = g_iface->get_column_header(atk_table(), column);
	return result;
}

const char*
Atk::Table::do_get_row_description(int row)
{
	const char *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_row_description)
		result = g_iface->get_row_description(atk_table(), row);
	return result;
}

AtkObject*
Atk::Table::do_get_row_header(int row)
{
	AtkObject *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_row_header)
		result = g_iface->get_row_header(atk_table(), row);
	return result;
}

AtkObject*
Atk::Table::do_get_summary()
{
	AtkObject *result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_summary)
		result = g_iface->get_summary(atk_table());
	return result;
}

void
Atk::Table::do_set_caption(AtkObject *caption)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_caption)
		g_iface->set_caption(atk_table(), caption);
}

void
Atk::Table::do_set_column_description(int column, const char *description)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_column_description)
		g_iface->set_column_description(atk_table(), column, description);
}

void
Atk::Table::do_set_column_header(int column, AtkObject *header)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_column_header)
		g_iface->set_column_header(atk_table(), column, header);
}

void
Atk::Table::do_set_row_description(int row, const char *description)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_row_description)
		g_iface->set_row_description(atk_table(), row, description);
}

void
Atk::Table::do_set_row_header(int row, AtkObject *header)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_row_header)
		g_iface->set_row_header(atk_table(), row, header);
}

void
Atk::Table::do_set_summary(AtkObject *accessible)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->set_summary)
		g_iface->set_summary(atk_table(), accessible);
}

int
Atk::Table::do_get_selected_columns(int **selected)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_selected_columns)
		result = g_iface->get_selected_columns(atk_table(), selected);
	return result;
}

int
Atk::Table::do_get_selected_rows(int **selected)
{
	int result = 0;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->get_selected_rows)
		result = g_iface->get_selected_rows(atk_table(), selected);
	return result;
}

bool
Atk::Table::do_is_column_selected(int column)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->is_column_selected)
		result = g_iface->is_column_selected(atk_table(), column);
	return result;
}

bool
Atk::Table::do_is_row_selected(int row)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->is_row_selected)
		result = g_iface->is_row_selected(atk_table(), row);
	return result;
}

bool
Atk::Table::do_is_selected(int row, int column)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->is_selected)
		result = g_iface->is_selected(atk_table(), row, column);
	return result;
}

bool
Atk::Table::do_add_row_selection(int row)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->add_row_selection)
		result = g_iface->add_row_selection(atk_table(), row);
	return result;
}

bool
Atk::Table::do_remove_row_selection(int row)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->remove_row_selection)
		result = g_iface->remove_row_selection(atk_table(), row);
	return result;
}

bool
Atk::Table::do_add_column_selection(int column)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->add_column_selection)
		result = g_iface->add_column_selection(atk_table(), column);
	return result;
}

bool
Atk::Table::do_remove_column_selection(int column)
{
	bool result = false;
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->remove_column_selection)
		result = g_iface->remove_column_selection(atk_table(), column);
	return result;
}

/*  Signal handlers
 */

void
Atk::Table::on_row_inserted(int row, int num_inserted)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->row_inserted)
		g_iface->row_inserted(atk_table(), row, num_inserted);
}

void
Atk::Table::on_column_inserted(int column, int num_inserted)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->column_inserted)
		g_iface->column_inserted(atk_table(), column, num_inserted);
}

void
Atk::Table::on_row_deleted(int row, int num_deleted)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->row_deleted)
		g_iface->row_deleted(atk_table(), row, num_deleted);
}

void
Atk::Table::on_column_deleted(int column, int num_deleted)
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->column_deleted)
		g_iface->column_deleted(atk_table(), column, num_deleted);
}

void
Atk::Table::on_row_reordered()
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->row_reordered)
		g_iface->row_reordered(atk_table());
}

void
Atk::Table::on_column_reordered()
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->column_reordered)
		g_iface->column_reordered(atk_table());
}

void
Atk::Table::on_model_changed()
{
	AtkTableIface *g_iface = peek_parent<AtkTableIface>(atk_table_iface());
	if (g_iface->model_changed)
		g_iface->model_changed(atk_table());
}

// Signals

const Atk::Table::RowInsertedSignalType Atk::Table::row_inserted_signal("row_inserted");

const Atk::Table::ColumnInsertedSignalType Atk::Table::column_inserted_signal("column_inserted");

const Atk::Table::RowDeletedSignalType Atk::Table::row_deleted_signal("row_deleted");

const Atk::Table::ColumnDeletedSignalType Atk::Table::column_deleted_signal("column_deleted");

const Atk::Table::RowReorderedSignalType Atk::Table::row_reordered_signal("row_reordered");

const Atk::Table::ColumnReorderedSignalType Atk::Table::column_reordered_signal("column_reordered");

const Atk::Table::ModelChangedSignalType Atk::Table::model_changed_signal("model_changed");

