/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: ToggleCommandGroup.java,v 1.3 2006/02/25 22:06:08 pietschy Exp $
 */

package org.pietschy.command;


/**
 * ToggleGroupCommand managed mutually exclusive collections of
 * {@link ToggleCommand ToggleCommands}.
 */
public class
ToggleCommandGroup
extends CommandGroup
{

   // the controller is lazily created in getController();
   private ToggleGroupController controller;

   /**
    * Creates a new anonymous command group with the specified id.  CommandGroups are lazy loading in
    * that they will store the only the command ids until they are asked to create a button
    * or menu.  The command will use the specifed {@link CommandManager} to obtain commands.
    *
    */
   public ToggleCommandGroup(CommandManager commandManager)
   {
      super(commandManager);
   }

   /**
    * Creates a new command groups with the specified id.  CommandGroups are lazy loading in
    * that they will store the only the command ids until they are asked to create a button
    * or menu.  The command will use the specifed {@link CommandManager} to obtain commands.
    * @param groupId this groups unique id.
    */
   public ToggleCommandGroup(CommandManager commandManager, String groupId)
   {
      super(commandManager, groupId);
   }

   /**
    * Gets the {@link ToggleGroupController} managing this group.
    * @return this groups {@link ToggleGroupController}.
    */
   protected ToggleGroupController
   getController()
   {
      // once again we have to lazy load the controller as the configuration if happening
      // in the constuctors call to super.  I really need to fix this at some point.
      if (controller == null)
         controller = new ToggleGroupController();

      return controller;
   }

   /**
    * Ensures that only {@link ToggleCommand} instances are added to this group.
    * @param prospectiveMember the {@link Command} being added.
    * @return <tt>true</tt> if the prospective member is an instance of {@link ToggleCommand},
    * <tt>false</tt> otherwise.
    */
   public boolean
   isAllowableMember(Command prospectiveMember)
   {
      internalLog.enter("isAllowableMember");
      boolean allowed = prospectiveMember instanceof ToggleCommand;
      internalLog.returned(String.valueOf(allowed));
      internalLog.exit("isAllowableMember()");
      return allowed;
   }

   /**
    * Configures if this group allows the last selected command to be deselected.
    * @param emptySelectionAllowed <tt>true</tt> to allow the last selected command to be
    * deslected, <tt>false</tt> to ensure at least one command is always selected.
    */
   public void
   setEmptySelectionAllowed(boolean emptySelectionAllowed)
   {
      getController().setEmptySelectionAllowed(emptySelectionAllowed);
   }

   /**
    * Checks if this group allows the last selected command to be deselected.
    * @return <tt>true</tt> if the last selected command can be deslected, <tt>false</tt>
    * if at least one command must always be selected.
    */
   public boolean
   isEmptySelectionAllowed()
   {
      return getController().isEmptySelectionAllowed();
   }


   public boolean
   isExclusive()
   {
      return controller.isExclusive();
   }

   public void
   setExclusive(boolean exclusive)
   {
      getController().setExclusive(exclusive);
   }

}
