/* GtkDatabox - An extension to the gtk+ library
 * Copyright (C) 1998 - 2007  Dr. Roland Bock
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <gtkdatabox_canvas.h>
#include <gtk/gtkgc.h>
#include <math.h>

/* IDs of properties */
enum
{
   CANVAS_WIDTH = 1,
   CANVAS_HEIGHT
};

struct _GtkDataboxCanvasPrivate
{
   gint width;
   gint height;
   gfloat total_left;
   gfloat total_right;
   gfloat total_top;
   gfloat total_bottom;
   gfloat visible_left;
   gfloat visible_right;
   gfloat visible_top;
   gfloat visible_bottom;
   GtkDataboxScaleType scale_type_x;
   GtkDataboxScaleType scale_type_y;
   gfloat translation_factor_x;
   gfloat translation_factor_y;
   gboolean changed;
};

static gpointer parent_class = NULL;

static void
gtk_databox_canvas_set_property (GObject * object,
				 guint property_id,
				 const GValue * value, GParamSpec * pspec)
{
   GtkDataboxCanvas *canvas = GTK_DATABOX_CANVAS (object);

   switch (property_id)
   {
   case CANVAS_WIDTH:
      {
	 gtk_databox_canvas_set_width (canvas, g_value_get_int (value));
      }
      break;
   case CANVAS_HEIGHT:
      {
	 gtk_databox_canvas_set_height (canvas, g_value_get_int (value));
      }
      break;
   default:
      /* We don't have any other property... */
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
   }
}

static void
gtk_databox_canvas_get_property (GObject * object,
				 guint property_id,
				 GValue * value, GParamSpec * pspec)
{
   GtkDataboxCanvas *canvas = GTK_DATABOX_CANVAS (object);

   switch (property_id)
   {
   case CANVAS_WIDTH:
      {
	 g_value_set_int (value, gtk_databox_canvas_get_width (canvas));
      }
      break;
   case CANVAS_HEIGHT:
      {
	 g_value_set_int (value, gtk_databox_canvas_get_height (canvas));
      }
      break;
   default:
      /* We don't have any other property... */
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
   }
}

static void
canvas_finalize (GObject * object)
{
   GtkDataboxCanvas *canvas = GTK_DATABOX_CANVAS (object);

   g_free (canvas->priv);

   /* Chain up to the parent class */
   G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gtk_databox_canvas_class_init (gpointer g_class /*, gpointer g_class_data */ )
{
   GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
   GtkDataboxCanvasClass *klass = GTK_DATABOX_CANVAS_CLASS (g_class);
   GParamSpec *canvas_param_spec;

   parent_class = g_type_class_peek_parent (klass);

   gobject_class->set_property = gtk_databox_canvas_set_property;
   gobject_class->get_property = gtk_databox_canvas_get_property;
   gobject_class->finalize = canvas_finalize;

   canvas_param_spec = g_param_spec_int ("width", "Canvas width", "Width of canvas in pixels", G_MININT, G_MAXINT, 1,	/* default value */
					 G_PARAM_READWRITE);

   g_object_class_install_property (gobject_class,
				    CANVAS_WIDTH, canvas_param_spec);

   canvas_param_spec = g_param_spec_int ("height", "Canvas height", "Height of canvas in pixels", G_MININT, G_MAXINT, 1,	/* default value */
					 G_PARAM_READWRITE);

   g_object_class_install_property (gobject_class,
				    CANVAS_HEIGHT, canvas_param_spec);

}

static void
gtk_databox_canvas_instance_init (GTypeInstance * instance
				  /*, gpointer g_class */ )
{
   GtkDataboxCanvas *canvas = GTK_DATABOX_CANVAS (instance);

   canvas->priv = g_new0 (GtkDataboxCanvasPrivate, 1);

   gtk_databox_canvas_set_width (canvas, 1);
   gtk_databox_canvas_set_height (canvas, 1);
   gtk_databox_canvas_set_total_limits (canvas, -1., 1., 1., -1.);
   gtk_databox_canvas_set_visible_limits (canvas, -1., 1., 1., -1.);

   canvas->priv->scale_type_x = GTK_DATABOX_SCALE_LINEAR;
   canvas->priv->scale_type_y = GTK_DATABOX_SCALE_LINEAR;
}

GType
gtk_databox_canvas_get_type (void)
{
   static GType type = 0;

   if (type == 0)
   {
      static const GTypeInfo info = {
	 sizeof (GtkDataboxCanvasClass),
	 NULL,			/* base_init */
	 NULL,			/* base_finalize */
	 (GClassInitFunc) gtk_databox_canvas_class_init,	/* class_init */
	 NULL,			/* class_finalize */
	 NULL,			/* class_data */
	 sizeof (GtkDataboxCanvas),	/* instance_size */
	 0,			/* n_preallocs */
	 (GInstanceInitFunc) gtk_databox_canvas_instance_init,	/* instance_init */
	 NULL,			/* value_table */
      };
      type = g_type_register_static (G_TYPE_OBJECT,
				     "GtkDataboxCanvas", &info, 0);
   }

   return type;
}

GObject *
gtk_databox_canvas_new (void)
{
   GtkDataboxCanvas *canvas;

   canvas = g_object_new (GTK_DATABOX_TYPE_CANVAS, NULL);

   return G_OBJECT (canvas);
}

static void
gtk_databox_canvas_calculate_translation_factor (GtkDataboxCanvas * canvas)
{
   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
      canvas->priv->translation_factor_x =
	 canvas->priv->width / (canvas->priv->visible_right -
				canvas->priv->visible_left);
   else
      canvas->priv->translation_factor_x =
	 canvas->priv->width / log10 (canvas->priv->visible_right /
				    canvas->priv->visible_left);

   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
      canvas->priv->translation_factor_y =
	 canvas->priv->height / (canvas->priv->visible_bottom -
				 canvas->priv->visible_top);
   else
      canvas->priv->translation_factor_y =
	 canvas->priv->height / log10 (canvas->priv->visible_bottom /
				     canvas->priv->visible_top);

   canvas->priv->changed = FALSE;
}

void
gtk_databox_canvas_set_visible_range_x (GtkDataboxCanvas * canvas,
					gfloat offset, gfloat page_size)
{
   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
   {
      canvas->priv->visible_left = canvas->priv->total_left
	 + (canvas->priv->total_right - canvas->priv->total_left) * offset;
      canvas->priv->visible_right = canvas->priv->visible_left
	 + (canvas->priv->total_right - canvas->priv->total_left) * page_size;
   }
   else
   {
      canvas->priv->visible_left = canvas->priv->total_left
	 * pow (canvas->priv->total_right / canvas->priv->total_left, offset);
      canvas->priv->visible_right = canvas->priv->visible_left
	 * pow (canvas->priv->total_right / canvas->priv->total_left,
		page_size);
   }
   canvas->priv->changed = TRUE;
}

void
gtk_databox_canvas_set_visible_range_y (GtkDataboxCanvas * canvas,
					gfloat offset, gfloat page_size)
{
   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
   {
      canvas->priv->visible_top = canvas->priv->total_top
	 + (canvas->priv->total_bottom - canvas->priv->total_top) * offset;
      canvas->priv->visible_bottom = canvas->priv->visible_top
	 + (canvas->priv->total_bottom - canvas->priv->total_top) * page_size;
   }
   else
   {
      canvas->priv->visible_top = canvas->priv->total_top
	 * pow (canvas->priv->total_bottom / canvas->priv->total_top, offset);
      canvas->priv->visible_bottom = canvas->priv->visible_top
	 * pow (canvas->priv->total_bottom / canvas->priv->total_top,
		page_size);
   }
   canvas->priv->changed = TRUE;
}

void
gtk_databox_canvas_values_to_pixels (GtkDataboxCanvas * canvas, guint len,
				     const gfloat * values_x,
				     const gfloat * values_y,
				     GdkPoint * pixels)
{
   guint i;

   if (canvas->priv->changed)
      gtk_databox_canvas_calculate_translation_factor (canvas);

   for (i = 0; i < len; ++i, ++values_x, ++values_y, ++pixels)
   {
      if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
	 pixels->x =
	    (*values_x -
	     canvas->priv->visible_left) * canvas->priv->translation_factor_x;
      else
	 pixels->x =
	    log10 (*values_x / canvas->priv->visible_left) *
	    canvas->priv->translation_factor_x;

      if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
	 pixels->y =
	    (*values_y -
	     canvas->priv->visible_top) * canvas->priv->translation_factor_y;
      else
	 pixels->y =
	    log10 (*values_y / canvas->priv->visible_top) *
	    canvas->priv->translation_factor_y;
   }
}

gint16
gtk_databox_canvas_value_to_pixel_x (GtkDataboxCanvas * canvas, gfloat value)
{
   if (canvas->priv->changed)
      gtk_databox_canvas_calculate_translation_factor (canvas);

   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
      return (value -
	      canvas->priv->visible_left) *
	 canvas->priv->translation_factor_x;
   else
      return log10 (value / canvas->priv->visible_left) *
	 canvas->priv->translation_factor_x;
}

gint16
gtk_databox_canvas_value_to_pixel_y (GtkDataboxCanvas * canvas, gfloat value)
{
   if (canvas->priv->changed)
      gtk_databox_canvas_calculate_translation_factor (canvas);

   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
      return (value -
	      canvas->priv->visible_top) * canvas->priv->translation_factor_y;
   else
      return log10 (value / canvas->priv->visible_top) *
	 canvas->priv->translation_factor_y;
}

gfloat
gtk_databox_canvas_pixel_to_value_x (GtkDataboxCanvas * canvas, gint16 pixel)
{
   if (canvas->priv->changed)
      gtk_databox_canvas_calculate_translation_factor (canvas);

   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
      return canvas->priv->visible_left +
	 pixel / canvas->priv->translation_factor_x;
   else
      return canvas->priv->visible_left * pow (10,
					       pixel /
					       canvas->priv->
					       translation_factor_x);
}

gfloat
gtk_databox_canvas_pixel_to_value_y (GtkDataboxCanvas * canvas, gint16 pixel)
{
   if (canvas->priv->changed)
      gtk_databox_canvas_calculate_translation_factor (canvas);

   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
      return canvas->priv->visible_top +
	 pixel / canvas->priv->translation_factor_y;
   else
      return canvas->priv->visible_top * pow (10,
					      pixel /
					      canvas->priv->
					      translation_factor_y);
}

gint
gtk_databox_canvas_get_width (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->width;
}

void
gtk_databox_canvas_set_width (GtkDataboxCanvas * canvas, gint width)
{
   g_return_if_fail (GTK_DATABOX_IS_CANVAS (canvas));
   g_return_if_fail (width >= 0);

   canvas->priv->width = width;
   canvas->priv->changed = TRUE;
}

gint
gtk_databox_canvas_get_height (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->height;
}

void
gtk_databox_canvas_set_height (GtkDataboxCanvas * canvas, gint height)
{
   g_return_if_fail (GTK_DATABOX_IS_CANVAS (canvas));
   g_return_if_fail (height >= 0);

   canvas->priv->height = height;
   canvas->priv->changed = TRUE;
}

gfloat
gtk_databox_canvas_get_total_left (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->total_left;
}

gfloat
gtk_databox_canvas_get_total_right (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->total_right;
}

gfloat
gtk_databox_canvas_get_total_top (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->total_top;
}

gfloat
gtk_databox_canvas_get_total_bottom (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->total_bottom;
}

gfloat
gtk_databox_canvas_get_visible_left (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->visible_left;
}

gfloat
gtk_databox_canvas_get_visible_right (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->visible_right;
}

gfloat
gtk_databox_canvas_get_visible_top (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->visible_top;
}

gfloat
gtk_databox_canvas_get_visible_bottom (GtkDataboxCanvas * canvas)
{
   g_return_val_if_fail (GTK_DATABOX_IS_CANVAS (canvas), -1);

   return canvas->priv->visible_bottom;
}

void
gtk_databox_canvas_set_total_limits (GtkDataboxCanvas * canvas, gfloat left,
				     gfloat right, gfloat top, gfloat bottom)
{
   g_return_if_fail (GTK_DATABOX_IS_CANVAS (canvas));

   canvas->priv->total_left = left;
   canvas->priv->total_right = right;
   canvas->priv->total_top = top;
   canvas->priv->total_bottom = bottom;

   gtk_databox_canvas_set_visible_limits (canvas, left, right, top, bottom);
}

void
gtk_databox_canvas_set_visible_limits (GtkDataboxCanvas * canvas, gfloat left,
				       gfloat right, gfloat top,
				       gfloat bottom)
{
   gboolean visible_inside_total = FALSE;

   g_return_if_fail (GTK_DATABOX_IS_CANVAS (canvas));

   visible_inside_total =
      ((canvas->priv->total_left <= left && left < right
	&& right <= canvas->priv->total_right)
       || (canvas->priv->total_left >= left && left > right
	   && right >= canvas->priv->total_right))
      &&
      ((canvas->priv->total_bottom <= bottom && bottom < top
	&& top <= canvas->priv->total_top)
       || (canvas->priv->total_bottom >= bottom && bottom > top
	   && top >= canvas->priv->total_top));

   g_return_if_fail (visible_inside_total);

   canvas->priv->visible_left = left;
   canvas->priv->visible_right = right;
   canvas->priv->visible_top = top;
   canvas->priv->visible_bottom = bottom;
   canvas->priv->changed = TRUE;
}

gfloat
gtk_databox_canvas_get_offset_x (GtkDataboxCanvas * canvas)
{
   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
      return (canvas->priv->visible_left - canvas->priv->total_left)
	 / (canvas->priv->total_right - canvas->priv->total_left);
   else
      return log10 (canvas->priv->visible_left / canvas->priv->total_left)
	 / log10 (canvas->priv->total_right / canvas->priv->total_left);
}

gfloat
gtk_databox_canvas_get_page_size_x (GtkDataboxCanvas * canvas)
{
   if (canvas->priv->scale_type_x == GTK_DATABOX_SCALE_LINEAR)
      return (canvas->priv->visible_left - canvas->priv->visible_right)
	 / (canvas->priv->total_left - canvas->priv->total_right);
   else
      return log10 (canvas->priv->visible_left / canvas->priv->visible_right)
	 / log10 (canvas->priv->total_left / canvas->priv->total_right);
}

gfloat
gtk_databox_canvas_get_offset_y (GtkDataboxCanvas * canvas)
{
   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
      return (canvas->priv->visible_top - canvas->priv->total_top)
	 / (canvas->priv->total_bottom - canvas->priv->total_top);
   else
      return log10 (canvas->priv->visible_top / canvas->priv->total_top)
	 / log10 (canvas->priv->total_bottom / canvas->priv->total_top);
}

gfloat
gtk_databox_canvas_get_page_size_y (GtkDataboxCanvas * canvas)
{
   if (canvas->priv->scale_type_y == GTK_DATABOX_SCALE_LINEAR)
      return (canvas->priv->visible_top - canvas->priv->visible_bottom)
	 / (canvas->priv->total_top - canvas->priv->total_bottom);
   else
      return log10 (canvas->priv->visible_top / canvas->priv->visible_bottom)
	 / log10 (canvas->priv->total_top / canvas->priv->total_bottom);
}

void
gtk_databox_canvas_set_scale_type_x (GtkDataboxCanvas * canvas,
				     GtkDataboxScaleType scale_type)
{
   canvas->priv->scale_type_x = scale_type;
}

void
gtk_databox_canvas_set_scale_type_y (GtkDataboxCanvas * canvas,
				     GtkDataboxScaleType scale_type)
{
   canvas->priv->scale_type_y = scale_type;
}

GtkDataboxScaleType
gtk_databox_canvas_get_scale_type_x (GtkDataboxCanvas * canvas)
{
   return canvas->priv->scale_type_x;
}

GtkDataboxScaleType
gtk_databox_canvas_get_scale_type_y (GtkDataboxCanvas * canvas)
{
   return canvas->priv->scale_type_y;
}
